/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.factory;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.FinderException;
import javax.jms.DeliveryMode;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CommandNotFound;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.collectiverun.bean.CollectiveRunInfo;
import com.clustercontrol.collectiverun.bean.CollectiveRunStatusConstant;
import com.clustercontrol.collectiverun.bean.CommandTypeConstant;
import com.clustercontrol.collectiverun.bean.ParameterTypeConstant;
import com.clustercontrol.collectiverun.bean.QueueConstant;
import com.clustercontrol.collectiverun.ejb.entity.CRunCmdMstLocal;
import com.clustercontrol.collectiverun.ejb.entity.CRunCmdMstUtil;
import com.clustercontrol.collectiverun.ejb.entity.CRunCmdParamMstLocal;
import com.clustercontrol.collectiverun.ejb.entity.CRunEndMstLocal;
import com.clustercontrol.collectiverun.ejb.entity.CRunEndMstPK;
import com.clustercontrol.collectiverun.ejb.entity.CRunEndMstUtil;
import com.clustercontrol.collectiverun.ejb.entity.CRunParamSelectMstLocal;
import com.clustercontrol.collectiverun.ejb.entity.CRunParamSelectMstUtil;
import com.clustercontrol.collectiverun.ejb.entity.CRunSessionDetailLocal;
import com.clustercontrol.collectiverun.ejb.entity.CRunSessionDetailPK;
import com.clustercontrol.collectiverun.ejb.entity.CRunSessionDetailUtil;
import com.clustercontrol.collectiverun.ejb.entity.CRunSessionLocal;
import com.clustercontrol.collectiverun.ejb.entity.CRunSessionParamLocal;
import com.clustercontrol.collectiverun.ejb.entity.CRunSessionParamPK;
import com.clustercontrol.collectiverun.ejb.entity.CRunSessionParamUtil;
import com.clustercontrol.collectiverun.ejb.entity.CRunSessionUtil;
import com.clustercontrol.collectiverun.message.CollectiveRunInstructionInfo;
import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.jobmanagement.bean.RunStatusConstant;
import com.clustercontrol.jobmanagement.bean.RunResultInfo;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ジョブ操作クラス
 *
 * @version 2.1.1
 * @since 1.0.0
 */
public class OperationRun {
	private static Log m_log = LogFactory.getLog( OperationRun.class );
	private static String SHELL_COMMAND = "sh";

	/**
	 * コマンド実行
	 * 
	 * @param sessionId
	 * @param property
	 * @param locale
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * @throws IOException
	 */
	public void run(String sessionId) throws CreateException, FinderException, NamingException, IOException {
		m_log.debug("run() : sessionId=" + sessionId);

		//セッションIDからセッション(実行履歴)を取得
		CRunSessionLocal session =
			CRunSessionUtil.getLocalHome().findByPrimaryKey(sessionId);

		//実行コマンド取得
		String command = getCommand(session);

		//ステータスが待機中の場合
		if(session.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_WAITING){
			//ステータスを実行中にする
			session.setStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_RUNNING));
			//開始日時を設定
			session.setStart_date(new Date());
		}

		//ノードへ実行
		startNode(session, command);
	}

	/**
	 * ノードへの実行処理
	 * 
	 * @param sessionId
	 * @param facilityId
	 * @param command
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	private void startNode(CRunSessionLocal session, String command) throws FinderException, NamingException {
		m_log.debug("startNode() start: sessionId=" + session.getSession_id());

		//セッションID取得
		String sessionId = session.getSession_id();

		//コマンドIDからコマンドマスタを取得
		CRunCmdMstLocal cmd = CRunCmdMstUtil.getLocalHome().findByPrimaryKey(session.getCommand_id());
		//コマンドマスタからコマンドタイプを取得
		int commandType = cmd.getCommand_type().intValue();

		String password = null;
		try {
			CRunSessionParamLocal param = CRunSessionParamUtil.getLocalHome().findByPrimaryKey(
					new CRunSessionParamPK(sessionId, "PASSWORD"));
			password = param.getParam_value();
		} catch (Exception e) {
			m_log.debug("startNode() : " + e.getClass().getSimpleName() + ", " + e.getMessage());
		}

		//セッションディテール(実行履歴詳細)を取得
		Collection<CRunSessionDetailLocal> collection = session.getSessionDetail();
		if(collection != null && collection.size() > 0){
			//Queue送信クラス生成
			SendQueue send = null;
			try {
				send = new SendQueue(QueueConstant.QUEUE_NAME_EXECUTE);
			} catch (Exception e) {
				m_log.debug("startNode() : sessionDetail send error : " + e.getMessage());
			}

			Iterator<CRunSessionDetailLocal> itr = collection.iterator();
			while(itr.hasNext()){
				CRunSessionDetailLocal sessionDetail = itr.next();

				//実行情報を作成
				CollectiveRunInstructionInfo info = new CollectiveRunInstructionInfo();
				info.setSessionId(sessionId);
				info.setFacilityId(sessionDetail.getFacility_id());
				if(commandType == CommandTypeConstant.TYPE_SHELL){
					info.setCommand(SHELL_COMMAND);
					info.setInputFile(command);
				}
				else{
					info.setCommand(command);
				}
				info.setCommandType(com.clustercontrol.jobmanagement.bean.CommandTypeConstant.NORMAL);

				info.setPassword(password);

				try {
					//Queueに送信
					send.put(info, DeliveryMode.PERSISTENT);
				} catch (Exception e) {
					AplLogger apllog = new AplLogger("CR","cr");
					String[] args = {info.getSessionId(), info.getFacilityId()};
					apllog.put("SYS","003", args);
					m_log.debug("startNode() : CollectiveRunInstructionInfo send error  : " + e.getMessage());
				}
			}

			//Queue送信クラス後処理
			try {
				send.terminate();
			} catch (Exception e) {
				m_log.debug("startNode() : CollectiveRunInstructionInfo send error  : " + e.getMessage());
			}
		}

		m_log.debug("startNode() end: sessionId=" + session.getSession_id());
	}

	/**
	 * ノード終了処理
	 * 
	 * @param info
	 * @throws FinderException
	 * @throws NamingException
	 */
	public void endNode(RunResultInfo info) throws FinderException, NamingException {
		m_log.debug("endNode() start: sessionId=" + info.getSessionId() + ", facilityId=" + info.getFacilityId() + ", status=" + info.getStatus());

		//セッションIDからセッション(実行履歴)を取得
		CRunSessionLocal session =
			CRunSessionUtil.getLocalHome().findByPrimaryKey(info.getSessionId());

		//コマンドタイプチェック
		if(info.getCommandType() == com.clustercontrol.jobmanagement.bean.CommandTypeConstant.NORMAL){
			//コマンドタイプが通常の場合

			//セッションディテール(実行履歴詳細)を取得
			CRunSessionDetailLocal sessionDetail =
				CRunSessionDetailUtil.getLocalHome().findByPrimaryKey(
						new CRunSessionDetailPK(info.getSessionId(), info.getFacilityId()));

			//ステータスで分岐
			if(info.getStatus() == RunStatusConstant.START){
				//開始の場合

				if(sessionDetail.getStart_date() == null){
					//ステータスを実行中にする
					sessionDetail.setStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_RUNNING));

					//開始日時を設定
					sessionDetail.setStart_date(new Date(info.getTime()));

					//実行中ノード数をインクリメント
					session.setRunning_cnt(Integer.valueOf(session.getRunning_cnt().intValue() + 1));
					//待機中ノード数をデクリメント
					session.setWait_cnt(Integer.valueOf(session.getWait_cnt().intValue() - 1));
				}
				else{
					AplLogger apllog = new AplLogger("CR","cr");
					String[] args = {info.getSessionId(), info.getFacilityId()};
					apllog.put("SYS","004", args);
					m_log.debug("endNode() : this messsage is already received. drop message. " +
							" SessionId=" + info.getSessionId() +
							", FacilityId=" + info.getFacilityId());
				}
				sessionDetail = null;
			}
			else{
				//ステータスが実行中の場合
				if(sessionDetail.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_RUNNING){
					//実行中ノード数をデクリメント
					session.setRunning_cnt(Integer.valueOf(session.getRunning_cnt().intValue() - 1));
				}

				if(info.getStatus() == RunStatusConstant.END){
					//終了の場合

					//ステータスが実行中の場合
					if(sessionDetail.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_RUNNING){
						//ステータスに終了を設定
						sessionDetail.setStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_END));
						//終了日時を設定
						sessionDetail.setEnd_date(new Date(info.getTime()));
						//終了値を設定
						sessionDetail.setEnd_value(Integer.valueOf(info.getEndValue()));
						//メッセージを設定
						sessionDetail.setMessage(info.getMessage());
						//エラーメッセージを設定
						sessionDetail.setError_message(info.getErrorMessage());

						//終了状態を設定
						setEndStatus(session, sessionDetail);
					}
				}
				else if(info.getStatus() == RunStatusConstant.ERROR){
					//失敗の場合

					//ステータスが実行中の場合
					if(sessionDetail.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_RUNNING){
						//ステータスに終了を設定
						sessionDetail.setStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_ERROR));
						//メッセージを設定
						sessionDetail.setMessage(info.getMessage());
						//エラーメッセージを設定
						sessionDetail.setError_message(info.getErrorMessage());

						//失敗ノード数をインクリメント
						session.setEnd_abnomal_cnt(Integer.valueOf(session.getEnd_abnomal_cnt().intValue() + 1));
					}
				}
				sessionDetail = null;


				//終了チェック
				if(checkAllNodeEnd(session)){
					//全ノード終了の場合

					//ノード数設定
					setNodeCount(session);

					//ステータスが実行中の場合
					if(session.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_RUNNING){
						//ステータスに終了を設定
						session.setStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_END));
						//終了日時を設定
						session.setEnd_date(new Date(info.getTime()));
					}
				}
			}
		}

		m_log.debug("endNode() end: sessionId=" + info.getSessionId() + ", facilityId=" + info.getFacilityId() + ", status=" + info.getStatus());
	}

	/**
	 * ノード数設定処理
	 * 
	 * @param sessionId
	 * @param facilityId
	 * @throws FinderException
	 * @throws NamingException
	 */
	private void setNodeCount(CRunSessionLocal session) throws FinderException, NamingException{
		m_log.debug("setNodeCount() : sessionId=" + session.getSession_id());

		int normal = 0;
		int warning = 0;
		int abnormal = 0;
		int running = 0;
		int waiting = 0;

		//セッションディテール(実行履歴詳細)を取得
		Collection collection = session.getSessionDetail();
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				CRunSessionDetailLocal sessionDetail = (CRunSessionDetailLocal)itr.next();

				//ステータスをチェック
				if(sessionDetail.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_RUNNING){
					//実行中
					running++;
				}
				else if(sessionDetail.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_WAITING){
					//待機中
					waiting++;
				}
				else if(sessionDetail.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_ERROR){
					//実行失敗
					abnormal++;
				}

				//終了状態をチェック
				if(sessionDetail.getEnd_status() != null){
					if(sessionDetail.getEnd_status().intValue() == EndStatusConstant.TYPE_NORMAL){
						//実行中
						normal++;
					}
					else if(sessionDetail.getEnd_status().intValue() == EndStatusConstant.TYPE_WARNING){
						//待機中
						warning++;
					}
					else if(sessionDetail.getEnd_status().intValue() == EndStatusConstant.TYPE_ABNORMAL){
						//実行失敗
						abnormal++;
					}
				}
			}
		}

		//実行中の数を設定
		session.setRunning_cnt(Integer.valueOf(running));
		//待機中の数を設定
		session.setWait_cnt(Integer.valueOf(waiting));
		//正常の数を設定
		session.setEnd_nomal_cnt(Integer.valueOf(normal));
		//警告の数を設定
		session.setEnd_warning_cnt(Integer.valueOf(warning));
		//異常の数を設定
		session.setEnd_abnomal_cnt(Integer.valueOf(abnormal));
	}

	/**
	 * 終了状態設定処理
	 * 
	 * @param sessionId
	 * @param facilityId
	 * @throws FinderException
	 * @throws NamingException
	 */
	private void setEndStatus(CRunSessionLocal session, CRunSessionDetailLocal sessionDetail) throws FinderException, NamingException{
		m_log.debug("setEndStatus() : sessionId=" + session.getSession_id() + ", facilityId=" + sessionDetail.getFacility_id());

		//終了状態（正常）を取得
		CRunEndMstLocal endInfoNormal = null;
		try {
			endInfoNormal = CRunEndMstUtil.getLocalHome().findByPrimaryKey(
					new CRunEndMstPK(session.getCommand_id(), Integer.valueOf(EndStatusConstant.TYPE_NORMAL)));
		} catch (FinderException e) {
			m_log.debug("setEndStatus() : EndMst normal find error" + e.getMessage());
		}
		//終了状態（警告）を取得
		CRunEndMstLocal endInfoWarning = null;
		try {
			endInfoWarning = CRunEndMstUtil.getLocalHome().findByPrimaryKey(
					new CRunEndMstPK(session.getCommand_id(), Integer.valueOf(EndStatusConstant.TYPE_WARNING)));
		} catch (FinderException e) {
			m_log.debug("setEndStatus() : EndMst warning find error" + e.getMessage());
		}

		Integer endValue = sessionDetail.getEnd_value();
		if(endValue instanceof Integer){
			if(endInfoNormal != null &&
					endValue.compareTo(endInfoNormal.getEnd_value_from()) >= 0 &&
					endValue.compareTo(endInfoNormal.getEnd_value_to()) <= 0){
				//終了状態（正常）の範囲内ならば、正常とする
				sessionDetail.setEnd_status(Integer.valueOf(EndStatusConstant.TYPE_NORMAL));
				//成功ノード数をインクリメント
				session.setEnd_nomal_cnt(Integer.valueOf(session.getEnd_nomal_cnt().intValue() + 1));
			}
			else if(endInfoWarning != null &&
					endValue.compareTo(endInfoWarning.getEnd_value_from()) >= 0 &&
					endValue.compareTo(endInfoWarning.getEnd_value_to()) <= 0){
				//終了状態（警告）の範囲内ならば、警告とする
				sessionDetail.setEnd_status(Integer.valueOf(EndStatusConstant.TYPE_WARNING));
				//警告ノード数をインクリメント
				session.setEnd_warning_cnt(Integer.valueOf(session.getEnd_warning_cnt().intValue() + 1));
			}
			else{
				//終了状態（異常）の範囲内ならば、異常とする
				sessionDetail.setEnd_status(Integer.valueOf(EndStatusConstant.TYPE_ABNORMAL));
				//失敗ノード数をインクリメント
				session.setEnd_abnomal_cnt(Integer.valueOf(session.getEnd_abnomal_cnt().intValue() + 1));
			}
		}
	}

	/**
	 * 全ノード終了チェック処理
	 * 
	 * @param session
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	private boolean checkAllNodeEnd(CRunSessionLocal session) throws FinderException, NamingException{
		m_log.debug("checkAllNodeEnd() start: sessionId=" + session.getSession_id());

		//終了フラグをfalseにする
		boolean end = false;

		//ステータスが実行中または待機中が存在するかチェック
		boolean runNode = false;
		if(session.getRunning_cnt().intValue() > 0 || session.getWait_cnt().intValue() > 0){
			runNode = true;
		}

		//全ノード終了の場合
		if(!runNode){
			//終了フラグをtrueにする
			end = true;
		}

		m_log.debug("checkAllNodeEnd() end: sessionId=" + session.getSession_id());
		return end;
	}

	/**
	 * コマンド取得処理
	 * 
	 * @param sessionId
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 * @throws IOException
	 */
	private String getCommand(CRunSessionLocal session) throws FinderException, NamingException, IOException {
		m_log.debug("getCommand() : sessionId=" + session.getSession_id());

		String command = null;

		//コマンドIDからコマンドマスタを取得
		CRunCmdMstLocal cmd = CRunCmdMstUtil.getLocalHome().findByPrimaryKey(session.getCommand_id());

		//------------------------------
		//コマンドの引数を作成する
		//------------------------------
		StringBuffer argument = new StringBuffer();
		//コマンドマスタからコマンドパラメータを取得
		Collection cmdPramList = cmd.getCmdParamMst();
		if(cmdPramList != null && cmdPramList.size() > 0){
			Iterator itr = cmdPramList.iterator();
			while(itr.hasNext()){
				//コマンドパラメータを取得
				CRunCmdParamMstLocal cmdParam = (CRunCmdParamMstLocal)itr.next();
				//セッションパラメータ(実行履歴パラメータ)を取得
				CRunSessionParamLocal sessionParam =
					CRunSessionParamUtil.getLocalHome().findByPrimaryKey(
							new CRunSessionParamPK(session.getSession_id(), cmdParam.getParam_id()));

				if(sessionParam.getParam_value() != null &&
						sessionParam.getParam_value().length() > 0){
					if(cmdParam.getParam_prefix() != null &&
							cmdParam.getParam_prefix().length() > 0){
						//Prefixがnullまたは空白文字の場合、Prefixは付けない
						argument.append(cmdParam.getParam_prefix());
						argument.append(" ");
					}
					//パラメータ値がnullまたは空白文字の場合、パラメータ値はつけない
					argument.append(sessionParam.getParam_value());
					argument.append(" ");
				}
			}
		}

		if(cmd.getCommand_type().intValue() == CommandTypeConstant.TYPE_SHELL){
			//コマンドタイプがシェルの場合、ファイル内容の前に'set - 'と引数を追加する
			command = "set - " + argument.toString() + "\n" + readShellFile(cmd.getCommand());
		}
		else{
			//コマンドタイプがシェルの場合、コマンドの後に引数を追加する
			command = cmd.getCommand() + " " + argument.toString();
		}

		m_log.debug("getCommand() : command=" + command);
		return command;
	}

	/**
	 * シェルファイル読み込み処理
	 * 
	 * @param fileName
	 * @return ファイル文字列
	 * @throws IOException
	 */
	private String readShellFile(String fileName) throws IOException {
		m_log.debug("readShellFile() : fileName=" + fileName);

		StringBuffer filePath = new StringBuffer();
		filePath.append(fileName);

		FileInputStream fileInputStream = null;
		InputStreamReader inputStreamReader = null;
		BufferedReader bufferedReader=null;
		String lineData;
		StringBuffer fileData = new StringBuffer();
		try {
			fileInputStream = new FileInputStream(filePath.toString());
			inputStreamReader = new InputStreamReader(fileInputStream);
			bufferedReader = new BufferedReader(inputStreamReader);
			//一行ずつ全て読み込むまで繰り返す
			while ((lineData = bufferedReader.readLine())!=null){
				fileData.append(lineData);
				//改行コードを追加
				fileData.append("\n");
			}
		} catch (FileNotFoundException e) {
			AplLogger apllog = new AplLogger("CR","cr");
			String[] args = {fileName};
			apllog.put("SYS","005", args);
			m_log.debug("readShellFile() : fileName=" + fileName + " is not found : " + e.getMessage());
			throw e;
		} catch (IOException e) {
			AplLogger apllog = new AplLogger("CR","cr");
			String[] args = {fileName};
			apllog.put("SYS","006", args);
			m_log.debug("readShellFile() : fileName=" + fileName + " IO error : " + e.getMessage());
			throw e;
		} finally {
			try{
				if(bufferedReader!=null){
					bufferedReader.close();
				}
				if(inputStreamReader!=null){
					inputStreamReader.close();
				}
				if(fileInputStream!=null){
					fileInputStream.close();
				}
			}catch(IOException e){
			}
		}

		return fileData.toString();
	}

	/**
	 * 一括制御の実行履歴情報を作成します。
	 * 
	 * @param typeId
	 * @param info
	 * @throws FacilityNotFound
	 * @throws NamingException
	 * @throws CreateException
	 * @throws CommandNotFound
	 * @throws HinemosUnknown
	 */
	@SuppressWarnings("unchecked")
	public String createSession(String typeId, CollectiveRunInfo info)
	throws FacilityNotFound, NamingException, CreateException, CommandNotFound, HinemosUnknown {
		m_log.debug("createSession() : typeId=" + typeId);

		//リポジトリセッションBean取得
		RepositoryControllerLocal repository;
		repository = RepositoryControllerUtil.getLocalHome().create();

		//ファシリティID（スコープ）取得
		String facilityId = null;
		if(info.getFacilityId() instanceof String && info.getFacilityId().length() > 0){
			facilityId = info.getFacilityId();
		}

		//名前取得
		String commandId = null;
		if(info.getName() instanceof String && info.getName().length() > 0){
			String nameString = info.getName();
			//名前からコマンドIDを取得する
			try {
				commandId = getCommandId(typeId, nameString);
			} catch (FinderException e) {
				m_log.warn("createSession " + e.getMessage());
				throw new CommandNotFound(e.getMessage(), e);
			}
		}

		//コマンドIDからコマンドマスタを取得
		CRunCmdMstLocal cmd;
		try {
			cmd = CRunCmdMstUtil.getLocalHome().findByPrimaryKey(commandId);
		} catch (FinderException e1) {
			m_log.warn("createSession " + e1.getMessage());
			throw new FacilityNotFound(e1.getMessage(), e1);
		}

		//セッションを作成
		boolean createFlg = false;
		String sessionId = null;
		Date createTime = new Date();
		CRunSessionLocal session = null;
		for(int i = 0; !createFlg; i++){
			try {
				SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
				sessionId = dateFormat.format(createTime);

				DecimalFormat format = new DecimalFormat("-000");
				sessionId = sessionId + format.format(i);

				//セッション(実行履歴)を作成
				session =
					CRunSessionUtil.getLocalHome().create(
							sessionId,
							null,
							null,
							cmd.getType_id(),
							cmd.getCommand_id(),
							facilityId,
							repository.getFacilityPath(facilityId, null),
							Integer.valueOf(0),
							Integer.valueOf(0),
							Integer.valueOf(0),
							Integer.valueOf(0),
							Integer.valueOf(0),
							Integer.valueOf(0),
							Integer.valueOf(CollectiveRunStatusConstant.TYPE_WAITING));

				createFlg = true;
			} catch (DuplicateKeyException e) {
				//プライマリーキー重複
				//再度セッション(実行履歴)の作成を行う
			}
		}

		//対象ファシリティIDからセッションディテール(実行履歴詳細)を作成
		ArrayList nodeIdList = repository.getExecTargetFacilityIdList(session.getFacility_id());

		if(nodeIdList instanceof ArrayList){
			for(int i = 0; i < nodeIdList.size(); i++){
				String facilityName = null;
				if (repository.isNode(facilityId)) {
					NodeInfo nodeInfo = repository.getNode(facilityId);
					facilityName = nodeInfo.getFacilityName();
				} else {
					ScopeInfo scopeInfo = repository.getScope(facilityId);
					facilityName = scopeInfo.getFacilityName();
				}

				//セッションディテール(実行履歴詳細)を作成
				CRunSessionDetailUtil.getLocalHome().create(
						sessionId,
						(String)nodeIdList.get(i),
						null,
						null,
						facilityName,
						null,
						null,
						Integer.valueOf(CollectiveRunStatusConstant.TYPE_WAITING),
						null,
						null);
			}

			//ノード数設定
			session.setNode_cnt(Integer.valueOf(nodeIdList.size()));
			//待機中ノード数設定
			session.setWait_cnt(Integer.valueOf(nodeIdList.size()));
		}

		//コマンドマスタからコマンドパラメータを取得
		Collection<CRunCmdParamMstLocal> cmdPramList = cmd.getCmdParamMst();
		if(cmdPramList != null && cmdPramList.size() > 0){
			for(CRunCmdParamMstLocal cmdParam : cmdPramList){
				int paramType = cmdParam.getParamMst().getParam_type().intValue();

				//パラメータ値取得
				String value = null;
				if(info.getParamValue(cmdParam.getParam_id()) instanceof String && info.getParamValue(cmdParam.getParam_id()).length() > 0){
					value = info.getParamValue(cmdParam.getParam_id());
				}

				//パラメータタイプが選択の場合、表示文字列からパラメータ値を取得する
				if(paramType == ParameterTypeConstant.TYPE_SELECT){
					try {
						value = getParamSelectValue(cmdParam.getParam_id(), value, Locale.getDefault());
					} catch (FinderException e) {
						m_log.warn("createSession " + e.getMessage());
						throw new CommandNotFound(e.getMessage(), e);
					}
				}

				//セッションパラメータ(実行履歴パラメータ)を作成
				CRunSessionParamUtil.getLocalHome().create(
						sessionId,
						cmdParam.getParam_id(),
						value);
			}
		}

		return sessionId;
	}

	/**
	 * コマンドID取得
	 * 
	 * 表示文字列からコマンドIDを取得する
	 * 
	 * @param typeId 種別ID
	 * @param name 表示文字列
	 * @param locale ロケール
	 * @return コマンドID
	 * @throws FinderException
	 * @throws NamingException
	 */
	public static String getCommandId(String typeId, String name) throws FinderException, NamingException {
		m_log.debug("getCommandId() : typeId=" + typeId + ", name=" + name);

		String commandId = null;

		//種別IDからコマンドマスタを取得
		Collection cmdList = CRunCmdMstUtil.getLocalHome().findByTypeId(typeId);

		if(cmdList != null && cmdList.size() > 0){
			Iterator itr = cmdList.iterator();
			while(itr.hasNext()){
				//コマンドマスタを取得
				CRunCmdMstLocal cmd = (CRunCmdMstLocal)itr.next();
				//表示項目IDからリソースを取得
				String nameValue = Messages.getString(cmd.getName_id(), Locale.getDefault());
				//リソースが一致したらコマンドIDを返す
				if(nameValue.equals(name)){
					commandId = cmd.getCommand_id();
					break;
				}
			}
		}

		return commandId;
	}

	/**
	 * パラメータ選択値取得
	 * 
	 * 表示文字列からパラメータ選択値を取得する
	 * 
	 * @param paramId パラメータID
	 * @param name 表示文字列
	 * @param locale ロケール
	 * @return パラメータ選択値
	 * @throws FinderException
	 * @throws NamingException
	 */
	private String getParamSelectValue(String paramId, String name, Locale locale) throws FinderException, NamingException {
		m_log.debug("getParamSelectValue() : paramId=" + paramId + ", name=" + name);

		String value = null;

		//パラメータIDからパラメータ選択マスタを取得
		Collection paramSelectList = CRunParamSelectMstUtil.getLocalHome().findByParamId(paramId);

		if(paramSelectList != null && paramSelectList.size() > 0){
			Iterator itr = paramSelectList.iterator();
			while(itr.hasNext()){
				//パラメータ選択マスタを取得
				CRunParamSelectMstLocal paramSelect = (CRunParamSelectMstLocal)itr.next();
				//表示項目IDからリソースを取得
				String nameValue = Messages.getString(paramSelect.getName_id(), locale);
				//リソースが一致したらコマンドIDを返す
				if(nameValue.equals(name)){
					value = paramSelect.getParam_value();
					break;
				}
			}
		}

		return value;
	}

	/**
	 * パラメータ選択値取得
	 * 
	 * 表示文字列からパラメータ選択値を取得する
	 * 
	 * @param paramId パラメータID
	 * @param name 表示文字列
	 * @param locale ロケール
	 * @return パラメータ選択値
	 * @throws FinderException
	 * @throws NamingException
	 */
	public void checkEndSession() throws FinderException, NamingException {
		m_log.debug("checkEndSession()");

		//実行中のセッションを取得
		Collection collection =
			CRunSessionUtil.getLocalHome().findByStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_RUNNING));

		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				CRunSessionLocal session = (CRunSessionLocal)itr.next();

				//ノード数設定
				setNodeCount(session);

				//終了チェック
				if(checkAllNodeEnd(session)){
					//全ノード終了の場合

					//ステータスが実行中の場合
					if(session.getStatus().intValue() == CollectiveRunStatusConstant.TYPE_RUNNING){
						//ステータスに終了を設定
						session.setStatus(Integer.valueOf(CollectiveRunStatusConstant.TYPE_END));
						//終了日時を設定
						session.setEnd_date(new Date());
					}
				}
			}
		}
	}
}
