/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 */

package com.clustercontrol.performance.ejb.session;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.performance.bean.CollectMasterInfo;
import com.clustercontrol.performance.factory.OperateCollectCalcMaster;
import com.clustercontrol.performance.factory.OperateCollectCategoryCollectMaster;
import com.clustercontrol.performance.factory.OperateCollectCategoryMaster;
import com.clustercontrol.performance.factory.OperateCollectItemCalcMethodMaster;
import com.clustercontrol.performance.factory.OperateCollectItemCodeMaster;
import com.clustercontrol.performance.factory.OperateCollectPollingMaster;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCalcMethodMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCalcMethodMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCalcMethodMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryCollectMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryCollectMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryCollectMstPK;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryCollectMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCalcMethodMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCalcMethodMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCalcMethodMstPK;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCalcMethodMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorPollingMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorPollingMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorPollingMstPK;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorPollingMstUtil;
import com.clustercontrol.repository.ejb.entity.CollectorPlatformMstData;
import com.clustercontrol.repository.ejb.entity.CollectorPlatformMstLocal;
import com.clustercontrol.repository.ejb.entity.CollectorPlatformMstUtil;

/**
 * 収集項目マスタ情報を制御するSession Bean <BR>
 * @ejb.bean name="PerformanceCollectMasterController"
 *           jndi-name="PerformanceCollectMasterController"
 *           type="Stateless"
 *           transaction-type="Container"
 *           view-type="local"
 * 
 * @ejb.transaction type="Required"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorCalcMethodMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorCategoryCollectMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorCategoryMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorItemCalcMethodMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorItemCodeMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorPollingMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorPlatformMstLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 */
public abstract class PerformanceCollectMasterControllerBean implements SessionBean {

	private static Log m_log = LogFactory.getLog( PerformanceCollectMasterControllerBean.class );

	@SuppressWarnings("unused")
	private SessionContext m_context;

	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * 計算方法定義を登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data 計算方法定義情報
	 * @return 登録に成功した場合、true
	 * @throws NamingException
	 * @throws CreateException
	 */
	public boolean addCollectCalcMaster(CollectorCalcMethodMstData data) throws CreateException, NamingException {

		// 計算ロジック情報を登録
		OperateCollectCalcMaster ope = new OperateCollectCalcMaster();
		boolean ret = false;

		try{
			ret = ope.add(data);
		}catch(CreateException e){
			m_log.warn("addCollectCalcMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("addCollectCalcMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * カテゴリ毎の収集方法定義を登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data カテゴリ毎の収集方法定義情報
	 * @return 登録に成功した場合、true
	 * @throws NamingException
	 * @throws CreateException
	 */
	public boolean addCollectCategoryCollectMaster(CollectorCategoryCollectMstData data) throws CreateException, NamingException {

		//カテゴリ毎の収集方法定義を登録
		OperateCollectCategoryCollectMaster ope = new OperateCollectCategoryCollectMaster();
		boolean ret = false;

		try{
			ret = ope.add(data);
		}catch(CreateException e){
			m_log.warn("addCollectCategoryCollectMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("addCollectCategoryCollectMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * 収集カテゴリ定義を登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data 収集カテゴリ定義情報
	 * @return 登録に成功した場合、true
	 * @throws NamingException
	 * @throws CreateException
	 */
	public boolean addCollectCategoryMaster(CollectorCategoryMstData data) throws CreateException, NamingException {

		// 収集カテゴリ定義を登録
		OperateCollectCategoryMaster ope = new OperateCollectCategoryMaster();
		boolean ret = false;

		try{
			ret = ope.add(data);
		}catch(CreateException e){
			m_log.warn("addCollectCategoryMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("addCollectCategoryMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * 収集項目毎の計算方法を登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data 収集項目毎の計算方法定義情報
	 * @return 登録に成功した場合、true
	 * @throws NamingException
	 * @throws CreateException
	 * @throws EJBException
	 */
	public boolean addCollectItemCalcMethodMaster(CollectorItemCalcMethodMstData data) throws CreateException, NamingException, EJBException {

		// 収集項目毎の計算方法を登録
		OperateCollectItemCalcMethodMaster ope = new OperateCollectItemCalcMethodMaster();
		boolean ret = false;

		try{
			ret = ope.add(data);
		}catch(CreateException e){
			m_log.warn("addCollectItemCalcMethodMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("addCollectItemCalcMethodMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;
	}

	/**
	 * 収集項目コードを登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data 収集項目コード情報
	 * @return 登録に成功した場合、true
	 * @throws NamingException
	 * @throws CreateException
	 */
	public boolean addCollectItemCodeMaster(CollectorItemCodeMstData data) throws CreateException, NamingException {

		// 収集項目コードを登録
		OperateCollectItemCodeMaster ope = new OperateCollectItemCodeMaster();
		boolean ret = false;

		try{
			ret = ope.add(data);
		}catch(CreateException e){
			m_log.warn("addCollectItemCodeMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("addCollectItemCodeMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * 収集方法・プラットフォーム毎の収集項目を登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data 収集方法・プラットフォーム毎の収集項目情報
	 * @return 登録に成功した場合、true
	 * @throws NamingException
	 * @throws CreateException
	 */
	public boolean addCollectPollingMaster(CollectorPollingMstData data) throws CreateException, NamingException {

		// 収集方法・プラットフォーム毎の収集項目を登録
		OperateCollectPollingMaster ope = new OperateCollectPollingMaster();
		boolean ret = false;

		try{
			ret = ope.add(data);
		}catch(CreateException e){
			m_log.warn("addCollectPollingMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("addCollectPollingMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * プラットフォーム定義を登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data プラットフォーム定義情報
	 * @return 登録に成功した場合、true
	 * @throws HinemosUnknown
	 */
	public boolean addCollectPlatformMaster(CollectorPlatformMstData data) throws HinemosUnknown {

		try{
			// プラットフォーム定義情報を登録
			CollectorPlatformMstUtil.getLocalHome().create(
					data.getPlatformId(),
					data.getPlatformName(),
					data.getOrderNo()
			);
		}catch(CreateException e){
			// プラットフォーム情報の追加に失敗した場合。
			// 既にあるものは残したままで追加させるロジックのため、ロールバックはせず追加できなかったことをフラグで示す。
			return false;
		}catch(NamingException e){
			m_log.warn("addCollectPlatformMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown("addCollectPlatformMaster(): " + e.getMessage(), e);
		}

		return true;

	}

	/**
	 * 収集項目マスタデータを一括で登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 登録に成功した場合、true
	 * @throws NamingException
	 * @throws CreateException
	 */
	public boolean addCollectMaster(CollectMasterInfo collectMasterInfo) throws HinemosUnknown {
		boolean rtnFlg = true;
		try {
			m_log.debug("addCollectCategoryMaster");
			for(CollectorCategoryMstData data : collectMasterInfo.getCollectorCategoryMstDataList()){
				rtnFlg = rtnFlg && addCollectCategoryMaster(data);
			}
			m_log.debug("addCollectItemCodeMaster");
			for(CollectorItemCodeMstData data : collectMasterInfo.getCollectorItemCodeMstDataList()){
				rtnFlg = rtnFlg && addCollectItemCodeMaster(data);
			}
			m_log.debug("addCollectCalcMaster");
			for(CollectorCalcMethodMstData data : collectMasterInfo.getCollectorCalcMethodMstDataList()){
				rtnFlg = rtnFlg && addCollectCalcMaster(data);
			}
			m_log.debug("addCollectItemCalcMethodMaster");
			for(CollectorItemCalcMethodMstData data : collectMasterInfo.getCollectorItemCalcMethodMstDataList()){
				rtnFlg = rtnFlg && addCollectItemCalcMethodMaster(data);
			}
			m_log.debug("addCollectPollingMaster");
			for(CollectorPollingMstData data : collectMasterInfo.getCollectorPollingMstDataList()){
				rtnFlg = rtnFlg && addCollectPollingMaster(data);
			}
			m_log.debug("addCollectCategoryCollectMaster");
			for(CollectorCategoryCollectMstData data : collectMasterInfo.getCollectorCategoryCollectMstDataList()){
				rtnFlg = rtnFlg && addCollectCategoryCollectMaster(data);
			}
		} catch (CreateException e) {
			rtnFlg = false;
			throw new HinemosUnknown("addCollectMaster(): " + e.getMessage(), e);
		} catch (NamingException e) {
			rtnFlg = false;
			throw new HinemosUnknown("addCollectMaster(): " + e.getMessage(), e);
		}

		return rtnFlg;
	}

	/**
	 * 計算方法定義を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data 計算方法定義情報
	 * @return 削除に成功した場合、true
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws EJBException
	 */
	public boolean deleteCollectCalcMaster(String calcMethod) throws EJBException, FinderException, NamingException, RemoveException {

		// 計算ロジック情報を削除
		OperateCollectCalcMaster ope = new OperateCollectCalcMaster();
		boolean ret = false;

		try{
			ret = ope.delete(calcMethod);
		}catch(EJBException e){
			m_log.warn("deleteCollectCalcMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("deleteCollectCalcMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteCollectCalcMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("deleteCollectCalcMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * 計算方法定義を全て削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 削除に成功した場合、true
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws EJBException
	 */
	public boolean deleteCollectCalcMasterAll() throws EJBException, FinderException, NamingException, RemoveException {

		// 計算ロジック情報を削除
		OperateCollectCalcMaster ope = new OperateCollectCalcMaster();
		boolean ret = false;

		try{
			ret = ope.deleteAll();
		}catch(EJBException e){
			m_log.warn("deleteCollectCalcMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("deleteCollectCalcMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteCollectCalcMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("deleteCollectCalcMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * カテゴリ毎の収集方法定義を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data カテゴリ毎の収集方法定義情報
	 * @return 削除に成功した場合、true
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws EJBException
	 */
	public boolean deleteCollectCategoryCollectMaster(CollectorCategoryCollectMstPK pk) throws EJBException, FinderException, NamingException, RemoveException {

		//カテゴリ毎の収集方法定義を削除
		OperateCollectCategoryCollectMaster ope = new OperateCollectCategoryCollectMaster();

		boolean ret = false;

		try{
			ret = ope.delete(pk);
		}catch(EJBException e){
			m_log.warn("deleteCollectCategoryCollectMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("deleteCollectCategoryCollectMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteCollectCategoryCollectMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("deleteCollectCategoryCollectMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * カテゴリ毎の収集方法定義を全て削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 削除に成功した場合、true
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws EJBException
	 */
	public boolean deleteCollectCategoryCollectMasterAll() throws EJBException, FinderException, NamingException, RemoveException {

		//カテゴリ毎の収集方法定義を削除
		OperateCollectCategoryCollectMaster ope = new OperateCollectCategoryCollectMaster();
		boolean ret = false;

		try{
			ret = ope.deleteAll();
		}catch(EJBException e){
			m_log.warn("deleteCollectCategoryCollectMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("deleteCollectCategoryCollectMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteCollectCategoryCollectMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("deleteCollectCategoryCollectMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * 収集カテゴリ定義を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data 収集カテゴリ定義情報
	 * @return 削除に成功した場合、true
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws EJBException
	 */
	public boolean deleteCollectCategoryMaster(String categoryCode) throws EJBException, FinderException, NamingException, RemoveException {

		// 収集カテゴリ定義を削除
		OperateCollectCategoryMaster ope = new OperateCollectCategoryMaster();
		boolean ret = false;

		try{
			ret = ope.delete(categoryCode);
		}catch(EJBException e){
			m_log.warn("deleteCollectCategoryMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("deleteCollectCategoryMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteCollectCategoryMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(FinderException e){
			m_log.warn("deleteCollectCategoryMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * 収集カテゴリ定義を全て削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 削除に成功した場合、true
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws EJBException
	 */
	public boolean deleteCollectCategoryMasterAll() throws EJBException, FinderException, NamingException, RemoveException {

		// 収集カテゴリ定義を削除
		OperateCollectCategoryMaster ope = new OperateCollectCategoryMaster();
		boolean ret = false;

		try{
			ret = ope.deleteAll();
		}catch(EJBException e){
			m_log.warn("deleteCollectCategoryMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("deleteCollectCategoryMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteCollectCategoryMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(FinderException e){
			m_log.warn("deleteCollectCategoryMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * 収集項目毎の計算方法を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data 収集項目毎の計算方法定義情報
	 * @return 削除に成功した場合、true
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws EJBException
	 */
	public boolean deleteCollectItemCalcMethodMaster(CollectorItemCalcMethodMstPK pk) throws EJBException, FinderException, NamingException, RemoveException {

		// 収集項目毎の計算方法を削除
		OperateCollectItemCalcMethodMaster ope = new OperateCollectItemCalcMethodMaster();
		boolean ret = false;

		try{
			ret = ope.delete(pk);
		}catch(EJBException e){
			m_log.warn("deleteCollectItemCalcMethodMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("deleteCollectItemCalcMethodMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteCollectItemCalcMethodMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(FinderException e){
			m_log.warn("deleteCollectItemCalcMethodMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * 収集項目毎の計算方法を全て削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 削除に成功した場合、true
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws EJBException
	 */
	public boolean deleteCollectItemCalcMethodMasterAll() throws EJBException, FinderException, NamingException, RemoveException {

		// 収集項目毎の計算方法を削除
		OperateCollectItemCalcMethodMaster ope = new OperateCollectItemCalcMethodMaster();
		boolean ret = false;

		try{
			ret = ope.deleteAll();
		}catch(EJBException e){
			m_log.warn("deleteCollectItemCalcMethodMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("deleteCollectItemCalcMethodMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteCollectItemCalcMethodMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(FinderException e){
			m_log.warn("deleteCollectItemCalcMethodMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * 収集項目コードを削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data 収集項目コード情報
	 * @return 削除に成功した場合、true
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws EJBException
	 */
	public boolean deleteCollectItemCodeMaster(String itemCode) throws EJBException, FinderException, NamingException, RemoveException {

		// 収集項目コードを削除
		OperateCollectItemCodeMaster ope = new OperateCollectItemCodeMaster();
		boolean ret = false;

		try{
			ret = ope.delete(itemCode);
		}catch(EJBException e){
			m_log.warn("deleteCollectItemCodeMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("deleteCollectItemCodeMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteCollectItemCodeMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(FinderException e){
			m_log.warn("deleteCollectItemCodeMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * 収集項目コードを全て削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data 収集項目コード情報
	 * @return 削除に成功した場合、true
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws EJBException
	 */
	public boolean deleteCollectItemCodeMasterAll() throws EJBException, FinderException, NamingException, RemoveException {

		// 収集項目コードを削除
		OperateCollectItemCodeMaster ope = new OperateCollectItemCodeMaster();
		boolean ret = false;

		try{
			ret = ope.deleteAll();
		}catch(EJBException e){
			m_log.warn("deleteCollectItemCodeMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("deleteCollectItemCodeMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteCollectItemCodeMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(FinderException e){
			m_log.warn("deleteCollectItemCodeMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * 収集方法毎の収集項目を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param data 収集方法・プラットフォーム毎の収集項目情報
	 * @return 削除に成功した場合、true
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws EJBException
	 */
	public boolean deleteCollectPollingMaster(CollectorPollingMstPK pk) throws EJBException, FinderException, NamingException, RemoveException {

		// 収集方法・プラットフォーム毎の収集項目を削除
		OperateCollectPollingMaster ope = new OperateCollectPollingMaster();
		boolean ret = false;

		try{
			ret = ope.delete(pk);
		}catch(EJBException e){
			m_log.warn("deleteCollectPollingMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("deleteCollectPollingMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteCollectPollingMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(FinderException e){
			m_log.warn("deleteCollectPollingMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * 収集方法毎の収集項目を全て削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 削除に成功した場合、true
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws EJBException
	 */
	public boolean deleteCollectPollingMasterAll() throws EJBException, FinderException, NamingException, RemoveException {

		// 収集方法・プラットフォーム毎の収集項目を削除
		OperateCollectPollingMaster ope = new OperateCollectPollingMaster();
		boolean ret = false;

		try{
			ret = ope.deleteAll();
		}catch(EJBException e){
			m_log.warn("deleteCollectPollingMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("deleteCollectPollingMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteCollectPollingMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(FinderException e){
			m_log.warn("deleteCollectPollingMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		return ret;

	}

	/**
	 * プラットフォーム定義を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param platformId プラットフォームID
	 * @return 削除に成功した場合、true
	 * @throws HinemosUnknown
	 */
	public boolean deleteCollectPlatformMaster(String platformId) throws HinemosUnknown {

		try{
			// プラットフォーム定義情報を登録
			CollectorPlatformMstUtil.getLocalHome().remove(platformId);
		}catch(EJBException e){
			m_log.warn("deleteCollectPlatformMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown("deleteCollectPlatformMaster(): " + e.getMessage(), e);
		}catch(NamingException e){
			m_log.warn("deleteCollectPlatformMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown("deleteCollectPlatformMaster(): " + e.getMessage(), e);
		}catch(RemoveException e){
			m_log.warn("deleteCollectPlatformMaster() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown("deleteCollectPlatformMaster(): " + e.getMessage(), e);
		}

		return true;

	}

	/**
	 * 収集項目のマスタ情報を全て削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 削除に成功した場合、true
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 * @throws EJBException
	 */
	public boolean deleteCollectMasterAll() throws HinemosUnknown {

		// 収集項目に関連する全てのマスタ情報を削除
		OperateCollectCategoryCollectMaster opeCategoryCollect = new OperateCollectCategoryCollectMaster();
		OperateCollectPollingMaster opePolling = new OperateCollectPollingMaster();
		OperateCollectItemCalcMethodMaster opeItemCalc = new OperateCollectItemCalcMethodMaster();
		OperateCollectItemCodeMaster opeItemCode = new OperateCollectItemCodeMaster();
		OperateCollectCategoryMaster opeCategory = new OperateCollectCategoryMaster();
		OperateCollectCalcMaster opeCalc = new OperateCollectCalcMaster();

		boolean ret = false;

		try{
			ret = opeCategoryCollect.deleteAll() && opePolling.deleteAll() && opeItemCalc.deleteAll() && opeItemCode.deleteAll() && opeCategory.deleteAll() && opeCalc.deleteAll();
		}catch(EJBException e){
			m_log.warn("deleteCollectMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown("deleteCollectMasterAll(): " + e.getMessage(), e);
		}catch(NamingException e){
			m_log.warn("deleteCollectMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown("deleteCollectMasterAll(): " + e.getMessage(), e);
		}catch(RemoveException e){
			m_log.warn("deleteCollectMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown("deleteCollectMasterAll(): " + e.getMessage(), e);
		}catch(FinderException e){
			m_log.warn("deleteCollectMasterAll() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown("deleteCollectMasterAll(): " + e.getMessage(), e);
		}

		return ret;

	}

	/**
	 * 全計算方法定義を取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 全計算方法定義
	 * @throws NamingException
	 * @throws FinderException
	 */
	public ArrayList<CollectorCalcMethodMstData> getCollectCalcMasterList() throws FinderException, NamingException {

		ArrayList<CollectorCalcMethodMstData> ret = new ArrayList<CollectorCalcMethodMstData>();

		Collection<CollectorCalcMethodMstLocal> list = CollectorCalcMethodMstUtil.getLocalHome().findAll();
		Iterator<CollectorCalcMethodMstLocal> itr = list.iterator();
		while(itr.hasNext()) {
			CollectorCalcMethodMstLocal local = itr.next();
			CollectorCalcMethodMstData data = new CollectorCalcMethodMstData(
					local.getCalcMethod(),
					local.getClassName(),
					local.getExpression());

			ret.add(data);
		}

		return ret;
	}

	/**
	 * カテゴリを取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return カテゴリ定義
	 * @throws NamingException
	 * @throws FinderException
	 */
	public ArrayList<CollectorCategoryMstData> getCollectCategoryList() throws FinderException, NamingException {

		ArrayList<CollectorCategoryMstData> ret = new ArrayList<CollectorCategoryMstData>();

		Collection<CollectorCategoryMstLocal> list = CollectorCategoryMstUtil.getLocalHome().findAll();
		Iterator<CollectorCategoryMstLocal> itr = list.iterator();
		while(itr.hasNext()) {
			CollectorCategoryMstLocal local = itr.next();
			CollectorCategoryMstData data = new CollectorCategoryMstData(
					local.getCategoryCode(),
					local.getCategoryName());

			ret.add(data);
		}

		return ret;
	}

	/**
	 * 収集項目コードを取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 収集項目コード
	 * @throws NamingException
	 * @throws FinderException
	 */
	public ArrayList<CollectorItemCodeMstData> getCollectItemCodeMasterList() throws FinderException, NamingException {

		ArrayList<CollectorItemCodeMstData> ret = new ArrayList<CollectorItemCodeMstData>();

		Collection<CollectorItemCodeMstLocal> list = CollectorItemCodeMstUtil.getLocalHome().findAll();
		Iterator<CollectorItemCodeMstLocal> itr = list.iterator();
		while(itr.hasNext()) {
			CollectorItemCodeMstLocal local = itr.next();
			CollectorItemCodeMstData data = new CollectorItemCodeMstData(
					local.getItemCode(),
					local.getCategoryCode(),
					local.getParentItemCode(),
					local.getItemName(),
					local.getMeasure(),
					local.getDeviceSupport(),
					local.getDeviceType(),
					local.getGraphRange());

			ret.add(data);
		}

		return ret;
	}

	/**
	 * 収集項目定義を取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 収集項目定義
	 * @throws NamingException
	 * @throws FinderException
	 */
	public ArrayList<CollectorItemCalcMethodMstData> getCollectItemCalcMasterList() throws FinderException, NamingException {

		ArrayList<CollectorItemCalcMethodMstData> ret = new ArrayList<CollectorItemCalcMethodMstData>();

		Collection<CollectorItemCalcMethodMstLocal> list = CollectorItemCalcMethodMstUtil.getLocalHome().findAll();
		Iterator<CollectorItemCalcMethodMstLocal> itr = list.iterator();
		while(itr.hasNext()) {
			CollectorItemCalcMethodMstLocal local = itr.next();
			CollectorItemCalcMethodMstData data = new CollectorItemCalcMethodMstData(
					local.getCollectMethod(),
					local.getPlatformId(),
					local.getSubPlatformId(),
					local.getItemCode(),
					local.getCalcMethod());

			ret.add(data);
		}

		return ret;
	}

	/**
	 * ポーリング対象定義を取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return ポーリング対象定義
	 * @throws NamingException
	 * @throws FinderException
	 */
	public ArrayList<CollectorPollingMstData> getCollectPollingList() throws FinderException, NamingException {

		ArrayList<CollectorPollingMstData> ret = new ArrayList<CollectorPollingMstData>();

		Collection<CollectorPollingMstLocal> list = CollectorPollingMstUtil.getLocalHome().findAll();
		Iterator<CollectorPollingMstLocal> itr = list.iterator();
		while(itr.hasNext()) {
			CollectorPollingMstLocal local = itr.next();
			CollectorPollingMstData data = new CollectorPollingMstData(
					local.getCollectMethod(),
					local.getPlatformId(),
					local.getSubPlatformId(),
					local.getItemCode(),
					local.getVariableId(),
					local.getEntryKey(),
					local.getValueType(),
					local.getPollingTarget(),
					local.getFailureValue());

			ret.add(data);
		}

		return ret;
	}

	/**
	 * カテゴリ毎の収集方法を取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return カテゴリ毎の収集方法
	 * @throws NamingException
	 * @throws FinderException
	 */
	public ArrayList<CollectorCategoryCollectMstData> getCollectCategoryCollectList() throws FinderException, NamingException {

		ArrayList<CollectorCategoryCollectMstData> ret = new ArrayList<CollectorCategoryCollectMstData>();

		Collection<CollectorCategoryCollectMstLocal> list = CollectorCategoryCollectMstUtil.getLocalHome().findAll();
		Iterator<CollectorCategoryCollectMstLocal> itr = list.iterator();
		while(itr.hasNext()) {
			CollectorCategoryCollectMstLocal local = itr.next();
			CollectorCategoryCollectMstData data = new CollectorCategoryCollectMstData(
					local.getPlatformId(),
					local.getSubPlatformId(),
					local.getCategoryCode(),
					local.getCollectMethod());

			ret.add(data);
		}

		return ret;
	}

	/**
	 * プラットフォーム定義を取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return プラットフォーム定義のリスト
	 * @throws HinemosUnknown
	 */
	public List<CollectorPlatformMstData> getCollectPlatformMaster() throws HinemosUnknown {

		ArrayList<CollectorPlatformMstData> platformMstList = new ArrayList<CollectorPlatformMstData>();

		try {
			// プラットフォームマスタのリストを取得
			for(CollectorPlatformMstLocal bean : (Collection<CollectorPlatformMstLocal>)CollectorPlatformMstUtil.getLocalHome().findAll()){
				platformMstList.add(new CollectorPlatformMstData(
						bean.getPlatformId(),
						bean.getPlatformName(),
						bean.getOrderNo()
				));
			}
		} catch (FinderException e) {
			throw new HinemosUnknown("getCollectPlatformMaster(): " + e.getMessage(), e);
		} catch (NamingException e) {
			throw new HinemosUnknown("getCollectPlatformMaster(): " + e.getMessage(), e);
		}

		return platformMstList;
	}


	/**
	 * 収集マスタ情報を一括で取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 収集マスタ情報
	 * @throws HinemosUnknown
	 */
	public CollectMasterInfo getCollectMasterInfo() throws HinemosUnknown {
		CollectMasterInfo info = new CollectMasterInfo();
		try {
			info.setCollectorCategoryMstDataList(getCollectCategoryList());
			info.setCollectorItemCodeMstDataList(getCollectItemCodeMasterList());
			info.setCollectorItemCalcMethodMstDataList(getCollectItemCalcMasterList());
			info.setCollectorPollingMstDataList(getCollectPollingList());
			info.setCollectorCategoryCollectMstDataList(getCollectCategoryCollectList());
			info.setCollectorCalcMethodMstDataList(getCollectCalcMasterList());
		} catch (FinderException e) {
			throw new HinemosUnknown("getCollectMasterInfo(): " + e.getMessage(), e);
		} catch (NamingException e) {
			throw new HinemosUnknown("getCollectMasterInfo(): " + e.getMessage(), e);
		}

		return info;
	}
}
