/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.commons.util;

import java.math.BigDecimal;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.bean.ScheduleConstant;
import com.clustercontrol.calendar.ejb.entity.CalendarInfoUtil;
import com.clustercontrol.commons.bean.Schedule;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.util.Messages;

/**
 * Hinemos の入力チェックで使用する共通メソッド
 * @since 4.0
 */
public class CommonValidator {

	/**
	 * 指定されたIDがHinemosのID規則にマッチするかを確認する。
	 * [a-z,A-Z,0-9,-,_]のみ許可する
	 * 
	 * @param id
	 * @throws InvalidSetting
	 */
	public static void validateId(String name, String id, int maxSize) throws InvalidSetting{

		// null check
		if(id == null || "".equals(id)){
			Object[] args = { name };
			throw new InvalidSetting(Messages.getString("message.common.5", args));
		}

		// string check
		validateString(name, id, false, 1, maxSize);

		/** メイン処理 */
		if(!id.matches("^[A-Za-z0-9-_]+$")){
			Object[] args = { id, name };
			throw new InvalidSetting(Messages.getString("message.common.6", args));
		}
	}

	/**
	 * 文字列の長さチェック
	 * 
	 * @param name
	 * @param str
	 * @param nullcheck
	 * @param minSize
	 * @param maxSize
	 * @throws InvalidSetting
	 */
	public static void validateString(String name, String str, boolean nullcheck, int minSize, int maxSize) throws InvalidSetting{
		if(str == null){
			if(nullcheck){
				Object[] args = { name };
				throw new InvalidSetting(Messages.getString("message.common.1", args));
			}
		}
		else{
			int size = str.length();
			if(size < minSize){
				if(size == 0){
					Object[] args = { name };
					throw new InvalidSetting(Messages.getString("message.common.1", args));
				}else{
					Object[] args = { name, Integer.toString(minSize) };
					throw new InvalidSetting(Messages.getString("message.common.3", args));
				}
			}

			if(size > maxSize){
				Object[] args = { name, Integer.toString(maxSize) };
				throw new InvalidSetting(Messages.getString("message.common.2", args));
			}
		}
	}

	/**
	 * 数値の上限下限チェック
	 * @throws InvalidSetting
	 */
	public static void validateDouble(String name, double i, double minSize, double maxSize) throws InvalidSetting {
		if (i < minSize || maxSize < i) {
			Object[] args = {name,
					((new BigDecimal(minSize)).toBigInteger()).toString(),
					((new BigDecimal(maxSize)).toBigInteger()).toString()};
			throw new InvalidSetting(Messages.getString("message.common.4", args));
		}
	}

	/**
	 * 数値の上限下限チェック
	 * @throws InvalidSetting
	 */
	public static void validateInt(String name, int i, int minSize, int maxSize) throws InvalidSetting {
		if (i < minSize || maxSize < i) {
			Object[] args = {name, Integer.toString(minSize), Integer.toString(maxSize)};
			throw new InvalidSetting(Messages.getString("message.common.4", args));
		}
	}

	/**
	 * calendarIdがnullまたは、対象のカレンダ設定が存在するかを確認する
	 * 
	 * @param calendarId
	 * @throws InvalidSetting
	 */
	public static void validateCalenderId(String calendarId, boolean nullcheck) throws InvalidSetting, HinemosUnknown {
		if(calendarId == null || "".equals(calendarId)){
			if(nullcheck){
				throw new InvalidSetting(Messages.getString("message.common.7"));
			}
			return;
		}
		else{
			try{
				CalendarInfoUtil.getLocalHome().findByPrimaryKey(calendarId);
			} catch (NamingException e) {
				throw new HinemosUnknown("unknown error. CalendarId = " + calendarId, e);
			} catch (FinderException e) {
				Object[] args = {calendarId};
				throw new InvalidSetting(Messages.getString("message.common.8", args));
			}
		}
		return;
	}

	/**
	 * notifyIdがnullまたは、対象の通知設定が存在するかを確認する
	 * 
	 * @param notifyId
	 * @param nullcheck
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	public static void validateNotifyId(String notifyId, boolean nullcheck) throws InvalidSetting, HinemosUnknown {
		if(notifyId == null){
			if(nullcheck){
				throw new InvalidSetting(Messages.getString("message.common.9"));
			}
			return;
		}
		else{
			try{
				NotifyInfoUtil.getLocalHome().findByPrimaryKey(notifyId);
			} catch (NamingException e) {
				throw new HinemosUnknown("unknown error. notifyId = " + notifyId, e);
			} catch (FinderException e) {
				Object[] args = {notifyId};
				throw new InvalidSetting(Messages.getString("message.common.10", args));
			}
		}
		return;
	}

	/**
	 * schedule型のチェック
	 */
	public static void validateScheduleHour(Schedule schedule) throws InvalidSetting {
		validateSchedule(schedule);
		// 分だけでなく、時も必須。
		if (schedule.getHour() == null ||
				schedule.getHour() < 0 || 24 <= schedule.getHour()) {
			throw new InvalidSetting(Messages.getString("message.job.28"));
		}
	}

	/**
	 * schedule型のチェック
	 */
	public static void validateSchedule(Schedule schedule) throws InvalidSetting {
		boolean emptyFlag = true;
		if (schedule.getType() == ScheduleConstant.TYPE_DAY) {
			if (schedule.getMonth() != null) {
				emptyFlag = false;
				if (schedule.getMonth() < 0 || 12 < schedule.getMonth()) {
					throw new InvalidSetting(Messages.getString("message.job.26"));
				}
			}
			if (schedule.getDay() != null) {
				emptyFlag = false;
				if (schedule.getDay() < 0 || 31 < schedule.getDay()) {
					throw new InvalidSetting(Messages.getString("message.job.27"));
				}
			} else if (!emptyFlag){
				// 月を入力した場合は日も必須。
				throw new InvalidSetting(Messages.getString("message.job.27"));
			}
		} else if (schedule.getType() == ScheduleConstant.TYPE_WEEK) {
			if (schedule.getWeek() == null ||
					schedule.getWeek() < 0 || 7 < schedule.getWeek()) {
				throw new InvalidSetting(Messages.getString("message.job.37"));
			}
		} else {
			throw new InvalidSetting("unknown schedule type");
		}

		if (schedule.getHour() != null) {
			emptyFlag = false;
			if (schedule.getHour() < 0 || 24 < schedule.getHour()) {
				throw new InvalidSetting(Messages.getString("message.job.28"));
			}
		} else if (!emptyFlag){
			// 日を入力した場合は時間も必須。
			throw new InvalidSetting(Messages.getString("message.job.28"));
		}

		if (schedule.getMinute() != null) {
			emptyFlag = false;
			if (schedule.getMinute() < 0 || 60 < schedule.getMinute()) {
				throw new InvalidSetting(Messages.getString("message.job.29"));
			}
		} else {
			// 分は必須。
			throw new InvalidSetting(Messages.getString("message.job.29"));
		}
	}

	/**
	 * for debug
	 * @param args
	 */
	public static void main(String[] args) {

		String id;

		// OKのパターン(半角英数字、-(半角ハイフン)、_(半角アンダーバー))
		id = "Linux-_1";
		try{
			validateId("name", id , 64);
			System.out.println("id = " + id + " is OK");
		} catch (Exception e) {
			System.out.println("???");
			e.printStackTrace();
		}

		// NGのパターン
		id = "/?/";
		try{
			validateId("name", id , 64);
		} catch (Exception e) {
			System.out.println("id = " + id + " is NG");
		}

	}


}
