/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.factory;

import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.JobInfoNotFound;
import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.jobmanagement.bean.DelayNotifyConstant;
import com.clustercontrol.jobmanagement.bean.EndStatusCheckConstant;
import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodePK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartInfoLocal;

/**
 * ジョブ操作の開始[即時]を行うクラスです。
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class OperateStartOfJob extends OperationJob {
	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( OperateStartOfJob.class );

	/**
	 * コンストラクタ
	 */
	public OperateStartOfJob(){
		super();
	}

	/**
	 * ジョブを開始[即時]します。
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @throws NamingException
	 * @throws EJBException
	 * @throws RemoveException
	 * @throws JobInfoNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#setStatus(String, String, String, Integer)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#startJob(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#startJobMain1(String, String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateStartOfJob#clearStatus(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.Notice#notify(String, String, Integer)
	 */
	public void startJob(String sessionId, String jobunitId, String jobId, String facilityId) throws NamingException, EJBException, RemoveException, JobInfoNotFound, HinemosUnknown {
		m_log.debug("startJob() : sessionId=" + sessionId + ", jobunitId=" + jobunitId  + ", jobId=" + jobId + ", facilityId=" + facilityId);

		//実行状態クリア
		clearStatus(sessionId, jobunitId, jobId, facilityId);

		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = null;
		try {
			sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
					new JobSessionJobPK(sessionId, jobunitId, jobId));
		} catch (FinderException e) {
			m_log.warn("JobSessionJobLocal.findByPrimaryKey()" + e.getMessage());
			JobInfoNotFound je = new JobInfoNotFound(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			throw je;
		}

		//リレーションを取得し、親ジョブのジョブIDを取得
		JobRelationInfoLocal relation = sessionJob.getJobRelationInfo();
		String parentJobUnitId = relation.getParent_jobunit_id();
		String parentJobId = relation.getParent_job_id();
		relation = null;

		//親ジョブネットの実行状態を設定
		if(parentJobId.compareTo("TOP") == 0){
			sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_RUNNING));
			sessionJob.setStart_date(new Date());
			//通知処理
			new Notice().notify(sessionId, jobunitId, jobId, Integer.valueOf(EndStatusConstant.TYPE_BEGINNING));
		}
		else{
			setStatus(sessionId, parentJobUnitId, parentJobId, Integer.valueOf(StatusConstant.TYPE_RUNNING));
		}

		//ジョブを取得
		JobInfoLocal job = sessionJob.getJobInfo();
		sessionJob = null;
		if(job.getJob_type().intValue() == JobConstant.TYPE_JOBUNIT){
			//ジョブユニットの場合、ジョブ開始処理を行う
			startJob(sessionId, jobunitId, jobId);
		}
		else{
			//ジョブネット、ジョブの場合、ジョブ開始処理メイン1を行う
			startJobMain1(sessionId, jobunitId, jobId, facilityId);
		}
	}

	/**
	 * ジョブの実行結果をクリアします。
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @throws NamingException
	 * @throws EJBException
	 * @throws RemoveException
	 * @throws JobInfoNotFound
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#clearJob(String, String)
	 */
	private void clearStatus(String sessionId, String jobunitId, String jobId, String facilityId) throws NamingException, EJBException, RemoveException, JobInfoNotFound {
		m_log.debug("clearStatus() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId + ", facilityId=" + facilityId);

		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = null;
		try {
			sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
					new JobSessionJobPK(sessionId, jobunitId, jobId));
		} catch (FinderException e) {
			m_log.warn("JobSessionJobLocal.findByPrimaryKey() : " + e.getMessage());
			JobInfoNotFound je = new JobInfoNotFound(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			throw je;
		}

		if(sessionJob != null){
			JobInfoLocal job = sessionJob.getJobInfo();
			JobStartInfoLocal start = job.getJobStartInfo();

			//終了状態、終了値、終了・中断日時、開始・再実行日時をクリア
			sessionJob.setEnd_status(null);
			sessionJob.setEnd_value(null);
			sessionJob.setEnd_date(null);
			sessionJob.setStart_date(null);
			sessionJob.setResult(null);
			sessionJob.setEnd_staus_check_flg(Integer.valueOf(EndStatusCheckConstant.NO_WAIT_JOB));
			sessionJob.setDelay_notify_flg(Integer.valueOf(DelayNotifyConstant.NONE));

			//開始時保留、開始時スキップをチェック
			if(start != null){
				if(start.getSuspend().intValue() == YesNoConstant.TYPE_YES){
					//JobSessionJobの実行状態に保留中を設定
					sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_RESERVING));
				}
				else if(start.getSkip().intValue() == YesNoConstant.TYPE_YES){
					//JobSessionJobの実行状態にスキップを設定
					sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_SKIP));
					//JobSessionJobの終了値を設定
					sessionJob.setEnd_value(start.getSkip_end_value());
				}
				else{
					//JobSessionJobの実行状態に待機を設定
					sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_WAIT));
				}
			}
			else{
				//JobSessionJobの実行状態に待機を設定
				sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_WAIT));
			}

			if(job.getJob_type().intValue() == JobConstant.TYPE_JOB){
				//ジョブの場合
				if(facilityId != null && facilityId.length() > 0){
					sessionJob = null;

					try {
						//ノードの場合
						JobSessionNodeLocal sessionNode =
							JobSessionNodeUtil.getLocalHome().findByPrimaryKey(
									new JobSessionNodePK(sessionId, jobunitId, jobId, facilityId));

						//セッションノードの実行状態に待機を設定
						sessionNode.setStatus(Integer.valueOf(StatusConstant.TYPE_WAIT));

						//終了値、終了・中断日時、開始・再実行日時をクリア
						sessionNode.setEnd_value(null);
						sessionNode.setEnd_date(null);
						sessionNode.setStart_date(null);
						sessionNode.setAgent_check(null);
						sessionNode.setCheck_date(null);
						sessionNode.setRetry_count(Integer.valueOf(0));
						sessionNode.setResult(null);
						sessionNode.setMessage(null);
						sessionNode.setStatus_buffer(null);
					} catch (FinderException e) {
						m_log.warn("JobSessionNodeLocal.findByPrimaryKey" + e.getMessage());
						JobInfoNotFound je = new JobInfoNotFound(e.getMessage(), e);
						je.setSessionId(sessionId);
						je.setJobunitId(jobunitId);
						je.setJobId(jobId);
						throw je;
					}
				}
				else{
					Collection collection = sessionJob.getJobSessionNode();
					sessionJob = null;
					if(collection != null && collection.size() > 0){
						Iterator itr = collection.iterator();
						while(itr.hasNext()){
							JobSessionNodeLocal sessionNode = (JobSessionNodeLocal)itr.next();

							//セッションノードの実行状態に待機を設定
							sessionNode.setStatus(Integer.valueOf(StatusConstant.TYPE_WAIT));

							//終了値、終了・中断日時、開始・再実行日時をクリア
							sessionNode.setEnd_value(null);
							sessionNode.setEnd_date(null);
							sessionNode.setStart_date(null);
							sessionNode.setAgent_check(null);
							sessionNode.setCheck_date(null);
							sessionNode.setRetry_count(Integer.valueOf(0));
							sessionNode.setResult(null);
							sessionNode.setMessage(null);
							sessionNode.setStatus_buffer(null);
						}
					}
				}
			}
			else if(job.getJob_type().intValue() == JobConstant.TYPE_FILEJOB){
				//ジョブ以外の場合
				sessionJob = null;

				//セッションIDとジョブIDから、直下のジョブを取得
				Collection collection = null;
				try {
					collection = JobRelationInfoUtil.getLocalHome().findByParentJobId(sessionId, jobId);
				} catch (FinderException e) {
					m_log.warn("JobRelationInfoLocal.findByParentJobId() : " + e.getMessage());
					JobInfoNotFound je = new JobInfoNotFound();
					je.setSessionId(sessionId);
					je.setParentJobId(jobId);
					throw je;
				}
				if(collection != null && collection.size() > 0){
					Iterator itr = collection.iterator();
					while(itr.hasNext()){
						JobRelationInfoLocal relation = (JobRelationInfoLocal)itr.next();

						if(relation.getJob_id().equals(jobId + CreateFileJob.FILE_LIST))
							//ジョブの実行状態をクリアする
							clearStatus(relation.getSession_id(), relation.getJobunit_id(), relation.getJob_id(), null);
						else
							//ジョブ削除を行う
							clearJob(relation.getSession_id(), relation.getJobunit_id(), relation.getJob_id());

					}
				}
			}
			else{
				//ジョブ以外の場合
				sessionJob = null;

				//セッションIDとジョブIDから、直下のジョブを取得
				Collection collection = null;
				try {
					collection = JobRelationInfoUtil.getLocalHome().findByParentJobId(sessionId, jobId);
				} catch (FinderException e) {
					m_log.warn("obRelationInfoLocal.findByParentJobId() : " + e.getMessage());
					JobInfoNotFound je = new JobInfoNotFound(e.getMessage(), e);
					je.setSessionId(sessionId);
					je.setParentJobId(jobId);
					throw je;
				}
				if(collection != null && collection.size() > 0){
					Iterator itr = collection.iterator();
					while(itr.hasNext()){
						JobRelationInfoLocal relation = (JobRelationInfoLocal)itr.next();

						//ジョブの実行状態をクリアする
						clearStatus(relation.getSession_id(), relation.getJobunit_id(), relation.getJob_id(), null);
					}
				}
			}
		}
	}

}
