/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.factory;

import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.JobMasterNotFound;
import com.clustercontrol.commons.bean.Schedule;
import com.clustercontrol.jobmanagement.bean.JobSchedule;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobScheduleLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSchedulePK;
import com.clustercontrol.jobmanagement.ejb.entity.JobScheduleUtil;
import com.clustercontrol.jobmanagement.util.ManagementUserUtil;

/**
 * スケジュール一覧情報を検索するクラスです。
 *
 * @version 4.0.0
 * @since 1.0.0
 */
public class SelectSchedule {
	private static Log m_log = LogFactory.getLog( SelectSchedule.class );

	/**
	 * scheduleIdのジョブスケジュールを取得します
	 * 
	 * @param scheduleId
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 * @throws JobMasterNotFound
	 */
	public JobSchedule getJobSchedule(String scheduleId) throws FinderException, NamingException, JobMasterNotFound{
		m_log.debug("getJobSchedule() scheduleId = " + scheduleId);
		JobScheduleLocal scheduleBean = JobScheduleUtil.getLocalHome().findByPrimaryKey(new JobSchedulePK(scheduleId));
		return createJobScheduleInfo(scheduleBean);
	}

	/**
	 * スケジュール一覧情報を取得します。
	 * 
	 * @return スケジュール一覧情報
	 * @throws NamingException
	 * @throws SchedulerException
	 * @throws JobMasterNotFound
	 */
	public ArrayList<JobSchedule> getScheduleList(String userId) throws NamingException, JobMasterNotFound {
		m_log.debug("getScheduleList()");

		ArrayList<JobSchedule> list = new ArrayList<JobSchedule>();

		Collection<JobScheduleLocal> jobScheduleList;
		try {
			jobScheduleList = JobScheduleUtil.getLocalHome().findAll();
		} catch (FinderException e) {
			m_log.warn("JobScheduleLocal.findAll" + e.getMessage());
			JobMasterNotFound je = new JobMasterNotFound(e.getMessage(), e);
			throw je;
		}

		for(JobScheduleLocal scheduleBean : jobScheduleList){
			//参照可能なジョブユニットIDを持つスケジュールを取得する
			if (ManagementUserUtil.isReferable(scheduleBean.getJobunit_id(), userId)) {
				list.add(createJobScheduleInfo(scheduleBean));
			}
		}
		return list;
	}


	/**
	 * JobScheduleLocalよりJobScheduleを作成するクラス
	 * 
	 * @param scheduleBean
	 * @return
	 * @throws NamingException
	 * @throws JobMasterNotFound
	 */
	private JobSchedule createJobScheduleInfo(JobScheduleLocal scheduleBean) throws NamingException, JobMasterNotFound{

		JobSchedule info = new JobSchedule();
		//スケジュールIDを取得
		info.setId(scheduleBean.getSchedule_id());
		//スケジュール名を取得
		info.setName(scheduleBean.getSchedule_name());
		//ジョブIDを取得
		info.setJobId(scheduleBean.getJob_id());
		//ジョブ名を取得
		try {
			String jobName = JobMasterUtil.getLocalHome().findByPrimaryKey(new JobMasterPK(scheduleBean.getJobunit_id(), scheduleBean.getJob_id())).getJob_name();
			info.setJobName(jobName);
		} catch (FinderException e) {
			m_log.warn("JobMasterLocal.findByPrimaryKey" + e.getMessage());
			JobMasterNotFound je = new JobMasterNotFound(e.getMessage(), e);
			je.setJobunitId(scheduleBean.getJobunit_id());
			je.setJobId(scheduleBean.getJob_id());
			throw je;
		}
		//ジョブユニットIDを取得
		info.setJobunitId(scheduleBean.getJobunit_id());
		//カレンダIDを取得
		info.setCalendarId(scheduleBean.getCalendar_id());

		//スケジュール定義を取得
		Schedule schedule = new Schedule(scheduleBean.getSchedule_type(),
				scheduleBean.getMonth(), scheduleBean.getDay(),
				scheduleBean.getWeek(), scheduleBean.getHour(), scheduleBean.getMinute());
		scheduleBean.getSchedule_type();

		info.setSchedule(schedule);
		//有効/無効を取得
		info.setValid(scheduleBean.getValid_flg());
		//登録者を取得
		info.setCreateUser(scheduleBean.getReg_user());
		//登録日時を取得
		if (scheduleBean.getReg_date() != null) {
			info.setCreateTime(scheduleBean.getReg_date().getTime());
		}
		//更新者を取得
		info.setUpdateUser(scheduleBean.getUpdate_user());
		//更新日時を取得
		if (scheduleBean.getUpdate_date() != null) {
			info.setUpdateTime(scheduleBean.getUpdate_date().getTime());
		}
		return info;
	}
}
