/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.util;

import java.util.ArrayList;
import java.util.HashSet;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.JobInvalid;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobManagementUserInfo;
import com.clustercontrol.jobmanagement.bean.JobObjectInfo;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobWaitRuleInfo;
import com.clustercontrol.jobmanagement.bean.JudgmentObjectConstant;
import com.clustercontrol.util.Messages;

/**
 * ジョブユーティリティクラス
 * 
 * 以下ィを提供します。<BR>
 * <li>ジョブツリーアイテムに関するユーティリティ
 * <li>ログインユーザが参照可能なジョブユニットかどうかをチェックするユーティリティ
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class JobUtil {
	/** ログ出力のインスタンス<BR> */
	private static Log m_log = LogFactory.getLog( JobUtil.class );

	/** コピーしたジョブIDの接頭語 */
	private static final String COPY_OF = "Copy_Of_";

	/**
	 * ジョブツリーアイテムからジョブIDが一致するインスタンスの有無を返す<BR>
	 * ジョブツリーアイテムのツリー階層の全てが検索対象
	 * 
	 * @param jobId ジョブID
	 * @param item ジョブツリーアイテム
	 * @return ジョブIDが一致するジョブツリーアイテムがあればtrue、なければfalse。
	 */
	public static boolean findJobId(String jobId, JobTreeItem item) {
		boolean find = false;

		//ジョブIDをチェック
		JobInfo info = item.getData();
		if(info != null && info instanceof JobInfo){
			if(jobId.compareTo(info.getId()) == 0){
				find = true;
				return find;
			}
		}

		//子JobTreeItemを取得
		JobTreeItem[] childrens = item.getChildrenArray();
		for(int i = 0; i < childrens.length; i++){
			find = findJobId(jobId, childrens[i]);
			if(find){
				break;
			}
		}

		return find;
	}

	/**
	 * ジョブツリーアイテム内のジョブIDが一致するインスタンスの有無を返す<BR>
	 * 
	 * @param item ジョブツリーアイテム
	 * @param isTop ツリーのトップか否か
	 * @return ジョブIDが一致するジョブ/ネット/ユニットが存在すればtrue、なければfalse。
	 * @throws JobInvalid
	 */
	public static void findDuplicateJobId(JobTreeItem item, boolean isTop) throws JobInvalid{
		m_log.debug("findDuplicateJobId() start : isTop = " + isTop);

		// 自身がtopの場合は何もしない
		if(!isTop){

			// 自身がジョブユニット/ジョブネット/ジョブの場合
			if(item.getData() instanceof JobInfo){
				String jobId = item.getData().getId();
				m_log.debug("findDuplicateJobId() jobId = " + jobId);

				JobTreeItem[] children = item.getChildrenArray();
				for (JobTreeItem child : children) {

					m_log.debug("findDuplicateJobId() child = " + child.getData().getId());

					// 配下のツリーにトップのジョブIDが含まれるか？
					if(findJobId(jobId, child)){

						// jobunitid内にjobidが重複している
						m_log.debug("findDuplicateJobId() jobId is in child " + child.getPath());
						Object[] args = {jobId, child.getData().getJobunitId()};
						throw new JobInvalid(Messages.getString("message.job.65",args));
					} else {
						m_log.debug("findDuplicateJobId() jobId is not in child " + child.getPath());
					}
				}
			}
		}

		JobTreeItem[] children = item.getChildrenArray();
		for (JobTreeItem child : children) {
			m_log.debug("findDuplicateJobId() call child " + child.getData().getId());
			findDuplicateJobId(child, false);
		}

		m_log.debug("findDuplicateJobId() success!!");
	}

	/**
	 * ジョブツリーアイテム内のジョブユニットIDが一致するインスタンスの有無を返す<BR>
	 * 
	 * @param item ジョブツリーアイテム
	 * @return ジョブユニットIDが一致するユニットが存在すればtrue、なければfalse。
	 * @throws JobInvalid
	 */
	public static void findDuplicateJobunitId(JobTreeItem item) throws JobInvalid{
		m_log.debug("findDuplicateJobunitId() start " + item.getPath());

		JobTreeItem top = item.getChildren(0);
		JobTreeItem[] jobunits = top.getChildrenArray();

		HashSet<String> set = new HashSet<String>();
		for (JobTreeItem jobunit : jobunits) {
			// ジョブユニットID
			String jobunitId = jobunit.getData().getJobunitId();
			m_log.debug("findDuplicateJobunitId() jobunitId = " + jobunitId);

			if(set.contains(jobunitId)){
				m_log.debug("findDuplicateJobunitId() hit " + jobunitId);
				Object[] args = {jobunitId};
				throw new JobInvalid(Messages.getString("message.job.64",args));
			} else {
				m_log.debug("findDuplicateJobunitId() add " + jobunitId + " to set");
				set.add(jobunitId);
			}
		}

		m_log.debug("findDuplicateJobunitId() success!!");
	}

	/**
	 * ジョブツリーアイテムのジョブ待ち条件情報をチェックする
	 * 
	 * @param item ジョブ待ち条件情報をチェックするジョブツリーアイテム
	 */
	public static boolean checkWaitRule(JobTreeItem item) throws JobInvalid{
		boolean check = true;

		if(item == null)
			return check;

		if(item.getData() != null && item.getData() instanceof JobInfo){
			//ジョブID取得
			String jobId = item.getData().getId();

			//待ち条件情報を取得する
			JobWaitRuleInfo waitRule = item.getData().getWaitRule();
			if(waitRule != null && waitRule instanceof JobWaitRuleInfo &&
					waitRule.getObject() != null && waitRule.getObject().size() > 0){

				for (JobObjectInfo objectInfo : waitRule.getObject()) {
					if(objectInfo.getType() != JudgmentObjectConstant.TYPE_TIME){
						//判定対象のジョブIDが同一階層に存在するかチェック
						boolean find = false;
						String targetJobId = objectInfo.getJobId();
						JobTreeItem[] childrens = item.getParent().getChildrenArray();
						for(int i = 0; i < childrens.length; i++){
							//ジョブIDをチェック
							JobInfo childInfo = childrens[i].getData();
							if(childInfo != null && childInfo instanceof JobInfo &&
									!jobId.equals(childInfo.getId())){
								if(targetJobId.compareTo(childInfo.getId()) == 0){
									find = true;
									break;
								}
							}
						}
						if(!find){
							String args[] = {jobId, targetJobId};
							throw new JobInvalid(Messages.getString("message.job.59", args));
						}
					}
				}
			}
		}

		//子JobTreeItemを取得
		JobTreeItem[] childrens = item.getChildrenArray();
		for(int i = 0; i < childrens.length; i++){
			check = checkWaitRule(childrens[i]);
			if(!check){
				break;
			}
		}

		return check;
	}

	/**
	 * ログインユーザで参照可能なジョブユニットかどうかチェックします。<BR>
	 * @param jobunit 選択されたジョブユニット
	 * @param loginUser ログインユーザのユーザID
	 * @return ログインユーザで参照可能なジョブユニットである場合true
	 */
	public static boolean isReferableJobunit (JobTreeItem jobunit, String loginUser) {

		boolean ret = false;

		ArrayList<JobManagementUserInfo> userInfo = jobunit.getData().getManagementUser();
		if (userInfo == null || userInfo.size() == 0) {
			ret = true;
		} else {
			for (JobManagementUserInfo i : userInfo) {
				if (loginUser.equals(i.getUserId())) {
					ret = true;
					break;
				}
			}
		}

		return ret;
	}
}