/*

Copyright (C) 2007 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.maintenance.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CalendarNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.MaintenanceDuplicate;
import com.clustercontrol.fault.MaintenanceNotFound;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.ejb.session.CalendarControllerLocal;
import com.clustercontrol.calendar.ejb.session.CalendarControllerUtil;
import com.clustercontrol.commons.scheduler.TriggerSchedulerException;
import com.clustercontrol.maintenance.bean.MaintenanceInfo;
import com.clustercontrol.maintenance.bean.MaintenanceTypeMst;
import com.clustercontrol.maintenance.factory.AddMaintenance;
import com.clustercontrol.maintenance.factory.DeleteMaintenance;
import com.clustercontrol.maintenance.factory.MaintenanceCrun;
import com.clustercontrol.maintenance.factory.MaintenanceEvent;
import com.clustercontrol.maintenance.factory.MaintenanceJob;
import com.clustercontrol.maintenance.factory.MaintenancePerf;
import com.clustercontrol.maintenance.factory.MaintenanceVM;
import com.clustercontrol.maintenance.factory.ModifyMaintenance;
import com.clustercontrol.maintenance.factory.ModifySchedule;
import com.clustercontrol.maintenance.factory.OperationMaintenance;
import com.clustercontrol.maintenance.factory.SelectMaintenanceInfo;
import com.clustercontrol.maintenance.factory.SelectMaintenanceTypeMst;
import com.clustercontrol.maintenance.util.MaintenanceValidator;

/**
 * 
 * メンテナンス機能を管理する Session Bean です。<BR>
 * 
 * @ejb.bean name="MaintenanceController"
 *           display-name="MaintenanceController"
 *           description="MaintenanceController"
 *           jndi-name="MaintenanceController"
 *           type="Stateless"
 *           view-type="local"
 * 
 * @ejb.transaction type="Required"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MaintenanceInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MaintenanceTypeMstLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 */
public abstract class MaintenanceControllerBean implements SessionBean {

	private static Log m_log = LogFactory.getLog( MaintenanceControllerBean.class );

	/** コンテキスト情報 */
	@SuppressWarnings("unused")
	private SessionContext m_context;

	public MaintenanceControllerBean() {
		super();
	}

	/**
	 * メンテナンス情報を追加します。
	 * 
	 * @throws HinemosUnknown
	 * @throws MaintenanceDuplicate
	 * 
	 * @ejb.interface-method
	 * 
	 */
	public void addMaintenance(MaintenanceInfo maintenanceInfo)
	throws HinemosUnknown, MaintenanceDuplicate,InvalidSetting {
		m_log.debug("addMaintenance");

		Principal pri = m_context.getCallerPrincipal();

		// メイン処理
		try {
			// 入力チェック
			MaintenanceValidator.validateMaintenanceInfo(maintenanceInfo);

			// メンテナンス情報を登録
			AddMaintenance add = new AddMaintenance();
			add.addMaintenance(maintenanceInfo, pri.getName());

			// Quartzへ登録
			ModifySchedule modify = new ModifySchedule();
			modify.addSchedule(maintenanceInfo, pri.getName());

		} catch (SecurityException e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (HinemosUnknown e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (DuplicateKeyException e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new MaintenanceDuplicate(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (RemoteException e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (TriggerSchedulerException e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (InvalidSetting e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new InvalidSetting(e.getMessage(), e);
		}
	}

	/**
	 * メンテナンス情報を変更します。
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws MaintenanceNotFound
	 * 
	 * @ejb.interface-method
	 * 
	 */
	public void modifyMaintenance(MaintenanceInfo maintenanceInfo) throws HinemosUnknown, NotifyNotFound, MaintenanceNotFound,InvalidSetting{
		m_log.debug("modifyMaintenance");

		Principal pri = m_context.getCallerPrincipal();

		// メイン処理
		try {
			// 入力チェック
			MaintenanceValidator.validateMaintenanceInfo(maintenanceInfo);

			// メンテナンス情報を登録
			ModifyMaintenance modify = new ModifyMaintenance();
			modify.modifyMaintenance(maintenanceInfo, pri.getName());

			// Quartzへ登録
			ModifySchedule modifySchedule = new ModifySchedule();
			modifySchedule.addSchedule(maintenanceInfo, pri.getName());

		} catch (SecurityException e) {
			m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (RemoveException e) {
			m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NotifyNotFound e) {
			m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (HinemosUnknown e) {
			m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new MaintenanceNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (TriggerSchedulerException e) {
			m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (RemoteException e) {
			m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (InvalidSetting e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new InvalidSetting(e.getMessage(), e);
		}
	}

	/**
	 * メンテナンス情報を削除します。
	 * @throws HinemosUnknown
	 * @throws MaintenanceNotFound
	 * @throws NotifyNotFound
	 * 
	 * @ejb.interface-method
	 * 
	 */
	public void deleteMaintenance(String maintenanceId) throws HinemosUnknown, MaintenanceNotFound, NotifyNotFound{
		m_log.debug("deleteMaintenance");

		try {
			// メンテナンス情報を削除
			DeleteMaintenance delete = new DeleteMaintenance();
			delete.deleteMaintenance(maintenanceId);

			ModifySchedule modify = new ModifySchedule();
			modify.deleteSchedule(maintenanceId);
		} catch (SecurityException e) {
			m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (FinderException e) {
			m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new MaintenanceNotFound(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NotifyNotFound e) {
			m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (HinemosUnknown e) {
			m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (RemoveException e) {
			m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (TriggerSchedulerException e) {
			m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}


	/**
	 * メンテナンス情報を取得します。
	 *
	 * @ejb.interface-method
	 * 
	 * @return
	 * @throws NotifyNotFound
	 * @throws MaintenanceNotFound
	 * @throws HinemosUnknown
	 * 
	 */
	public MaintenanceInfo getMaintenanceInfo(String maintenanceId) throws NotifyNotFound, MaintenanceNotFound, HinemosUnknown {
		m_log.debug("getMaintenanceInfo()");

		SelectMaintenanceInfo select = new SelectMaintenanceInfo();
		MaintenanceInfo info;
		try {
			info = select.getMaintenanceInfo(maintenanceId);
		} catch (NotifyNotFound e) {
			throw e;
		} catch (HinemosUnknown e) {
			throw e;
		}
		return info;
	}

	/**
	 * メンテナンス情報の一覧を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @return メンテナンス情報の一覧を保持するArrayList
	 * @throws MaintenanceNotFound
	 * @throws HinemosUnknown
	 * 
	 */
	public ArrayList<MaintenanceInfo> getMaintenanceList() throws HinemosUnknown, MaintenanceNotFound {
		m_log.debug("getMaintenanceList()");

		SelectMaintenanceInfo select = new SelectMaintenanceInfo();
		ArrayList<MaintenanceInfo> list;
		try {
			list = select.getMaintenanceList();
		} catch (HinemosUnknown e) {
			throw e;
		} catch (MaintenanceNotFound e) {
			throw e;
		}

		return list;

	}

	/**
	 * メンテナンス種別の一覧を取得します。<BR>
	 * 下記のようにして生成されるArrayListを、要素として持つArrayListが一覧として返されます。
	 * 
	 * <p>
	 * MaintenanceTypeMstLocal mst = (MaintenanceTypeMstLocal)itr.next();
	 * ArrayList info = new ArrayList();
	 * info.add(mst.getType_id());
	 * info.add(mst.getName_id());
	 * info.add(mst.getOrder_no());
	 * ist.add(info);
	 * </p>
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @return メンテナンス種別の一覧を保持するArrayList
	 * @throws MaintenanceNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MaintenanceTypeMst> getMaintenanceTypeList() throws MaintenanceNotFound, HinemosUnknown {
		m_log.debug("getMaintenanceTypeList()");

		SelectMaintenanceTypeMst select = new SelectMaintenanceTypeMst();
		ArrayList<MaintenanceTypeMst> list;
		try {
			list = select.getMaintenanceTypeList();
		} catch (FinderException e) {
			throw new MaintenanceNotFound(e.getMessage(), e);
		} catch (NamingException e) {
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (CreateException e) {
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return list;
	}

	/**
	 * 
	 * メンテナンスの有効、無効を変更するメソッドです。
	 * @throws HinemosUnknown
	 * @throws MaintenanceNotFound
	 * @throws NotifyNotFound
	 * 
	 * @ejb.interface-method
	 */
	public void setMaintenanceStatus(String maintenanceId, boolean validFlag) throws NotifyNotFound, MaintenanceNotFound, HinemosUnknown{
		m_log.debug("setMaintenanceStatus() : maintenanceId=" + maintenanceId + ", validFlag=" + validFlag);
		// null check
		if(maintenanceId == null || "".equals(maintenanceId)){
			throw new HinemosUnknown("target maintenanceId is null or empty.");
		}

		MaintenanceInfo info = this.getMaintenanceInfo(maintenanceId);
		info.setValidFlg(ValidConstant.booleanToType(validFlag));

		try{
			this.modifyMaintenance(info);
		} catch (InvalidSetting  e) {
			m_log.warn("Unknown InvalidSettingException", e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * イベントログを削除します。
	 * 
	 * @param dataRetentionPeriod 保存期間(日)
	 * @param status 削除対象のステータス(true=全イベント、false=確認済みイベント)
	 * @return 削除件数
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws MaintenanceNotFound
	 * 
	 * @ejb.interface-method
	 * 
	 * 時間を要する処理のため、NotSupportedを採用してJTAの管理下から除外する
	 * @ejb.transaction
	 *     type="NotSupported"
	 */
	public int deleteEventLog(int dataRetentionPeriod, boolean status) throws HinemosUnknown, NotifyNotFound, MaintenanceNotFound{
		m_log.debug("deleteEventLog() : dataRetentionPeriod = " + dataRetentionPeriod + ", status = " + status);
		MaintenanceEvent event = new MaintenanceEvent();

		int ret = 0;

		try{
			ret = event.delete(dataRetentionPeriod, status);
		}catch(NamingException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(CreateException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(SQLException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(RemoveException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(HinemosUnknown e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		}catch(NotifyNotFound e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (FinderException e) {
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new MaintenanceNotFound(e.getMessage(), e);
		}

		return ret;
	}


	/**
	 * ジョブ実行履歴を削除します。
	 * 
	 * @param dataRetentionPeriod 保存期間(日)
	 * @param status 削除対象のステータス(true=全履歴、false=実行状態が「終了」または「変更済み」の履歴)
	 * @return 削除件数
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws MaintenanceNotFound
	 * 
	 * @ejb.interface-method
	 * 
	 * 時間を要する処理のため、NotSupportedを採用してJTAの管理下から除外する
	 * @ejb.transaction
	 *     type="NotSupported"
	 */
	public int deleteJobHistory(int dataRetentionPeriod, boolean status) throws HinemosUnknown, NotifyNotFound, MaintenanceNotFound {
		m_log.debug("deleteJobHistory() : dataRetentionPeriod = " + dataRetentionPeriod + ", status = " + status);
		MaintenanceJob job = new MaintenanceJob();

		int ret = 0;

		try{
			ret = job.delete(dataRetentionPeriod, status);
		}catch(NamingException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(CreateException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(SQLException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(RemoveException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(HinemosUnknown e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		}catch(NotifyNotFound e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (FinderException e) {
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new MaintenanceNotFound(e.getMessage(), e);
		}

		return ret;
	}

	/**
	 * 性能実績を削除します。
	 * 
	 * @param dataRetentionPeriod 保存期間(日)
	 * @param status 削除対象のステータス(true=全履歴、false=実行状態が「終了」または「変更済み」の履歴)
	 * @return 削除件数
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws MaintenanceNotFound
	 * 
	 * @ejb.interface-method
	 * 
	 * 時間を要する処理のため、NotSupportedを採用してJTAの管理下から除外する
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 */
	public int deletePerfData(int dataRetentionPeriod, boolean status) throws HinemosUnknown, NotifyNotFound, MaintenanceNotFound{
		m_log.debug("deleteJobHistory() : dataRetentionPeriod = " + dataRetentionPeriod + ", status = " + status);
		MaintenancePerf perf = new MaintenancePerf();

		int ret = 0;

		try{
			ret = perf.delete(dataRetentionPeriod, status);
		}catch(NamingException e){
			m_log.warn("deletePerfData() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(CreateException e){
			m_log.warn("deletePerfData() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(SQLException e){
			m_log.warn("deletePerfData() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(RemoveException e){
			m_log.warn("deletePerfData() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(HinemosUnknown e){
			m_log.warn("deletePerfData() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		}catch(NotifyNotFound e){
			m_log.warn("deletePerfData() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (FinderException e) {
			m_log.warn("deletePerfData() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new MaintenanceNotFound(e.getMessage(), e);
		}

		return ret;
	}

	/**
	 * 
	 * 一括制御履歴を削除します。
	 * 
	 * @param dataRetentionPeriod 保存期間(日)
	 * @param status 削除対象のステータス(true=全履歴、false=実行状態が「終了」の履歴)
	 * @return 削除件数
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws MaintenanceNotFound
	 * 
	 * @ejb.interface-method
	 * 
	 * 時間を要する処理のため、NotSupportedを採用してJTAの管理下から除外する
	 * @ejb.transaction
	 *     type="NotSupported"
	 */
	public int deleteCollectiveRunHistory(int dataRetentionPeriod, boolean status) throws HinemosUnknown, NotifyNotFound, MaintenanceNotFound {
		m_log.debug("deleteCollectiveRunHistory() : dataRetentionPeriod = " + dataRetentionPeriod + ", status = " + status);
		MaintenanceCrun crun = new MaintenanceCrun();

		int ret = 0;

		try{
			ret = crun.delete(dataRetentionPeriod, status);
		}catch(NamingException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(CreateException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(SQLException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(RemoveException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(HinemosUnknown e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		}catch(NotifyNotFound e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (FinderException e) {
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new MaintenanceNotFound(e.getMessage(), e);
		}

		return ret;
	}

	/**
	 * 
	 * 仮想化操作ログを削除します。
	 * 
	 * @param dataRetentionPeriod 保存期間(日)
	 * @param status 削除対象のステータス(true=全履歴、false=実行状態が「終了」の履歴)
	 * @return 削除件数
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws MaintenanceNotFound
	 * 
	 * @ejb.interface-method
	 * 
	 * 時間を要する処理のため、NotSupportedを採用してJTAの管理下から除外する
	 * @ejb.transaction
	 *     type="NotSupported"
	 */
	public int deleteVmLog(int dataRetentionPeriod, boolean status) throws HinemosUnknown, NotifyNotFound, MaintenanceNotFound {
		m_log.debug("deleteVmLog() : dataRetentionPeriod = " + dataRetentionPeriod + ", status = " + status);
		MaintenanceVM vm = new MaintenanceVM();

		int ret = 0;

		try{
			ret = vm.delete(dataRetentionPeriod, status);
		}catch(NamingException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(CreateException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(SQLException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(RemoveException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}catch(HinemosUnknown e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		}catch(NotifyNotFound e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (FinderException e) {
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new MaintenanceNotFound(e.getMessage(), e);
		}

		return ret;
	}


	/**
	 * 
	 * メンテナンス機能をスケジュール実行します。<BR>
	 * Quartzからスケジュール実行時に呼び出されます。
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @ejb.interface-method
	 * 
	 * トランザクション開始はユーザが制御する。
	 * また、追加実装により、トランザクションの入れ子が予期せず生じることを避けるため、Neverを採用する。
	 * @ejb.transaction type="Never"
	 */
	public void scheduleRunMaintenance(String maintenanceId, String calendarId) throws HinemosUnknown, NotifyNotFound {
		m_log.debug("scheduleRunMaintenance() : maintenanceId=" + maintenanceId + ", calendarId=" + calendarId);

		//カレンダをチェック
		try {
			boolean check = false;
			if(calendarId != null && calendarId.length() > 0){
				CalendarControllerLocal calendar =
					CalendarControllerUtil.getLocalHome().create();

				//カレンダによる実行可/不可のチェック
				if(calendar.isRun(calendarId, new Date().getTime()).booleanValue()){
					check = true;
				}
			}
			else{
				check = true;
			}

			if(!check)
				return;

			//メンテナンス実行
			runMaintenance(maintenanceId);
		} catch (CreateException e) {
			m_log.error("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(),e);
		} catch (NamingException e) {
			m_log.error("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(),e);
		} catch (CalendarNotFound e) {
			m_log.error("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(),e);
		} catch (HinemosUnknown e) {
			m_log.error("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (NotifyNotFound e) {
			m_log.error("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		}
	}

	/**
	 * 
	 * メンテナンス機能を実行するメソッドです。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="Supports"
	 * 
	 * @param maintenanceId
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 */
	public void runMaintenance(String maintenanceId) throws HinemosUnknown, NotifyNotFound {
		m_log.debug("runMaintenance() : maintenanceId=" + maintenanceId);

		OperationMaintenance operation = new OperationMaintenance();
		try {
			operation.runMaintenance(maintenanceId);
		} catch (HinemosUnknown e) {
			m_log.warn("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (NotifyNotFound e) {
			m_log.warn("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
	}

	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean がインスタンスプールに格納される際に行う処理を実装します。
	 * 
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
}
