/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.factory;

import java.sql.Timestamp;
import java.util.Date;
import java.util.List;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.jms.DeliveryMode;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoDetail;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.bean.RenotifyTypeConstant;
import com.clustercontrol.notify.ejb.entity.MonitorStatusPK;
import com.clustercontrol.notify.ejb.entity.NotifyHistoryLocal;
import com.clustercontrol.notify.ejb.entity.NotifyHistoryPK;
import com.clustercontrol.notify.ejb.entity.NotifyHistoryUtil;
import com.clustercontrol.notify.queue.NotifyQueueException;
import com.clustercontrol.notify.queue.NotifyQueueSender;
import com.clustercontrol.notify.util.MonitorResultStatusUpdater;
import com.clustercontrol.notify.util.NotifyCache;

/**
 * 通知を通知ID毎のキューに振り分ける処理を行うユーティリティクラス
 */
public class NotifyDispatcher {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( NotifyDispatcher.class );

	public static final String MODE_VERBOSE = "verbose";
	public static final String MODE_NORMAL = "normal";

	private static final String NOTIFY_MODE_KEY = "common.notify.mode";

	private static String _mode = "normal";

	static {
		_mode = HinemosProperties.getProperty(NOTIFY_MODE_KEY, MODE_NORMAL);
		m_log.info("notify mode : " + _mode);
	}

	/**
	 * 引数で指定された情報を各種通知に出力します。<BR>
	 * イベント通知、ステータス通知、メール通知、ジョブ通知、ログエスカレーション通知を行います。
	 * 
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyEventBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyStatusBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyMailBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyJobBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyLogEscalationBean#onMessage(javax.jms.Message)
	 * 
	 * @param info 通知出力情報
	 * @param notifyIdList 通知対象の通知ID
	 * @param queueDeliveryMode JMSの永続化モード指定
	 */
	@SuppressWarnings("unchecked")
	public static void notifyAction(OutputBasicInfo info, List<String> notifyIdList, int queueDeliveryMode) {
		if(info == null || notifyIdList == null) {
			m_log.error("notifyAction() invalid argument. info = " + info
					+ ", notifyIdList = " + notifyIdList
					+ ", queueDeliveryMode = " + queueDeliveryMode);
			return;
		}

		// 監視結果重要度ステータスを更新
		boolean prioityChangeFlag = MonitorResultStatusUpdater.update(info);

		Timestamp outputDate = new Timestamp(System.currentTimeMillis());

		// 通知対象のリスト分の通知種別を振り分ける。
		if(notifyIdList != null && notifyIdList.size() > 0) {
			for (String notifyId : notifyIdList) {
				NotifyInfo notifyInfo = NotifyCache.getNotifyInfo(notifyId);
				Integer notifyType = notifyInfo.getNotifyType();
				Integer notifyValid = notifyInfo.getValidFlg();
				if (notifyType == null || notifyValid == null) {
					// 該当の通知IDの設定が見つからない場合はエラーログを出力し次の通知IDの処理を継続する
					m_log.warn("notifyAction() : notifyId = " + notifyId +" not found.");
					continue;
				}
				try {
					// 通知情報が「通知する」となっている場合
					if (notifyValid == ValidConstant.TYPE_VALID) {
						// デフォルトでは抑制せずに通知する（予期せぬエラーの場合は通知を行う）
						boolean isNotify = notifyCheck(
								info.getFacilityId(),
								info.getPluginId(),
								info.getMonitorId(),
								info.getSubKey(),
								notifyId,
								info.getPriority(),
								outputDate,
								prioityChangeFlag,
								_mode);

						if(isNotify){
							// 通知IDは設定されていないため、ここで設定する
							NotifyQueueSender.getInstance(notifyType).send(
									new NotifyRequestMessage(
											info,
											notifyId,
											outputDate,
											prioityChangeFlag
									), queueDeliveryMode);

							if (m_log.isDebugEnabled()) {
								m_log.debug("sending message to jms. ("
										+ (queueDeliveryMode == DeliveryMode.PERSISTENT ? "persistent" : "non-persistent")
										+ " : notifyId=" + notifyId
										+ ", pluginId=" + info.getPluginId()
										+ ", priority=" + info.getPriority()
										+ ", generationDate=" + info.getGenerationDate()
										+ ", monitorId=" + info.getMonitorId()
										+ ", facilityId=" + info.getFacilityId());
							}
						}
					} else {
						// TODO:通知しない場合は、キャッシュ情報をアップデートしたほうが良い
					}
				} catch (NotifyQueueException e) {
					m_log.warn("notifyAction() : " + info + " : " + e.getMessage(), e);
				} catch (Exception e) {
					m_log.warn("notifyAction() : " + info + " : " + e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * 抑制条件を確認し現時点で通知すべきか確認する。
	 * 
	 * @param facilityId ファシリティID
	 * @param pluginId プラグインID
	 * @param monitorId 監視ID
	 * @param subkey 通知抑制用サブキー
	 * @param notifyId 通知ID
	 * @param priority 重要度
	 * @param outputDate 出力日時
	 * @param priorityChangeFlag 重要度変更フラグ
	 * @param mode 抑制モード（VERBOSE:重要度が変化した場合は出力する, NORMAL:重要度が変化しても前回通知の重要度と同じ場合は通知しない）
	 * @return 通知する場合は true
	 */
	private static boolean notifyCheck(String facilityId, String pluginId, String monitorId, String subkey, String notifyId, int priority, Date outputDate, boolean priorityChangeFlag, String mode){
		if(m_log.isDebugEnabled()){
			m_log.debug("notifyCheck() " +
					"facilityId=" + facilityId +
					", pluginId=" +  pluginId +
					", monitorId=" + monitorId +
					", subkey=" + subkey +
					", notifyId=" + notifyId +
					", priority=" + priority +
					", outputDate=" + outputDate +
					", priorityChangeFlag=" + priorityChangeFlag +
					", mode="+ mode);
		}

		MonitorStatusPK mspk = new MonitorStatusPK(facilityId, pluginId, monitorId, subkey);
		MonitorStatusPK lockObject = MonitorResultStatusUpdater.getLock(mspk);

		// チェック処理を（ファシリティID, プラグインID, 監視項目ID, サブキー）の粒度で排他
		synchronized (lockObject) {
			NotifyHistoryPK pk = new NotifyHistoryPK(facilityId, pluginId, monitorId, subkey, notifyId);
			try {
				if(MODE_VERBOSE.equals(mode)){
					// 重要度変化があった場合
					if(priorityChangeFlag == true){
						m_log.debug("priorityChangeFlag == true. remove entity. " + pk);

						// 通知履歴の該当タプルを削除する
						try {
							NotifyHistoryUtil.getLocalHome().findByPrimaryKey(pk).remove();
						} catch (FinderException e) {
							m_log.debug(e.getMessage() + " NotifyHistory=" + pk);
							// 何もしない
						} catch (RemoveException e) {
							m_log.warn(e.getMessage() + " NotifyHistory=" + pk);
							// 何もしない
						}
					}
				}

				// 重要度単位の通知フラグを確認する
				NotifyInfo notifyInfo = NotifyCache.getNotifyInfo(notifyId);
				NotifyInfoDetail notifyDetail = NotifyCache.getNotifyInfoDetail(notifyId, priority);

				// 重要度単位の有効無効を確認
				if(notifyDetail.getValidFlg().intValue() == ValidConstant.TYPE_INVALID){
					// 無効の場合は通知しない
					m_log.debug("ValidFlg is invalid. " + pk + ", priority = " + priority);
					m_log.debug("notify NG. (VALIDFLAG IS INVALID)." + pk);
					return false;
				}

				// 通知条件である同一重要度カウンタ数を満たしているか確認
				// 同一重要度カウンタを保持する監視結果ステータス情報を検索
				Long counter = null;
				try {
					MonitorResultStatusUpdater.refreshCache(mspk);
					counter = MonitorResultStatusUpdater.getCounter(mspk);
				} catch (FinderException e) {
					// ジョブからの通知はここを通る。
					m_log.debug("notify OK. (MONITOR STATUS NOT FOUND)." + pk);
					return true;
				}

				if(counter >= notifyInfo.getInitialCount()){
					// カウンタが条件を満たしている場合
					m_log.debug("counter check. " + counter + " >= " + notifyInfo.getInitialCount() + "  " + pk);

					try{
						NotifyHistoryLocal history = null;
						if(!MODE_VERBOSE.equals(mode)){
							// 通知履歴の該当タプルを検索
							history = NotifyHistoryUtil.getLocalHome().findByPrimaryKey(pk);
							// 現在の監視結果の重要度と最終通知時の重要度が異なる場合は通知する
							if (priority != history.getPriority()) {
								m_log.debug("update notify history." + pk);
								history.setLastNotify(new Timestamp(outputDate.getTime()));
								history.setPriority(priority);
								m_log.debug("notify OK. (PRIORITY CHANGE)." + pk);
								return true;
							}
						}

						// 該当のタプルが存在する場合
						// 再通知種別を確認
						if(notifyInfo.getRenotifyType() == RenotifyTypeConstant.TYPE_NO_NOTIFY){
							// 再通知なしの場合
							m_log.debug("notify NG. (RENOTIFY NO)." + pk);
							return false;
						} else if(notifyInfo.getRenotifyType() == RenotifyTypeConstant.TYPE_ALWAYS_NOTIFY){
							// 常に再通知の場合
							// history.setLastNotify(new Timestamp(outputDate.getTime())); 常に通知するため更新の必要がない
							// history.setPriority(priority); 常に通知するため更新の必要がない
							m_log.debug("notify OK. (RENOTIFY ALWAYS)." + pk);
							return true;
						} else {
							if (history == null) {
								// 通知履歴の該当タプルを検索
								history = NotifyHistoryUtil.getLocalHome().findByPrimaryKey(pk);
							}
							if(outputDate != null && outputDate.getTime() >=
								(history.getLastNotify().getTime() + (notifyInfo.getRenotifyPeriod() * 60 * 1000l))){
								m_log.debug("update notify history." + pk);
								// 通知時刻が抑制期間を超えている場合
								history.setLastNotify(new Timestamp(outputDate.getTime()));
								history.setPriority(priority);
								m_log.debug("notify OK. (RENOTIFY PERIOD)." + pk);
								return true;
							} else {
								m_log.debug("notify NG. (RENOTIFY PERIOD)." + pk);
								return false;
							}
						}
					} catch (FinderException e) {
						// 該当のタプルが存在しない場合
						// 初回通知
						m_log.debug("first notify. " + pk + ", priority = " + priority);

						// 新規に通知履歴を作成する
						NotifyHistoryUtil.getLocalHome().create(
								facilityId,
								pluginId,
								monitorId,
								subkey,
								notifyId,
								priority,
								outputDate==null?null:new Timestamp(outputDate.getTime()));

						m_log.debug("notify OK. (NEW)." + pk);
						return true;
					}
				} else {
					m_log.debug("notify NG. (PRIORITY CHANGE)." + pk);
					return false;
				}

			} catch (EJBException e) {
				m_log.error(e.getMessage(), e);
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
			} catch (CreateException e) {
				m_log.error(e.getMessage(), e);
			}

			m_log.error("notify OK. (cause unexpected errors)" + pk);
			return true;
		}
	}
}
