/*

Copyright (C) 2009 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.rmi.RemoteException;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
//
import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.ejb.entity.NotifyCommandInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyCommandInfoPK;
import com.clustercontrol.notify.ejb.entity.NotifyCommandInfoUtil;
import com.clustercontrol.util.CommandCreator;
import com.clustercontrol.util.CommandExecutor;
import com.clustercontrol.util.StringBinder;
import com.clustercontrol.util.CommandExecutor.CommandResult;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * コマンドを実行するクラス<BR>
 *
 * @version 3.0.0
 * @since 3.0.0
 */
public class ExecCommand implements Notifier {
	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( ExecCommand.class );

	private static final String THREAD_POOL_COUNT_KEY="common.notify.command.thread.pool.count";
	private static final String THREAD_POOL_COUNT_DEFAULT="10";
	private static final int THREAD_POOL_COUNT;

	private static final String COMMAND_CREATE_MODE="common.notify.command.create.mode";
	private static final String COMMAND_SUCCESS_EXIT="common.notify.command.success.exit";
	private static final String COMMAND_SUCCESS_EXIT_DEFAULT="0";

	// コマンド実行にタイムアウトを設けるため、2段のExecutorの構成とする
	// 1.「コマンドを実行するスレッド（CommandTask）」を実行するExecutor
	//   （スレッド起動後直ぐに制御を戻す）
	// 2.コマンドを実行するスレッド（CommandTask）用のExecutor
	//   （スレッド起動後、コマンドの実行終了もしくは、タイムアウトを待つ）

	// 「コマンドを実行するスレッド（CommandTask）」を実行し、その終了を待つスレッド（CommandCallerTask）用の
	// スレッドプールを保持するExecutorService
	private static final ExecutorService _callerExecutorService;

	// コマンド作成モード（OS毎に異なる）
	private static final String _mode;
	private static final CommandCreator.PlatformType _modeType;

	private static final int _successExitCode;

	static {
		// hinemos.propertiesからスレッドプール数を取得する
		THREAD_POOL_COUNT = Integer.parseInt(HinemosProperties.getProperty(
				THREAD_POOL_COUNT_KEY, THREAD_POOL_COUNT_DEFAULT));

		// コマンド作成モードをプロパティから取得する
		_mode = HinemosProperties.getProperty(COMMAND_CREATE_MODE, "");
		_modeType = CommandCreator.convertPlatform(_mode);

		// コマンドの成功時の終了値をプロパティから取得する
		_successExitCode = Integer.parseInt(HinemosProperties.getProperty(COMMAND_SUCCESS_EXIT, COMMAND_SUCCESS_EXIT_DEFAULT));

		// 「コマンドを実行するスレッド（CommandTask）」を実行し、その終了を待つスレッド（CommandCallerTask）用の
		// スレッドプールを保持するExecutorService
		_callerExecutorService = Executors.newFixedThreadPool(THREAD_POOL_COUNT,
				new CommandTaskThreadFactory());
	}

	/**
	 * 指定されたコマンドを呼出します。
	 * 
	 * @param outputInfo 通知・抑制情報
	 * @throws RemoteException
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FinderException
	 */
	@Override
	public synchronized void notify(NotifyRequestMessage message)
	throws RemoteException, NamingException, CreateException, FinderException {
		if(m_log.isDebugEnabled()){
			m_log.debug("notify() " + message);
		}

		executeCommand(message.getOutputInfo(), message.getNotifyId());
	}

	// 文字列を置換する
	private String getCommandString(OutputBasicInfo outputInfo, String beforeCommand){
		// 文字列を置換する
		try {
			Map<String, String> param = NotifyUtil.createParameter(outputInfo);
			StringBinder binder = new StringBinder(param);
			return binder.bindParam(beforeCommand);
		} catch (Exception e) {
			m_log.error(e.getMessage(), e);

			// 例外が発生した場合は、置換前の文字列を返す
			return beforeCommand;
		}
	}

	/**
	 * 指定されたコマンドを呼出します。
	 * 
	 * @param outputInfo 通知・抑制情報
	 * @throws RemoteException
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FinderException
	 */
	private synchronized void executeCommand(
			OutputBasicInfo outputInfo,
			String notifyId
	) {
		if(m_log.isDebugEnabled()){
			m_log.debug("notify() " + outputInfo);
		}

		NotifyCommandInfoLocal commandInfo;
		try {
			commandInfo = NotifyCommandInfoUtil.getLocalHome().findByPrimaryKey(
					new NotifyCommandInfoPK(notifyId, outputInfo.getPriority()));


			// 「commandInfo.getCommand()」と「command」の違いに注意が必要。
			// 「commandInfo.getCommand()」は、設定時の実行コマンドで、
			// TextReplacerによる文字列置換前の実行コマンド
			// 「command」は、実際に実行対象のコマンド文字列
			String command = getCommandString(outputInfo, commandInfo.getCommand());

			/**
			 * 実行
			 */
			// 起動ユーザ名取得
			String sysUserName = System.getProperty("user.name");
			String effectiveUser = commandInfo.getEffectiveUser();
			boolean setEnvFlag;
			if(commandInfo.getSetEnvironment() == ValidConstant.TYPE_VALID){
				setEnvFlag = true;
			} else {
				setEnvFlag = false;
			}
			long commadTimeout = commandInfo.getTimeout();

			// JBossの起動ユーザがroot以外の場合で、
			// 起動ユーザとコマンド実行ユーザが異なる場合は、コマンド実行しない
			if ((!sysUserName.equals("root")) && (!sysUserName.equals(effectiveUser))) {
				// 起動失敗
				String detailMsg = "The execution user of the command and jboss's user are different.";
				m_log.error(detailMsg);
				internalErrorNotify(notifyId, "007", detailMsg);
				return;
			} else {
				m_log.debug("NotifyCommand Submit : " + outputInfo + " command=" + command);

				// 並列でコマンド実行
				_callerExecutorService.submit(
						new CommandCallerTask(
								effectiveUser,
								command,
								setEnvFlag,
								notifyId,
								outputInfo,
								commadTimeout));
			}
		} catch (FinderException e) {
			String detailMsg = e.getMessage();
			m_log.error(detailMsg, e);
			internalErrorNotify(notifyId, "007", detailMsg);
		} catch (NamingException e) {
			String detailMsg = e.getMessage();
			m_log.error(detailMsg, e);
			internalErrorNotify(notifyId, "007", detailMsg);
		}
	}

	// 並列でコマンドを実行するためのクラス
	class CommandCallerTask implements Callable<Long> {
		// 実効ユーザ
		private final String _effectiveUser;

		// 実行するコマンド
		private final String _execCommand;

		// ユーザを切り替える際にユーザで設定されているログイン時のスタートアップファイルを読ませるか否かのフラグ
		// 具体的には、suコマンド実行時に引数に「-」を加えるか否か
		private final boolean _setEnvFlag;

		private final String _notifyId;

		private final OutputBasicInfo _outputInfo;

		private final long _commadTimeout;

		public CommandCallerTask(
				String effectiveUser,
				String execCommand,
				boolean setEnvFlag,
				String notifyId,
				OutputBasicInfo outputInfo,
				long commadTimeout) {
			_effectiveUser = effectiveUser;
			_execCommand = execCommand;
			_setEnvFlag = setEnvFlag;
			_notifyId = notifyId;
			_outputInfo = outputInfo;
			_commadTimeout = commadTimeout;
		}

		/**
		 * CommandTaskを実行しその終了（もしくはタイムアウト）まで待つ処理を実行します
		 */
		@Override
		public Long call() throws Exception {
			// 初期値（コマンドが時間内に終了せずリターンコードが取得できない場合は、この値が返る）
			long returnValue = Long.MIN_VALUE;

			// コマンドを実行する
			String[] cmd = CommandCreator.createCommand(_effectiveUser, _execCommand, _modeType);

			m_log.info("excuting command. (effectiveUser = " + _effectiveUser + ", command = " + _execCommand + ", mode = " + _modeType + ", timeout = " + _commadTimeout + ")");

			// 戻り値を格納する
			CommandResult ret = new CommandExecutor(cmd, _commadTimeout).execute();

			if (ret != null) {
				m_log.info("executed command. (exitCode = " + ret.exitCode + ", stdout = " + ret.stdout + ", stderr = " + ret.stderr + ")");
			}

			if (ret == null || ret.exitCode == null) {
				internalErrorNotify(_notifyId, "007", "command execution failure (timeout). [command = " + _execCommand + "]");
			} else {
				returnValue = ret.exitCode;
				if (returnValue != _successExitCode) {
					internalErrorNotify(_notifyId, "007", "command execution failure. [command = "
							+ _execCommand + ", exit code = " +  returnValue + ", stdout = " + ret.stdout + ", stderr = " + ret.stderr + "]");
				}
			}

			return returnValue;
		}
	}

	// CommandCallerTaskのスレッドに名前を定義するためFactoryを実装
	private static class CommandCallerTaskThreadFactory implements ThreadFactory {
		private volatile int _count = 0;

		@Override
		public Thread newThread(Runnable r) {
			return new Thread(r, "NotifyCommandCallerTask-" + _count++);
		}
	}

	// CommandTaskのスレッドに名前を定義するためFactoryを実装
	private static class CommandTaskThreadFactory implements ThreadFactory {
		private volatile int _count = 0;

		@Override
		public Thread newThread(Runnable r) {
			return new Thread(r, "NotifyCommandTask-" + _count++);
		}
	}

	/**
	 * 通知失敗時の内部エラー通知を定義します
	 */
	@Override
	public void internalErrorNotify(String notifyId, String msgID, String detailMsg) {
		AplLogger apllog = new AplLogger("NOTIFY", "notify");
		String[] args = { notifyId };
		// 通知失敗メッセージを出力
		apllog.put("SYS", msgID, args, detailMsg);
	}
}
