/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.notify.ejb.entity.NotifyRelationInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyRelationInfoUtil;

/**
 * 通知グループIDと通知IDとの関連をマップで管理するクラス。
 * ジョブセッションで利用されている通知グループはキャッシュされない。
 */
public class NotifyRelationCache {
	private static Log m_log = LogFactory.getLog( NotifyRelationCache.class );

	private static ConcurrentHashMap<String, List<String>> m_notifyIdMap = new ConcurrentHashMap<String, List<String>>();

	static {
		refresh();
	}

	/**
	 * 通知グループIDが関連を持つ通知IDのリストを返す。
	 * 
	 * @param notifyGroupId 通知グループID
	 * @return 通知IDのリスト。エラー時は空のリストを返す。
	 */
	public static List<String> getNotifyIdList(String notifyGroupId){
		try {
			// キャッシュから取得
			List<String> notifyIdList = m_notifyIdMap.get(notifyGroupId);
			if(notifyIdList != null){
				return notifyIdList;
			}
			// ジョブセッションで利用されている通知グループの以外はキャッシュされているはず。
			if(notifyGroupId.startsWith(HinemosModuleConstant.JOB_SESSION + "-") == false){
				return m_notifyIdMap.get(notifyGroupId);
			}
			m_log.info("NotifyRelationCache Job Session. " + notifyGroupId);
			Collection<NotifyRelationInfoLocal> nriList = NotifyRelationInfoUtil.getLocalHome().findByNotifyGroupId(notifyGroupId);
			notifyIdList = new ArrayList<String>();
			for(NotifyRelationInfoLocal nri : nriList){
				notifyIdList.add(nri.getNotifyId());
			}
			return notifyIdList;
		} catch (FinderException e) {
			m_log.error("notifyGroupId not found. notifyGroupId = " + notifyGroupId, e);
			return new ArrayList<String>(); // 空のリストを返す
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
			return new ArrayList<String>(); // 空のリストを返す
		}
	}

	public static void refresh(){
		m_log.info("clear NotifyRelationCache");

		ConcurrentHashMap<String, List<String>> notifyIdMap = new ConcurrentHashMap<String, List<String>>();
		Collection<NotifyRelationInfoLocal> nriList = null;
		try {
			nriList = NotifyRelationInfoUtil.getLocalHome().findAll();
		} catch (Exception e) {
			m_log.error("refresh() : " + e.getMessage(), e);
			return;
		}
		for (NotifyRelationInfoLocal nri : nriList) {
			String notifyGroupId = nri.getNotifyGroupId();
			// ジョブセッションで利用されている通知グループの場合は、キャッシュしない。
			if(notifyGroupId.startsWith(HinemosModuleConstant.JOB_SESSION + "_") == false){
				List<String> notifyIdList = notifyIdMap.get(notifyGroupId);
				if (notifyIdList == null) {
					notifyIdList = new ArrayList<String>();
					notifyIdList.add(nri.getNotifyId());
					notifyIdMap.put(notifyGroupId, notifyIdList);
				} else {
					notifyIdList.add(nri.getNotifyId());
				}
			}
		}
		m_notifyIdMap = notifyIdMap;
	}
}
