/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 */

package com.clustercontrol.performance.util.code;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.performance.bean.CollectorItemInfo;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryCollectMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryCollectMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCalcMethodMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCalcMethodMstUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstUtil;
import com.clustercontrol.performance.util.CollectorMasterCache;
import com.clustercontrol.performance.util.PollingDataManager;
import com.clustercontrol.repository.bean.NodeDeviceInfo;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.repository.factory.FacilitySelector;
import com.clustercontrol.repository.factory.NodeProperty;

/**
 * 収集項目コードの情報を生成するファクトリクラス
 * 
 * @version 4.0
 * @since 1.0
 */
public class CollectorItemCodeTable {
	private static Log m_log = LogFactory.getLog( CollectorItemCodeTable.class );
	private static final HashMap<String, CollectorItemTreeItem> m_codeTable;

	/**
	 * static field
	 */
	static {
		m_codeTable = new HashMap<String, CollectorItemTreeItem>(); // 収集項目コードがキー

		// カテゴリコードからカテゴリ情報を参照するためのテーブル
		HashMap<String, CollectorItemTreeItem> categoryTable =
			new HashMap<String, CollectorItemTreeItem>();

		// カテゴリ情報の読み込み
		Collection<CollectorCategoryMstLocal> cate = null;
		try {
			cate = CollectorCategoryMstUtil.getLocalHome().findAll();
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}

		Iterator<CollectorCategoryMstLocal> cateItr = cate.iterator();
		while(cateItr.hasNext()){
			CollectorCategoryMstLocal bean = cateItr.next();

			CollectorCategoryMstData category  =
				new CollectorCategoryMstData(bean.getCategoryCode(), bean.getCategoryName());

			CollectorItemTreeItem categoryItem =
				new CollectorItemTreeItem(null, null, category, null, null);  // 親の要素はないためnull

			// カテゴリコードにマッピングするようにカテゴリ情報を登録
			categoryTable.put(bean.getCategoryCode(), categoryItem);
		}



		try {
			// 収集項目コードの読み込み
			Collection<CollectorItemCodeMstLocal> ctItemCodeMst = CollectorItemCodeMstUtil.getLocalHome().findAll();

			Iterator<CollectorItemCodeMstLocal> itrItemCode = ctItemCodeMst.iterator();

			while(itrItemCode.hasNext()) {
				CollectorItemCodeMstLocal codeBean = itrItemCode.next();
				String itemCode = codeBean.getItemCode();

				if(itemCode == null){
					continue;
				}

				if(itemCode != null && codeBean.getCategoryCode() != null
						&& codeBean.getDeviceSupport() != null && codeBean.getGraphRange() != null){

					// 収集項目コードマスタ情報のDTOを作成
					CollectorItemCodeMstData codeData = new CollectorItemCodeMstData(
							codeBean.getItemCode(),
							codeBean.getCategoryCode(),
							codeBean.getParentItemCode(),
							codeBean.getItemName(),
							codeBean.getMeasure(),
							codeBean.getDeviceSupport(),
							codeBean.getDeviceType(),
							codeBean.getGraphRange()
					);

					// カテゴリ名を調べます
					CollectorItemTreeItem categoryTreeItem = categoryTable.get(codeBean.getCategoryCode());

					// 親のオブジェクトを取得（存在しない場合はnull）
					CollectorItemTreeItem parentItem =
						m_codeTable.get(codeBean.getParentItemCode());

					// 親のコードが存在しない場合はカテゴリの直下の要素とする
					if(parentItem == null){
						parentItem = categoryTreeItem;
					}

					CollectorItemTreeItem ctItem =
						new CollectorItemTreeItem(parentItem, codeData, null, null, null);

					// 収集項目コードをキーとして登録
					m_codeTable.put(itemCode, ctItem);
				}
			}

		} catch (Exception e) {
			m_codeTable.clear();
			// エラー処理
			m_log.warn("CollectorItemCodeTable static field error . ", e);
		}

	}

	/**
	 * 収集項目IDを探索するためのテンポラリ用の内部クラス
	 * 
	 * @author hinemos
	 */
	private static class PlatformIdAndSubPlatformId {
		private String m_platformId;
		private String m_subPlatformId;

		public PlatformIdAndSubPlatformId(String platformId, String subPlatformId){
			m_platformId = platformId;
			m_subPlatformId = subPlatformId;
		}

		public String getPlatformId(){
			return m_platformId;
		}

		public String getSubPlatformId(){
			return m_subPlatformId;
		}

		@Override
		public boolean equals(Object other) {
			if (other instanceof PlatformIdAndSubPlatformId) {
				PlatformIdAndSubPlatformId info = (PlatformIdAndSubPlatformId)other;

				if (this.m_platformId == null && this.m_subPlatformId == null){
					if (info.m_platformId == null && info.m_subPlatformId == null){
						return true;
					}
				} else if (this.m_platformId == null && this.m_subPlatformId != null){
					if (info.m_platformId == null && this.m_subPlatformId.equals(info.m_subPlatformId)){
						return true;
					}
				} else if (this.m_platformId != null && this.m_subPlatformId == null){
					if (this.m_platformId.equals(info.m_platformId) && info.m_subPlatformId == null){
						return true;
					}
				} else {
					if (this.m_platformId.equals(info.m_platformId)){
						return this.m_subPlatformId.equals(info.m_subPlatformId);
					}
				}
				return false;
			} else {
				return false;
			}
		}

		@Override
		public int hashCode() {
			int result = 17;

			result = 37 * result + ((this.m_platformId != null) ? this.m_platformId.hashCode() : 0);

			result = 37 * result + ((this.m_subPlatformId != null) ? this.m_subPlatformId.hashCode() : 0);

			return result;
		}
	}


	/**
	 * 指定したfacilityId配下の全てのノードで選択可能な収集項目IDの集合を返却する
	 * 
	 * @param facilityId
	 * @return
	 */
	private static Set<CollectorItemCodeMstData> getEnableCodeSet(String facilityId){
		m_log.debug("getEnableCodeSet() facilityId = " + facilityId);

		// 選択可能な収集項目コード情報のセット
		Set<CollectorItemCodeMstData> codeSet = null;

		// 対象ファシリティIDが空またはNULLなら
		if(facilityId == null || "".equals(facilityId)){
			m_log.debug("getEnableCodeSet() codeSet is 0");
			return codeSet;
		}

		// 全てのノードのプラットフォームIDのセット
		Set<PlatformIdAndSubPlatformId> platformSet = new HashSet<PlatformIdAndSubPlatformId>();
		try {
			// リポジトリ管理機能のSessionBean
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();

			////
			// 探査対象となるノードリストを作成
			////
			List<String> nodeList = null;
			if(bean.isNode(facilityId)){	// 対象ファシリティがノードの場合
				m_log.debug("getEnableCodeSet() facilityId is node");

				// 自分自身のみを登録
				nodeList = new ArrayList<String>();
				nodeList.add(facilityId);
			} else {	// 対象ファシリティがスコープの場合
				m_log.debug("getEnableCodeSet() facilityId is scope");

				// スコープにふくまれるノードのIDのリストを取得
				nodeList = bean.getNodeFacilityIdList(facilityId, RepositoryControllerBean.ALL);
			}

			////
			// 存在する全てのプラットフォーム情報を取得
			////
			for(String nodeId : nodeList){
				m_log.debug("getEnableCodeSet() target node  = " + nodeId);
				NodeInfo node = bean.getNode(nodeId);

				platformSet.add(new PlatformIdAndSubPlatformId(node.getPlatformFamily(), node.getVirtualizationSolution()));
			}

			////
			// プラットフォームで共通の収集項目IDを取得する
			////
			for(PlatformIdAndSubPlatformId platform : platformSet){

				m_log.debug("getEnableCodeSet() " +
						"platformId = " + platform.getPlatformId() +
						", subPlatformId = " + platform.getSubPlatformId());

				// プラットフォーム、サブプラットフォーム単位のカテゴリコード、収集方法を取得
				Collection<CollectorCategoryCollectMstLocal> collects;
				// プラットフォーム、サブプラットフォーム単位の収集方法、アイテムコード、算出方法を取得
				Collection<CollectorItemCalcMethodMstLocal> calsMethods;
				try {
					collects = CollectorCategoryCollectMstUtil.getLocalHome().findByPlatformIdAndSubPlatformId(
							platform.getPlatformId(),
							platform.getSubPlatformId());

					calsMethods = CollectorItemCalcMethodMstUtil.getLocalHome().findByPlatformIdAndSubPlatformId(
							platform.getPlatformId(),
							platform.getSubPlatformId());
				} catch (FinderException e) {
					m_log.error("getEnableCodeSet()", e);
					return null;
				} catch (NamingException e) {
					m_log.error("getEnableCodeSet()", e);
					return null;
				}

				// categoryCodeをキーにcollectMethodを取得するマップ
				HashMap<String, String> categoryMap = new HashMap<String, String>();
				for(CollectorCategoryCollectMstLocal collect : collects){
					m_log.debug("getEnableCodeSet() add map categoryCode = " + collect.getCategoryCode() + ", collectMethod = " + collect.getCollectMethod());
					categoryMap.put(collect.getCategoryCode(), collect.getCollectMethod());
				}

				// プラットフォームごとに収集可能な収集項目コードのリストを求める
				HashSet<CollectorItemCodeMstData> tmpByPlatform = new HashSet<CollectorItemCodeMstData>();

				// 計算方法管理テーブルの要素ごとにループをまわす
				for(CollectorItemCalcMethodMstLocal calcBean : calsMethods){

					CollectorItemCodeMstData codeBean;
					try {
						m_log.debug("getEnableCodeSet() search itemCode = " + calcBean.getItemCode());
						codeBean = CollectorMasterCache.getCategoryCodeMst(calcBean.getItemCode());
					} catch (FinderException e) {
						m_log.error(e.getMessage(), e);
						return null;
					} catch (NamingException e) {
						m_log.error(e.getMessage(), e);
						return null;
					}

					// カテゴリ別収集情報の収集方法と収集項目マスタの収集方法を比較し、同じならば追加する
					if(categoryMap.get(codeBean.getCategoryCode()) != null &&
							categoryMap.get(codeBean.getCategoryCode()).equals(calcBean.getCollectMethod())) {
						// 収集項目コードマスタ情報のDTOを作成
						CollectorItemCodeMstData codeData = new CollectorItemCodeMstData(
								codeBean.getItemCode(),
								codeBean.getCategoryCode(),
								codeBean.getParentItemCode(),
								codeBean.getItemName(),
								codeBean.getMeasure(),
								codeBean.getDeviceSupport(),
								codeBean.getDeviceType(),
								codeBean.getGraphRange()
						);
						m_log.debug("getEnableCodeSet() add itemCode = " + calcBean.getItemCode());
						tmpByPlatform.add(codeData);
					}

				}
				// プラットフォームごとに収集可能な項目をANDもしくはOR条件でセットに格納する
				// 同じものだけ追加する(AND条件)
				if(codeSet == null){
					m_log.debug("getEnableCodeSet() codeSet is null");
					codeSet = tmpByPlatform;
					m_log.debug("getEnableCodeSet() codeSet size = " + codeSet.size());
				} else {
					m_log.debug("getEnableCodeSet() codeSet is not null");
					codeSet.retainAll(tmpByPlatform);
					m_log.debug("getEnableCodeSet() codeSet size = " + codeSet.size());
				}
			}

		} catch (FacilityNotFound e) {
			m_log.error("getEnableCodeSet() facility not found. (facilityId = " + facilityId + ")", e);
		} catch (CreateException e) {
			throw new EJBException(e);
		} catch (NamingException e) {
			throw new EJBException(e);
		}

		// 1つもない場合は、0サイズのSetを返却
		if(codeSet == null){
			codeSet = new HashSet<CollectorItemCodeMstData>();
		}
		return codeSet;
	}

	/**
	 * 指定したfacilityIdで収集可能な収集項目のリストを作成する
	 * 
	 * @param facilityId
	 * @return
	 */
	public static List<CollectorItemInfo> getAvailableCollectorItemList(String facilityId) {
		m_log.debug("getAvailableCollectorItemList() facilityId = " + facilityId);

		// null check
		if(facilityId == null || "".equals(facilityId)){
			return new ArrayList<CollectorItemInfo>();
		}

		// 全てのノードに含まれるデバイスのセット
		Set<NodeDeviceInfo> deviceSet = getDeviceSetContainsAllNodes(facilityId);
		m_log.debug("getAvailableCollectorItemList() facilityId = " + facilityId + ", deviceSet size = " + deviceSet.size());

		// 選択可能な収集項目コード情報のセット
		Set<CollectorItemCodeMstData> itemCodeSet = getEnableCodeSet(facilityId);
		m_log.debug("getAvailableCollectorItemList() facilityId = " + facilityId + ", itemCodeSet size = " + itemCodeSet.size());

		// 戻り値
		List<CollectorItemInfo> list = new ArrayList<CollectorItemInfo>();

		for(CollectorItemCodeMstData itemCode : itemCodeSet){
			m_log.debug("getAvailableCollectorItemList() facilityId = " + facilityId + ", itemCode = " + itemCode.getItemCode() + ", deviceSupport = " + itemCode.getDeviceSupport().intValue());

			CollectorItemInfo itemInfo = null;

			switch (itemCode.getDeviceSupport().intValue()) {
			//デバイスサポートあり
			case YesNoConstant.TYPE_YES:
				for(NodeDeviceInfo deviceInfo : deviceSet){
					if(itemCode.getDeviceType() != null && itemCode.getDeviceType().equals(deviceInfo.getDeviceType())){
						itemInfo = new CollectorItemInfo(null, itemCode.getItemCode(), deviceInfo.getDeviceDisplayName());//collectorId is null

						m_log.debug("getAvailableCollectorItemList() facilityId = " + facilityId + ", itemCode = " + itemCode.getItemCode() + ", deviceDisplayName = " + deviceInfo.getDeviceDisplayName());
						list.add(itemInfo);
					}
				}

				// ALL Deviceの追加
				m_log.debug("getAvailableCollectorItemList() facilityId = " + facilityId + ", itemCode = " + itemCode.getItemCode() + ", deviceDisplayName = " + PollingDataManager.ALL_DEVICE_NAME);
				itemInfo = new CollectorItemInfo(null, itemCode.getItemCode(), PollingDataManager.ALL_DEVICE_NAME);//collectorId is null
				list.add(itemInfo);

				break;

				//デバイスサポートなし
			case YesNoConstant.TYPE_NO:
				itemInfo = new CollectorItemInfo(null, itemCode.getItemCode(), "");//collectorId is null

				m_log.debug("getAvailableCollectorItemList() facilityId = " + facilityId + ", itemCode = " + itemCode.getItemCode());
				list.add(itemInfo);
				break;

			default:
				break;
			}
		}

		// Sort
		Collections.sort(list,new Comparator<CollectorItemInfo>() {
			@Override
			public int compare(CollectorItemInfo o1, CollectorItemInfo o2) {
				// TODO Auto-generated method stub
				return o1.getItemCode().compareTo(o2.getItemCode());
			}
		});

		m_log.debug("getAvailableCollectorItemList() facilityId = " + facilityId + ", list size = " + list.size());
		return list;
	}

	/**
	 * 指定したfacilityId配下の全てのノードに含まれるデバイスの集合を返却する。
	 * 
	 * @param facilityId
	 * @return
	 */
	private static Set<NodeDeviceInfo> getDeviceSetContainsAllNodes(String facilityId){
		m_log.debug("getDeviceSetContainsAllNodes() facilityId = " + facilityId);

		// 戻り値
		Set<NodeDeviceInfo> deviceSet = new HashSet<NodeDeviceInfo>();

		// 対象ファシリティIDが空またはNULLなら
		if(facilityId == null || "".equals(facilityId)){
			return deviceSet;
		}

		// 対象のファシリティIDリスト
		// targetFacilityIdListはノードのみ含まれる。
		List<String> targetFacilityIdList = FacilitySelector.getNodeFacilityIdList(facilityId, RepositoryControllerBean.ALL, false, true);
		List<NodeDeviceInfo> checkTargetList = null;
		for (String targetFacilityId : targetFacilityIdList){

			// (ノード)targetFacilityIdのデバイス一覧を取得
			List<NodeDeviceInfo> checkList = new ArrayList<NodeDeviceInfo>();
			try {
				NodeInfo node = NodeProperty.getProperty(targetFacilityId);
				checkList.addAll(node.getNodeDeviceInfo());
				checkList.addAll(node.getNodeCpuInfo());
				checkList.addAll(node.getNodeMemoryInfo());
				checkList.addAll(node.getNodeNetworkInterfaceInfo());
				checkList.addAll(node.getNodeDiskInfo());
				checkList.addAll(node.getNodeFilesystemInfo());
			} catch (FacilityNotFound e) {
				m_log.warn("FacilityNotFound " + e.getMessage());
			}

			// 1ノード目は全部セット
			if(checkTargetList == null){
				checkTargetList = checkList;
			}
			// 2ノード目は同じものだけを残す
			else{
				List<NodeDeviceInfo> tmpList = new ArrayList<NodeDeviceInfo>();

				for(NodeDeviceInfo checkInfo : checkList){
					for(NodeDeviceInfo targetInfo : checkTargetList){

						// デバイスタイプと表示名が同じものがあるか？
						if(checkInfo.getDeviceType().equals(targetInfo.getDeviceType()) &&
								checkInfo.getDeviceDisplayName().equals(targetInfo.getDeviceDisplayName())){
							tmpList.add(checkInfo);
							m_log.debug("getDeviceSetContainsAllNodes() facilityId = " + facilityId + ", " +
									"add device deviceType = " + checkInfo.getDeviceType() + ", deviceDisplayName = " + checkInfo.getDeviceDisplayName());
							break;
						}
					}
				}
				checkTargetList = tmpList;
			}
		}

		if(checkTargetList != null){
			deviceSet.addAll(checkTargetList);
		}

		return deviceSet;
	}

	
	/**
	 * 収集項目コードとリポジトリ表示名から収集項目表示用文字列を生成し返します。
	 * 形式 ：　収集名[リポジトリ表示名]
	 * 
	 * @param itemCode 収集項目コード
	 * @param displayName リポジトリ表示名
	 * @return 収集項目表示
	 */
	public static String getFullItemName(String itemCode, String displayName){
		m_log.debug("getFullItemName() itemCode = " + itemCode + ", displayName = " + displayName);

		try {
			CollectorItemCodeMstData bean = CollectorMasterCache.getCategoryCodeMst(itemCode);
			String itemName = bean.getItemName();
			if(bean.getDeviceSupport().intValue() == YesNoConstant.TYPE_YES){
				itemName = itemName + "[" + displayName + "]";
			}

			m_log.debug("getFullItemName() itemCode = " + itemCode + ", displayName = " + displayName + " : itemName = " + itemName);
			return itemName;
		} catch (FinderException e) {
			m_log.error("getFullItemName()", e);
		} catch (NamingException e) {
			m_log.error("getFullItemName()", e);
		}

		m_log.debug("getFullItemName() itemCode = " + itemCode + ", displayName = " + displayName + " : itemName = " + itemCode + " is not found.");
		return itemCode + " is not found.";
	}

	/**
	 * 収集項目コードをキーにした、CollectorItemTreeItemのHashMapを返します。
	 * @return
	 */
	public static HashMap<String, CollectorItemTreeItem> getItemCodeMap(){
		return m_codeTable;
	}

}