package com.clustercontrol.poller.ejb.session;

import java.rmi.RemoteException;
import java.util.concurrent.ConcurrentHashMap;

import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.poller.PollerManager;
import com.clustercontrol.poller.PollingController;
import com.clustercontrol.util.apllog.AplLogger;

/**
 *　スケジューラ(Quartz)からキックされる情報収集処理の実行クラス
 * 
 * @ejb.bean name="PollingJob"
 *           jndi-name="PollingJob"
 *           type="Stateless"
 *           transaction-type="Container"
 *           view-type="local"
 * 
 * @ejb.transaction type="Required"
 * 
 * @ejb.permission
 *     unchecked="true"
 */
public abstract class PollingJobBean implements SessionBean {

	private static Log log = LogFactory.getLog( PollingJobBean.class );

	public static final String METHOD_NAME = "run";

	/** コンテキスト情報。 */
	@SuppressWarnings("unused")
	private SessionContext m_context;

	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean の生成時に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	// Quartzに起動されたスレッドの情報を格納するクラス
	private class PollingThreadInfo {
		public PollingThreadInfo(String pollerGroup, String pollerName) {
			this.pollerGroup = pollerGroup;
			this.pollerName = pollerName;
			threadName = Thread.currentThread().getName();
		}
		private final long startTime = System.currentTimeMillis();
		public long getStartTime() {
			return startTime;
		}
		private final String threadName;
		public String getThreadName() {
			return threadName;
		}
		private final String pollerGroup;
		public String getPollerGroup() {
			return pollerGroup;
		}
		private final String pollerName;
		public String getPollerName() {
			return pollerName;
		}
		private volatile boolean isLogged = false;
		public boolean getIsLogged() {
			return isLogged;
		}
		public void setIsLogged() {
			isLogged = true;
		}
	}
	// quartzより起動されて現在アクティブなスレッドに関する情報をスレッドIDをキーに保持するマップ
	private static final ConcurrentHashMap<Long, PollingThreadInfo> activePollingThreadInfoMap = new ConcurrentHashMap<Long, PollingThreadInfo>();

	/**
	 * Quartzからのコールバックメソッド<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * トランザクション開始はユーザが制御する。
	 * また、追加実装により、トランザクションの入れ子が予期せず生じることを避けるため、Neverを採用する。
	 * @ejb.transaction type="Never"
	 * 
	 * @since 4.0.0
	 */
	public void run(String jndiName, String pollerGroup, String pollerName) {
		// デバッグログ出力
		log.debug("execute start : ");

		try {
			// 指定のポーラを取得する
			InitialContext ctx = new InitialContext();
			Object obj = ctx.lookup(jndiName);

			PollerManager manager =
				(PollerManager)PortableRemoteObject.narrow(obj, PollerManager.class);

			PollingController poller = manager.getPoller(pollerGroup, pollerName);

			// 現在走行中のスレッド情報を走査して、長時間（ここでは60*60*1000(ms)=1h）
			// 経過していて、 なおかつ今回初めて長いと判定されたスレッド情報を探し出す。
			final int timeout_min = 60;
			for (final PollingThreadInfo threadInfo : activePollingThreadInfoMap.values()) {
				if (System.currentTimeMillis() - (timeout_min*60*1000) > threadInfo.getStartTime() &&
						threadInfo.getIsLogged() == false) {

					// 今回リークした可能性があるスレッドについて情報を取りだし、ログ出力済みフラグを立てる
					final String longPollerGroup = threadInfo.getPollerGroup();
					final String longPollerName = threadInfo.getPollerName();
					threadInfo.setIsLogged();

					// 全スレッドに関する統計情報を取得する
					final int currentThreadCount = activePollingThreadInfoMap.size();
					int longThreadCount = 0;
					for (final PollingThreadInfo tmpInfo : activePollingThreadInfoMap.values()) {
						if (tmpInfo.getIsLogged()) {
							longThreadCount++;
						}
					}

					// 今回新規にリークした可能性があるスレッドの情報と、全スレッドの統計情報をログ出力する
					AplLogger apllog = new AplLogger("COMMON", "common");
					String[] args = {
							Integer.toString(timeout_min),
							longPollerGroup,
							longPollerName,
							Integer.toString(longThreadCount),
							Integer.toString(currentThreadCount)
					};
					apllog.put("SYS", "001", args);
				}
			}

			// ポーリング開始前に自身のスレッド情報をマップに登録
			activePollingThreadInfoMap.put(
					Thread.currentThread().getId(),
					new PollingThreadInfo(pollerGroup, pollerName)
			);

			// ポーリングを実行
			poller.run();
		} catch (NamingException e) {
			log.error(e.getMessage(), e);
		} finally {
			// 自身のスレッド情報を登録解除
			activePollingThreadInfoMap.remove(Thread.currentThread().getId());
		}

		// デバッグログ出力
		log.debug("execute end   : " + jndiName);
	}

}
