
/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.repository.factory;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.SnmpVersionConstant;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.poller.bean.PollerProtocolConstant;
import com.clustercontrol.poller.impl.SnmpPollerImpl;
import com.clustercontrol.repository.bean.DeviceTypeConstant;
import com.clustercontrol.repository.bean.NodeCpuInfo;
import com.clustercontrol.repository.bean.NodeDeviceInfo;
import com.clustercontrol.repository.bean.NodeDiskInfo;
import com.clustercontrol.repository.bean.NodeFilesystemInfo;
import com.clustercontrol.repository.bean.NodeHostnameInfo;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.NodeNetworkInterfaceInfo;
import com.clustercontrol.repository.util.SearchNodeProperties;
import com.clustercontrol.sharedtable.DataTable;
import com.clustercontrol.sharedtable.TableEntry;

/**
 * SNMPでノードのデータを作成するクラス<BR>
 * @author miyamoto
 * @version 4.0.0
 * @since 2.1.2
 */

public class SearchNodeBySNMP {

	private static Log m_log = LogFactory.getLog(SearchNodeBySNMP.class);

	private static final String OID_NAME      =".1.3.6.1.2.1.1.5.0";
	private static final String OID_DESCR     =".1.3.6.1.2.1.1.1.0";
	private static final String OID_CONTACT   =".1.3.6.1.2.1.1.4.0";

	private static final String OID_CPU_INDEX=".1.3.6.1.2.1.25.3.3.1.2";

	private static final String OID_DISK_INDEX=".1.3.6.1.4.1.2021.13.15.1.1.1";
	private static final String OID_DISK_NAME =".1.3.6.1.4.1.2021.13.15.1.1.2";
	private static final String OID_DISK_ION_READ = ".1.3.6.1.4.1.2021.13.15.1.1.3";
	private static final String OID_DISK_ION_WRITE = ".1.3.6.1.4.1.2021.13.15.1.1.4";
	private static final String OID_DISK_IO_READ = ".1.3.6.1.4.1.2021.13.15.1.1.5";
	private static final String OID_DISK_IO_WRITE = ".1.3.6.1.4.1.2021.13.15.1.1.6";

	private static final String OID_NIC_INDEX =".1.3.6.1.2.1.2.2.1.1";
	private static final String OID_NIC_NAME  =".1.3.6.1.2.1.2.2.1.2";
	private static final String OID_NIC_MAC_ADDRESS = ".1.3.6.1.2.1.2.2.1.6";
	private static final String OID_NIC_IP_ADDRESSV4 = ".1.3.6.1.2.1.4.34.1.3.1.4";
	private static final String OID_NIC_IP_ADDRESSV6 = ".1.3.6.1.2.1.4.34.1.3.2.16";
	private static final String OID_NIC_IP_ADDRESSV4_TYPE = ".1.3.6.1.2.1.4.34.1.4.1.4";
	private static final String OID_NIC_IP_ADDRESSV6_TYPE = ".1.3.6.1.2.1.4.34.1.4.2.16";
	private static final String OID_NIC_IN_UCAST_PKTS = ".1.3.6.1.2.1.2.2.1.11";
	private static final String OID_NIC_OUT_UCAST_PKTS = ".1.3.6.1.2.1.2.2.1.17";
	private static final String OID_NIC_IN_ERROR = ".1.3.6.1.2.1.2.2.1.14";
	private static final String OID_NIC_OUT_ERROR = ".1.3.6.1.2.1.2.2.1.20";
	private static final String OID_NIC_IN_OCTET = ".1.3.6.1.2.1.2.2.1.10";
	private static final String OID_NIC_OUT_OCTET = ".1.3.6.1.2.1.2.2.1.16";

	private static final String OID_FILESYSTEM_INDEX = ".1.3.6.1.2.1.25.2.3.1.1";
	private static final String OID_FILESYSTEM_TYPE = ".1.3.6.1.2.1.25.2.3.1.2";
	private static final String OID_FILESYSTEM_NAME=   ".1.3.6.1.2.1.25.2.3.1.3";


	/**
	 * ノードの情報を探す。
	 * @param pollingData
	 * @param mode
	 * @param locale
	 * @return
	 */
	public static NodeInfo searchNode(String ipAddress, int port, String community, String version) throws NamingException, UnknownHostException{
		m_log.debug("searchNode() ipAddress = " + ipAddress + ", port = " + port + ", community = " + community + ", version = " + version);

		// 初期化
		SearchNodeProperties pollingProp = SearchNodeProperties.getProperties();
		int retries = pollingProp.getRetry();
		int timeout = pollingProp.getTimeOut();
		ArrayList<String> oidList = pollingProp.getOidList();

		int versionInt = SnmpVersionConstant.stringToType(version);

		//pollingします。
		SnmpPollerImpl poller = new SnmpPollerImpl();
		m_log.debug("searchNode() polling start");
		DataTable tmpDataTable = poller.polling(ipAddress,
				port,
				versionInt,
				community,
				retries,
				timeout,
				oidList,
				false);
		m_log.debug("searchNode() polling stop");

		// 取得したデータを詰め替える
		DataTable ret = new DataTable();
		ret.putAll(tmpDataTable);

		//ポーリングした結果をつめるメソッドを呼び出します
		return stractProperty(ipAddress, port, community, version, ret);
	}

	/**
	 * DataTableに格納するためのEntryKeyを返すメソッド
	 * 
	 * @param oidString OID
	 */
	private static String getEntryKey(String oidString){

		return PollerProtocolConstant.PROTOCOL_SNMP + "." + oidString;
	}

	/**
	 *  ノードのプロパティーを構成します。
	 * @param IPadder
	 * @param ret
	 * @param mode
	 * @param locale
	 * @return
	 */
	private static NodeInfo stractProperty(String ipAddress, int port, String community, String version,DataTable ret) throws NamingException, UnknownHostException {

		NodeInfo property = new NodeInfo();

		/*
		 * hinemos.propertiesにrepository.find.by.snmp.verbose=trueと書くと、
		 * ディスクIOが0でも、そのディスクがfind対象になる。
		 * デフォルトはfalseであり、0のものはfind対象から外れる。
		 * since 3.2.0
		 */
		String verbose = HinemosProperties.getProperty("repository.find.by.snmp.verbose", "false");

		//ノード情報"説明"の生成
		if(ret.getValue(getEntryKey(OID_DESCR)) != null){
			if(((String)ret.getValue(getEntryKey(OID_DESCR)).getValue()).length() !=0){
				property.setDescription("Auto detect at " + (new Date()).toString());
			}
		}


		int ipAddressVersion = 0;
		try {
			InetAddress address = InetAddress.getByName(ipAddress);

			if (address instanceof Inet4Address){
				//IPv4の場合はさらにStringをチェック
				if (ipAddress.matches(".{1,3}?\\..{1,3}?\\..{1,3}?\\..{1,3}?")){
					property.setIpAddressV4(ipAddress);
					ipAddressVersion = 4;
				}

			} else if (address instanceof Inet6Address){
				property.setIpAddressV6(ipAddress);
				ipAddressVersion = 6;
			}

			//IPアドレスのバージョン
			property.setIpAddressVersion(new Integer(ipAddressVersion));
		} catch (UnknownHostException e) {
			m_log.error(e.getMessage());
			throw e;
		}


		//SNMP関連データの設定
		property.setSnmpPort(port);

		property.setSnmpCommunity(community);

		property.setSnmpVersion(version);


		//hostname の設定
		if(ret.getValue(getEntryKey(OID_NAME)) != null){
			String hostname = (String)ret.getValue(getEntryKey(OID_NAME)).getValue();
			if(hostname.length() != 0){
				//hosname.domainであればhostnameだけを入力
				if(hostname.indexOf(".") != -1){
					hostname = hostname.substring(0,hostname.indexOf("."));
				}
				//ホスト名、ノード名にそれぞれ設定
				ArrayList<NodeHostnameInfo> list = new ArrayList<NodeHostnameInfo> ();
				list.add(new NodeHostnameInfo(hostname));
				property.setNodeHostnameInfo(list);
				property.setNodeName(hostname);
			}
		}


		//連絡先はsnmpd.confに書かれている内容を設定する。
		if(ret.getValue(getEntryKey(OID_CONTACT)) != null){
			if(((String)ret.getValue(getEntryKey(OID_CONTACT)).getValue()).length() != 0){
				property.setAdministrator((String)ret.getValue(getEntryKey(OID_CONTACT)).getValue());
			}
		}

		//プラットフォーム名は、Windows, Linux, Solaris以外はOtherとする
		String platform = "OTHER";

		if(ret.getValue(getEntryKey(OID_DESCR)) != null){
			String description = ((String)ret.getValue(getEntryKey(OID_DESCR)).getValue());
			if(description.length() != 0){
				String OsName = "";

				//OSの種別はキーワードマッチで行う

				if(description.matches(".*indows.*")){
					OsName = "Windows";
					platform = "WINDOWS";
				}else if(description.matches(".*inux.*")){
					OsName = "Linux";
					platform = "LINUX";
				}else if(description.matches(".*SunOS.*") || description.matches("Solaris")){
					OsName= "Solaris";
					platform = "SOLARIS";
				}else{
					if(description.indexOf(" ") != -1){
						OsName = description.substring(0,description.indexOf(" "));
					}
				}

				//OS名は上の判定ロジックを利用した値を設定
				property.setOsName(OsName);

				//OSバージョン
				property.setOsVersion(description);
			}
		}

		// プラットフォームファミリー
		// デフォルト値はOTHER
		property.setPlatformFamily(platform);

		HashMap<String, Boolean> diskNameDuplicateSet = createDiskNameDuplicateSet(ret);
		HashMap<String, Boolean> nicNameDuplicateSet = createNicNameDuplicateSet(ret);

		// ノード作成時、「デバイス」の入力項目が１つ存在するため、
		// カウンタを利用して、既存項目として入力するか、新規項目として入力するかを判定する
		int deviceCount=0;


		// Diskの情報を設定
		ArrayList<NodeDiskInfo> diskList = new ArrayList<NodeDiskInfo> ();
		for(String fullOid : ret.keySet()){
			//DISK情報で始まるか判定
			if(!fullOid.startsWith(getEntryKey(OID_DISK_INDEX) + ".")){
				continue;
			}
			// OID_DISK_INDEX=".1.3.6.1.4.1.2021.13.15.1.1.1";
			// OID_DISK_NAME =".1.3.6.1.4.1.2021.13.15.1.1.2";
			if( ret.getValue(fullOid) == null){
				continue;
			}
			if(((Long)ret.getValue(fullOid).getValue()) == 0 ){
				continue;
			}
			m_log.debug("Find Disk : fullOid = " + fullOid);

			String i = fullOid.substring(fullOid.lastIndexOf(".") + 1);
			String disk = (String)ret.getValue(getEntryKey(OID_DISK_NAME+"."+i)).getValue();
			Long ionRead = ret.getValue(getEntryKey(OID_DISK_ION_READ + "." + i)) == null ? 0 : (Long)ret.getValue(getEntryKey(OID_DISK_ION_READ + "." + i)).getValue();
			Long ionWrite = ret.getValue(getEntryKey(OID_DISK_ION_WRITE + "." + i)) == null ? 0 : (Long)ret.getValue(getEntryKey(OID_DISK_ION_WRITE + "." + i)).getValue();
			Long ioRead = ret.getValue(getEntryKey(OID_DISK_IO_READ + "." + i)) == null ? 0 : (Long)ret.getValue(getEntryKey(OID_DISK_IO_READ + "." + i)).getValue();
			Long ioWrite = ret.getValue(getEntryKey(OID_DISK_IO_WRITE + "." + i)) == null ? 0 : (Long)ret.getValue(getEntryKey(OID_DISK_IO_WRITE + "." + i)).getValue();
			//DISK_IOが0の物は除外します。
			if("true".equals(verbose) ||
					(ionRead != 0 && ionWrite != 0 && ioRead != 0 && ioWrite != 0)) {

				// デバイス名が重複している場合は、(OIDインデックス)をつける
				if (diskNameDuplicateSet != null && diskNameDuplicateSet.get(disk) != null && diskNameDuplicateSet.get(disk)){
					disk = disk + "(" + i + ")";
				}

				NodeDiskInfo diskInfo = new NodeDiskInfo();
				// デバイス表示名
				diskInfo.setDeviceDisplayName(disk);
				// デバイス名
				diskInfo.setDeviceName(disk);
				// デバイスインデックス
				diskInfo.setDeviceIndex(((Long)ret.getValue(fullOid).getValue()).intValue());
				// デバイス種別
				diskInfo.setDeviceType(DeviceTypeConstant.DEVICE_DISK);
				// デバイス回転数(デフォルト0)
				diskInfo.setDiskRpm(0);
				deviceCount++;
				diskList.add(diskInfo);
			}
		}
		Collections.sort(diskList, new Comparator<NodeDeviceInfo>() {
			@Override
			public int compare(NodeDeviceInfo o1, NodeDeviceInfo o2) {
				int ret = 0;
				ret = o1.getDeviceType().compareTo(o2.getDeviceType());
				if (ret == 0) {
					ret = o1.getDeviceDisplayName().compareTo(o2.getDeviceDisplayName());
				}
				return ret;
			}
		}
		);
		property.setNodeDiskInfo(diskList);

		// NICの情報を設定
		ArrayList<NodeNetworkInterfaceInfo> nicList = new ArrayList<NodeNetworkInterfaceInfo> ();
		for (String fullOid : ret.keySet()) {
			//OID_NIC_INDEX =".1.3.6.1.2.1.2.2.1.1";
			//OID_NIC_NAME  =".1.3.6.1.2.1.2.2.1.2";
			//NIC情報で始まるか判定
			if(!fullOid.startsWith(getEntryKey(OID_NIC_INDEX) + ".")){
				continue;
			}
			String tmpIndex = fullOid.substring(fullOid.lastIndexOf(".") + 1);
			String deviceName = "";
			if( ((Long)ret.getValue(fullOid).getValue()) == 0 ){
				continue;
			}
			m_log.debug("Find Nic : fullOid = " + fullOid);

			deviceName = (String)ret.getValue(getEntryKey(OID_NIC_NAME + "." + tmpIndex)).getValue();

			String nicMacAddress = "";
			if (ret.getValue(getEntryKey(OID_NIC_MAC_ADDRESS + "." + tmpIndex)) != null) {
				nicMacAddress = (String)ret.getValue(getEntryKey(OID_NIC_MAC_ADDRESS + "." + tmpIndex)).getValue();
			}

			String nicIpAddress = "";

			if (ipAddressVersion == 4) {
				// IPv4 address from IP-MIB::ipAddressIfIndex.ipv4
				//
				// (sample)
				// # snmpwalk -c public -v 2c localhost .1.3.6.1.2.1.4.34.1.3.1.4
				// IP-MIB::ipAddressIfIndex.ipv4."127.0.0.1" = INTEGER: 1
				// IP-MIB::ipAddressIfIndex.ipv4."192.168.10.211" = INTEGER: 3
				// IP-MIB::ipAddressIfIndex.ipv4."192.168.10.255" = INTEGER: 3
				// IP-MIB::ipAddressIfIndex.ipv4."192.168.11.211" = INTEGER: 2
				// IP-MIB::ipAddressIfIndex.ipv4."192.168.11.255" = INTEGER: 2
				Pattern ipAddrV4Pattern = Pattern.compile("(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3})$");
				if (ret.getValueSetStartWith(getEntryKey(OID_NIC_IP_ADDRESSV4)) != null) {
					for (TableEntry entry : ret.getValueSetStartWith(getEntryKey(OID_NIC_IP_ADDRESSV4))) {
						if (entry.getValue() != null && tmpIndex.equals(((Long)entry.getValue()).toString())) {
							Matcher matcher = ipAddrV4Pattern.matcher(entry.getKey());
							if (matcher.find()) {
								// check address type (allow only unicast)
								// # snmpwalk -On -c public -v 2c 192.168.10.101 IP-MIB::ipAddressType.ipv4
								// .1.3.6.1.2.1.4.34.1.4.1.4.127.0.0.1 = INTEGER: unicast(1)
								// .1.3.6.1.2.1.4.34.1.4.1.4.192.168.10.101 = INTEGER: unicast(1)
								// .1.3.6.1.2.1.4.34.1.4.1.4.192.168.10.255 = INTEGER: broadcast(3)
								if (ret.getValue(getEntryKey(OID_NIC_IP_ADDRESSV4_TYPE + "." + matcher.group(1))) != null) {
									if ((Long)ret.getValue(getEntryKey(OID_NIC_IP_ADDRESSV4_TYPE + "." + matcher.group(1))).getValue() != 1) {
										continue;
									}
								}

								// set first matched address
								nicIpAddress = matcher.group(1);
								break;
							}
						}
					}
				}
			} else {
				// IPv6 Address from IP-MIB::ipAddressIfIndex.ipv6
				//
				// (sample)
				// # snmpwalk -c public -v 2c localhost .1.3.6.1.2.1.4.34.1.3.2.16
				// IP-MIB::ipAddressIfIndex.ipv6."00:00:00:00:00:00:00:00:00:00:00:00:00:00:00:01" = INTEGER: 1
				// IP-MIB::ipAddressIfIndex.ipv6."fe:80:00:00:00:00:00:00:50:54:00:ff:fe:be:83:2a" = INTEGER: 2
				// IP-MIB::ipAddressIfIndex.ipv6."fe:80:00:00:00:00:00:00:50:54:00:ff:fe:ed:d4:3c" = INTEGER: 3
				Pattern ipAddrV6Pattern = Pattern.compile("((\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3}))$");
				if (ret.getValueSetStartWith(getEntryKey(OID_NIC_IP_ADDRESSV6)) != null) {
					for (TableEntry entry : ret.getValueSetStartWith(getEntryKey(OID_NIC_IP_ADDRESSV6))) {
						if (entry.getValue() != null && tmpIndex.equals(((Long)entry.getValue()).toString())) {
							System.out.println(entry.getKey() + " : " + entry.getValue());
							Matcher matcher = ipAddrV6Pattern.matcher(entry.getKey());
							if (matcher.find()) {
								// check address type (allow only unicast)
								// # snmpwalk -On -c public -v 2c localhost IP-MIB::ipAddressType.ipv6
								// .1.3.6.1.2.1.4.34.1.4.2.16.0.0.0.0.0.0.0.0.0.0.0.0.0.0.0.1 = INTEGER: unicast(1)
								// .1.3.6.1.2.1.4.34.1.4.2.16.254.128.0.0.0.0.0.0.80.84.0.255.254.190.131.42 = INTEGER: unicast(1)
								if (ret.getValue(getEntryKey(OID_NIC_IP_ADDRESSV6_TYPE + "." + matcher.group(1))) != null) {
									if ((Long)ret.getValue(getEntryKey(OID_NIC_IP_ADDRESSV6_TYPE + "." + matcher.group(1))).getValue() != 1) {
										continue;
									}
								}

								// set first matched address
								String hex = "";
								for (int i = 1; i < 17; i++) {
									hex = String.format("%02x", Integer.parseInt(matcher.group(i + 1))).toUpperCase();
									nicIpAddress += "".equals(nicIpAddress) ? hex : ":" + hex;
								}
								break;
							}
						}
					}
				}
			}

			// NICのIN/OUTが0のものは除外する。
			String key = "";
			key = getEntryKey(OID_NIC_IN_UCAST_PKTS + "." + tmpIndex);
			Long inPkts = ret.getValue(key) == null ? 0 :(Long)ret.getValue(key).getValue();
			key = getEntryKey(OID_NIC_OUT_UCAST_PKTS + "." + tmpIndex);
			Long outPkts = ret.getValue(key) == null ? 0 : (Long)ret.getValue(key).getValue();
			key = getEntryKey(OID_NIC_IN_ERROR + "." + tmpIndex);
			Long inError = ret.getValue(key) == null ? 0 : (Long)ret.getValue(key).getValue();
			key = getEntryKey(OID_NIC_OUT_ERROR + "." + tmpIndex);
			Long outError = ret.getValue(key) == null ? 0 : (Long)ret.getValue(key).getValue();
			key = getEntryKey(OID_NIC_IN_OCTET + "." + tmpIndex);
			Long inOctet = ret.getValue(key) == null ? 0 : (Long)ret.getValue(key).getValue();
			key = getEntryKey(OID_NIC_OUT_OCTET + "." + tmpIndex);
			Long outOctet = ret.getValue(key) == null ? 0 : (Long)ret.getValue(key).getValue();
			if (!"true".equals(verbose) &&
					inPkts == 0 && outPkts == 0 &&
					inError == 0 && outError == 0 &&
					inOctet == 0 && outOctet == 0) {
				continue;
			}
			NodeNetworkInterfaceInfo nicInfo = new NodeNetworkInterfaceInfo();
			// デバイス名が重複している場合は、(OIDインデックス)をつける
			if (nicNameDuplicateSet != null && nicNameDuplicateSet.get(deviceName) != null && nicNameDuplicateSet.get(deviceName)){
				deviceName = deviceName + "(." + tmpIndex + ")";
			}
			// デバイス表示名
			nicInfo.setDeviceDisplayName(deviceName);
			// デバイス名の設定
			nicInfo.setDeviceName(deviceName);
			// MACアドレスの設定
			nicInfo.setNicMacAddress(nicMacAddress);
			// IPアドレスの設定
			nicInfo.setNicIpAddress(nicIpAddress);
			// デバイスINDEXの設定
			Long deviceIndex = (Long)ret.getValue(getEntryKey(OID_NIC_INDEX + "." + tmpIndex)).getValue();
			nicInfo.setDeviceIndex(deviceIndex.intValue());
			// デバイスTypeの設定
			nicInfo.setDeviceType(DeviceTypeConstant.DEVICE_NIC);
			deviceCount++;
			nicList.add(nicInfo);
		}
		Collections.sort(nicList, new Comparator<NodeDeviceInfo>() {
			@Override
			public int compare(NodeDeviceInfo o1, NodeDeviceInfo o2) {
				int ret = 0;
				ret = o1.getDeviceType().compareTo(o2.getDeviceType());
				if (ret == 0) {
					ret = o1.getDeviceDisplayName().compareTo(o2.getDeviceDisplayName());
				}
				return ret;
			}
		}
		);
		property.setNodeNetworkInterfaceInfo(nicList);

		//ファイルシステム
		deviceCount=0;
		ArrayList<NodeFilesystemInfo> filesystemList = new ArrayList<NodeFilesystemInfo> ();
		for(String fullOid : ret.keySet()) {
			// OID_FILESYSTEM_INDEX = ".1.3.6.1.2.1.25.2.3.1.1";
			// OID_FILESYSTEM_NAME  = ".1.3.6.1.2.1.25.2.3.1.3";
			//DISK情報で始まるか判定
			if(!fullOid.startsWith(getEntryKey(OID_FILESYSTEM_INDEX) + ".")){
				continue;
			}
			if( ret.getValue(fullOid) == null){
				continue;
			}
			if( ((Long)ret.getValue(fullOid).getValue()) == 0 ){
				continue;
			}
			m_log.debug("Find FileSystem : fullOid = " + fullOid);

			//hrStrageFixedDiskの場合のみノード情報に追加
			//.1.3.6.1.2.1.25.2.1.4←hrStrageFixedDisk
			String i = fullOid.substring(fullOid.lastIndexOf(".") + 1);
			String strageType =  ret.getValue(getEntryKey(OID_FILESYSTEM_TYPE+"."+i)).getValue().toString();

			if(strageType.equals(".1.3.6.1.2.1.25.2.1.4")){

				NodeFilesystemInfo filesystem = new NodeFilesystemInfo();

				//表示名
				filesystem.setDeviceName(convStringFilessystem(((String)ret.getValue(getEntryKey(OID_FILESYSTEM_NAME+"."+i)).getValue())));
				//デバイス名
				filesystem.setDeviceDisplayName(convStringFilessystem(((String)ret.getValue(getEntryKey(OID_FILESYSTEM_NAME+"."+i)).getValue())));
				//デバイスINDEX
				filesystem.setDeviceIndex(((Long)ret.getValue(getEntryKey(OID_FILESYSTEM_INDEX+"."+i)).getValue()).intValue());
				//デバイス種別
				filesystem.setDeviceType(DeviceTypeConstant.DEVICE_FILESYSTEM);

				deviceCount++;
				filesystemList.add(filesystem);
			}
		}
		Collections.sort(filesystemList, new Comparator<NodeDeviceInfo>() {
			@Override
			public int compare(NodeDeviceInfo o1, NodeDeviceInfo o2) {
				int ret = 0;
				ret = o1.getDeviceType().compareTo(o2.getDeviceType());
				if (ret == 0) {
					ret = o1.getDeviceDisplayName().compareTo(o2.getDeviceDisplayName());
				}
				return ret;
			}
		}
		);
		property.setNodeFilesystemInfo(filesystemList);

		// CPU
		deviceCount=0;
		ArrayList<NodeCpuInfo> cpuList = new ArrayList<NodeCpuInfo> ();
		for(String fullOid : ret.keySet()) {
			// OID_CPU_INDEX=".1.3.6.1.2.1.25.3.3.1.2";
			if(!fullOid.startsWith(getEntryKey(OID_CPU_INDEX) + ".")){
				continue;
			}
			if( ret.getValue(fullOid) == null){
				continue;
			}
			m_log.debug("Find Cpu : fullOid = " + fullOid);

			String indexStr = fullOid.replaceFirst(getEntryKey(OID_CPU_INDEX) + ".","");
			m_log.debug("cpu fullOid = " + fullOid + ", index = " + indexStr);
			NodeCpuInfo cpu = new NodeCpuInfo();
			cpu.setDeviceDisplayName(DeviceTypeConstant.DEVICE_CPU + deviceCount);
			cpu.setDeviceIndex(Integer.valueOf(indexStr));
			cpu.setDeviceName(indexStr);
			cpu.setDeviceType(DeviceTypeConstant.DEVICE_CPU);

			cpuList.add(cpu);
			deviceCount++;
		}
		Collections.sort(cpuList, new Comparator<NodeDeviceInfo>() {
			@Override
			public int compare(NodeDeviceInfo o1, NodeDeviceInfo o2) {
				int ret = 0;
				ret = o1.getDeviceType().compareTo(o2.getDeviceType());
				if (ret == 0) {
					ret = o1.getDeviceDisplayName().compareTo(o2.getDeviceDisplayName());
				}
				return ret;
			}
		}
		);
		property.setNodeCpuInfo(cpuList);


		m_log.debug("find-by-snmp " + property.toString());
		return property;
	}

	/**
	 * SNMPで取得できるファイルシステム名(マウント場所)から表示名に変換するメソッド。
	 * 例：
	 * C:\ Label: Serial Number 9058dd61
	 * ↓
	 * C:\
	 */
	private static String convStringFilessystem(String str){

		if(str.matches("/.*")){
			//UNIX系OSの場合
			return str;
		}
		else if(str.matches("[A-Z]:\\\\.*")){
			//Winodws系OSの場合
			return str;
		}
		else{
			//strはなぜかASCIIが16進で入っている。
			//文字が化けて入力された場合には、変換を行う。
			char[] chars;
			short  first;
			short second;
			chars=str.toCharArray();
			StringBuffer ret = new StringBuffer();

			int j=0;

			for(int i = 0; i<chars.length;){


				first  = (short) (chars[i] - 48); //48は"0"なので、"1"→1の変換
				second = (short)(chars[i+1]- 48);

				if(second >10){
					//16進でA-Fの場合には、さらに7を引く
					//10 ⇔ A(16進)なので("A"→65)を10にする、 10 = "A"(65) - "0"(48) - 7
					second -=7;
				}


				//				ret[j] = (char) (first*16+second);
				ret.append((char) (first*16+second));

				i+=3;//1文字は"56 "とかで構成するので、配列2つを読んで" "をスキップ
				j++;//リターンも文字もインクリメント
			}

			m_log.info("Find By SNMP : Filesystem Name str = " + str + " convert to " + ret.substring(0,3));
			return ret.substring(0,3);
		}

	}


	private static HashMap<String, Boolean> createDiskNameDuplicateSet(DataTable dataTable){
		HashMap<String, Boolean> ret = new HashMap<String, Boolean>();


		int i = 1;

		while (true) {

			// OID_DISK_INDEX=".1.3.6.1.4.1.2021.13.15.1.1.1";
			// OID_DISK_NAME =".1.3.6.1.4.1.2021.13.15.1.1.2";
			String oidCounter = OID_DISK_INDEX+"."+i;

			if( dataTable.getValue(oidCounter) != null){
				if(((String)dataTable.getValue(oidCounter).getValue()).length() != 0 ){
					String disk = (String)dataTable.getValue(OID_DISK_NAME+"."+i).getValue();

					//ramディスクは除外します。
					if(!(disk.startsWith("ram"))){

						// デバイス名
						String diskName = (String)dataTable.getValue(OID_DISK_NAME+"."+i).getValue();

						Boolean isDuplicate = ret.get(diskName);
						if (isDuplicate == null) {
							ret.put(diskName, false);
						} else if (!isDuplicate) {
							ret.put(diskName, true);
						}
					}

					i++;

				} else {
					break;
				}
			} else {
				break;
			}
		}


		return ret;
	}

	private static HashMap<String, Boolean> createNicNameDuplicateSet(DataTable dataTable){
		HashMap<String, Boolean> ret = new HashMap<String, Boolean>();

		for (String fullOid : dataTable.keySet()) {
			//OID_NIC_INDEX =".1.3.6.1.2.1.2.2.1.1";
			//OID_NIC_NAME  =".1.3.6.1.2.1.2.2.1.2";

			if(fullOid.startsWith(OID_NIC_INDEX + ".")){//NIC情報で始まるか判定

				String tmpIndex = fullOid.substring(fullOid.lastIndexOf("."));

				if( ((String)dataTable.getValue(fullOid).getValue()).length() != 0 ){

					//String nicName = convString((String)dataTable.getValue(OID_NIC_NAME+tmpIndex).getValue(),isWindows);
					String nicName = (String)dataTable.getValue(OID_NIC_NAME+tmpIndex).getValue();
					m_log.info("NIC name = " + nicName);

					Boolean isDuplicate = ret.get(nicName);
					if (isDuplicate == null) {
						ret.put(nicName, false);
					} else if (!isDuplicate) {
						ret.put(nicName, true);
					}

				}
			}
		}

		return ret;
	}

}
