/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.selfcheck;

import java.util.Date;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.selfcheck.monitor.DBSizeMonitor;
import com.clustercontrol.selfcheck.monitor.DataSourceMonitor;
import com.clustercontrol.selfcheck.monitor.FileSystemMonitor;
import com.clustercontrol.selfcheck.monitor.JMSQueueMonitor;
import com.clustercontrol.selfcheck.monitor.JVMHeapMonitor;
import com.clustercontrol.selfcheck.monitor.JobRunSessionMonitor;
import com.clustercontrol.selfcheck.monitor.QuartzMonitor;
import com.clustercontrol.selfcheck.monitor.RAMSwapOutMonitor;
import com.clustercontrol.selfcheck.monitor.SyslogFilterMonitor;

public class SelfCheckTaskSubmitter implements Runnable {

	private static Log log = LogFactory.getLog( SelfCheckTaskSubmitter.class );

	private final ExecutorService _executorService;

	public SelfCheckTaskSubmitter() {
		_executorService = Executors.newFixedThreadPool(
				SelfCheckConfig.getThreadPoolSize(),
				new SelfCheckTaskThreadFactory()
		);
	}

	@Override
	public void run() {
		/** メイン処理 */
		// Java VM Heap
		if (SelfCheckConfig.getJvmHeapMonitoring()) {
			_executorService.submit(
					new SelfCheckTask(
							new JVMHeapMonitor(SelfCheckConfig.getJvmHeapThresholdMByte())
					)
			);
		}

		// Datasource
		if (SelfCheckConfig.getDatasourceMonitoring()) {
			// HinemosDS
			_executorService.submit(
					new SelfCheckTask(
							new DataSourceMonitor(SelfCheckConfig.getJndiDatasourceHinemos())
					)
			);
		}

		// FileSystem
		if (SelfCheckConfig.getFileSystemMonitoring()) {
			// DB
			_executorService.submit(
					new SelfCheckTask(
							new FileSystemMonitor(
									SelfCheckConfig.getMountPointDatabase(),
									SelfCheckConfig.getFileSystemDBThresholdPer(),
									SelfCheckConfig.getSnmpPort(),
									SelfCheckConfig.getSnmpVersion(),
									SelfCheckConfig.getSnmpCommunity(),
									SelfCheckConfig.getSnmpRetries(),
									SelfCheckConfig.getSnmpTimeoutMSec()
							)
					)
			);

			if (SelfCheckConfig.getMountPointLog().equals(SelfCheckConfig.getMountPointDatabase())) {
				if (log.isDebugEnabled()) log.debug("skipped log's mount point because database's mount point is same.");
			} else {
				// Log
				_executorService.submit(
						new SelfCheckTask(
								new FileSystemMonitor(
										SelfCheckConfig.getMountPointLog(),
										SelfCheckConfig.getFileSystemLogThresholdPer(),
										SelfCheckConfig.getSnmpPort(),
										SelfCheckConfig.getSnmpVersion(),
										SelfCheckConfig.getSnmpCommunity(),
										SelfCheckConfig.getSnmpRetries(),
										SelfCheckConfig.getSnmpTimeoutMSec()
								)
						)
				);
			}
		}

		// Scheduler
		if (SelfCheckConfig.getSchedulerMonitoring()) {
			_executorService.submit(
					new SelfCheckTask(
							new QuartzMonitor(
									SelfCheckConfig.getJndiDBMSScheduler(),
									SelfCheckConfig.getTriggerNameDBMS(),
									SelfCheckConfig.getTriggerGroupDBMS(),
									SelfCheckConfig.getSchedulerDelayThresholdSec()
							)
					)
			);
		}

		// swap-out
		if (SelfCheckConfig.getSwapOutMonitoring()) {
			_executorService.submit(
					new SelfCheckTask(
							new RAMSwapOutMonitor(
									SelfCheckConfig.getIntervalSec(),
									SelfCheckConfig.getSnmpPort(),
									SelfCheckConfig.getSnmpVersion(),
									SelfCheckConfig.getSnmpCommunity(),
									SelfCheckConfig.getSnmpRetries(),
									SelfCheckConfig.getSnmpTimeoutMSec()
							)
					)
			);
		}

		// syslogforward
		if (SelfCheckConfig.getSyslogFilterMonitoring()) {
			for (String host : SelfCheckConfig.getSyslogFilterHost()) {
				_executorService.submit(
						new SelfCheckTask(
								new SyslogFilterMonitor(host)
						)
				);
			}
		}

		// JMS
		if (SelfCheckConfig.getJmsMonitoring()) {
			for (String queue : SelfCheckConfig.getMonitorQueue()) {
				_executorService.submit(
						new SelfCheckTask(
								new JMSQueueMonitor(
										"Queue",
										queue,
										"MessageCount",
										SelfCheckConfig.getJmsMessageCountThreshold()
								)
						)
				);
			}

			for (String topic : SelfCheckConfig.getMonitorTopic()) {
				_executorService.submit(
						new SelfCheckTask(
								new JMSQueueMonitor(
										"Topic",
										topic,
										"AllMessageCount",
										SelfCheckConfig.getJmsMessageCountThreshold()
								)
						)
				);
			}
		}

		// log table
		if (SelfCheckConfig.getLogTableSizeMonitoring()) {
			// event
			_executorService.submit(
					new SelfCheckTask(
							new DBSizeMonitor(
									SelfCheckConfig.getJndiLogTableDS(),
									SelfCheckConfig.getTableNameEventLog(),
									SelfCheckConfig.getTableNameEventLogDesc(),
									SelfCheckConfig.getThresholdEventLog(),
									SelfCheckConfig.getThresholdTypeEventLog()
							)
					)
			);

			// resource collection
			_executorService.submit(
					new SelfCheckTask(
							new DBSizeMonitor(
									SelfCheckConfig.getJndiLogTableDS(),
									SelfCheckConfig.getTableNameCollectedResourceLog(),
									SelfCheckConfig.getTableNameCollectedResourceLogDesc(),
									SelfCheckConfig.getThresholdCollectedResourceLog(),
									SelfCheckConfig.getThresholdTypeCollectedResourceLog()
							)
					)
			);

			// job
			_executorService.submit(
					new SelfCheckTask(
							new DBSizeMonitor(
									SelfCheckConfig.getJndiLogTableDS(),
									SelfCheckConfig.getTableNameJobLog(),
									SelfCheckConfig.getTableNameJobLogDesc(),
									SelfCheckConfig.getThresholdJobLog(),
									SelfCheckConfig.getThresholdTypeJobLog()
							)
					)
			);

			// collective run
			_executorService.submit(
					new SelfCheckTask(
							new DBSizeMonitor(
									SelfCheckConfig.getJndiLogTableDS(),
									SelfCheckConfig.getTableNameCrunLog(),
									SelfCheckConfig.getTableNameCrunLogDesc(),
									SelfCheckConfig.getThresholdCrunLog(),
									SelfCheckConfig.getThresholdTypeCrunLog()
							)
					)
			);
		}

		// job
		if(SelfCheckConfig.getJobMonitoring()){
			_executorService.submit(
					new SelfCheckTask(
							new JobRunSessionMonitor(
									SelfCheckConfig.getJndiLogTableDS(),
									SelfCheckConfig.getThresholdJobRunSession()
							)
					)
			);
		}

		// set timestamp of last monitoring
		SelfCheckConfig.setLastMonitorDate(new Date());
	}

	private static class SelfCheckTaskThreadFactory implements ThreadFactory {
		private volatile int _count = 0;

		@Override
		public Thread newThread(Runnable r) {
			return new Thread(r, "SelfCheckWorker-" + _count++);
		}
	}

}
