/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.selfcheck.monitor;

import java.util.Date;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.util.Messages;

/**
 * Java VMの利用可能なヒープ容量を確認する処理の実装クラス
 */
public class JVMHeapMonitor extends SelfCheckMonitorBase {

	private static Log m_log = LogFactory.getLog( JVMHeapMonitor.class );

	private int jvmMinHeapThreshold = 0;

	private String monitorId = "SYS_JVM_HEAP";
	private String application = "SELFCHECK (Java VM Heap)";

	/**
	 * コンストラクタ
	 * @param jvmMinHeapThreshold Java VMの利用可能なヒープ容量の最小値
	 */
	public JVMHeapMonitor(int jvmMinHeapThreshold) {
		this.jvmMinHeapThreshold = jvmMinHeapThreshold;
	}

	/**
	 * セルフチェック処理名
	 */
	@Override
	public String toString() {
		return "monitoring jvm free heap";
	}

	/**
	 * 監視項目ID
	 */
	@Override
	public String getMonitorId() {
		return monitorId;
	}

	/**
	 * Java VMの利用可能なヒープ容量が最小値以上であるかを確認する処理
	 * @return 通知情報（アプリケーション名は未格納）
	 */
	@Override
	public OutputBasicInfo execute() {
		/** ローカル変数 */
		int freeHeapMByte = 0;

		OutputBasicInfo notifyInfo = null;

		/** メイン処理 */
		if (m_log.isDebugEnabled()) m_log.debug("monitoring java vm heap size.");

		// 利用可能なヒープ容量をMByte単位で取得する
		freeHeapMByte = (int)(Runtime.getRuntime().freeMemory() / 1024 / 1024);

		if (jvmMinHeapThreshold >= 0 && freeHeapMByte >= jvmMinHeapThreshold) {
			if (m_log.isDebugEnabled()) m_log.debug("size of java vm's free heap is enough. (free heap's size = " + freeHeapMByte + " [MByte], threshold = " + jvmMinHeapThreshold + " [MByte])");

			// set result, but do not notify (priority == info)
			notifyInfo = new OutputBasicInfo();
			notifyInfo.setPluginId(PLUGIN_ID);
			notifyInfo.setMonitorId(monitorId);
			notifyInfo.setPriority(PriorityConstant.TYPE_INFO);
		} else {
			m_log.warn("size of java vm's free heap is low. (free heap's size = " + freeHeapMByte + " [MByte], threshold = " + jvmMinHeapThreshold + " [MByte])");
			notifyInfo = new OutputBasicInfo();

			String[] msgAttr1 = { new Integer(freeHeapMByte).toString(), new Integer(jvmMinHeapThreshold).toString() };
			String[] msgAttr2 = { new Integer(freeHeapMByte).toString(), new Integer(jvmMinHeapThreshold).toString() };

			notifyInfo.setPluginId(PLUGIN_ID);
			notifyInfo.setMonitorId(monitorId);
			notifyInfo.setApplication(application);
			notifyInfo.setMessageId("001");
			notifyInfo.setMessage(Messages.getString("message.selfcheck.notify.jvm.failure.msg", msgAttr1));
			notifyInfo.setMessageOrg(Messages.getString("message.selfcheck.notify.jvm.failure.origmsg", msgAttr2));
			notifyInfo.setFacilityId(FACILITY_ID);
			notifyInfo.setScopeText(FACILITY_TEXT);
			notifyInfo.setPriority(PriorityConstant.TYPE_WARNING);
			notifyInfo.setGenerationDate(new Date().getTime());
		}

		return notifyInfo;
	}

}
