/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.selfcheck.monitor;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Date;

import javax.naming.InitialContext;
import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.util.Messages;

/**
 * ジョブセッション数を確認する処理の実装クラス
 */
public class JobRunSessionMonitor extends SelfCheckMonitorBase {

	private static Log m_log = LogFactory.getLog( JobRunSessionMonitor.class );

	private String dataSourceName = "";
	private String jndiName = "";
	private final String jndiPrefix = "java:/";

	private long threshold = 0;
	private String monitorId = "SYS_JOB_SESSION";
	private String application = "SELFCHECK (Job Run Session)";

	public JobRunSessionMonitor(String dataSourceName, long threshold) {
		this.dataSourceName = dataSourceName;
		this.jndiName = jndiPrefix + dataSourceName;

		this.threshold = threshold;
	}

	/**
	 * セルフチェック処理名
	 */
	@Override
	public String toString() {
		return "monitoring job run session";
	}

	/**
	 * 監視項目ID
	 */
	@Override
	public String getMonitorId() {
		return monitorId;
	}

	/**
	 * データソースへの疎通確認
	 * @return 通知情報（アプリケーション名は未格納）
	 */
	@Override
	public OutputBasicInfo execute() {
		/** ローカル変数 */
		InitialContext initCtx = null;
		DataSource dataSource = null;
		Connection conn = null;

		long thresholdOrig = threshold;
		long count = -1;

		OutputBasicInfo notifyInfo = null;

		/** メイン処理 */
		if (m_log.isDebugEnabled())
			m_log.debug("monitoring job run session. (dataSource = " + dataSourceName + ", jndi=" + jndiName
					+ ", threshold=" + threshold);
		try {
			// データソースのオブジェクトをJNDI経由で取得し、取得したコネクションが正しく動作するかを確認する
			initCtx = new InitialContext();
			dataSource = (DataSource)initCtx.lookup(jndiName);

			conn = dataSource.getConnection();

			// 判定対象値を取得する
			count = getJobRunSessionCount(conn);

			if (count == -1) {
				if (m_log.isInfoEnabled()) {
					m_log.info("skipped monitoring job run session. (dataSource = " + dataSourceName + ", jndi=" + jndiName
							+ ", threshold=" + thresholdOrig);
				}
			} else if (count <= threshold) {
				if (m_log.isDebugEnabled()) {
					m_log.debug("job run session size is low. (dataSource = " + dataSourceName + ", jndi=" + jndiName
							+ ", count=" + count + ", threshold=" + thresholdOrig);
				}

				// set result, but do not notify (priority == info)
				notifyInfo = new OutputBasicInfo();
				notifyInfo.setPluginId(PLUGIN_ID);
				notifyInfo.setMonitorId(monitorId);
				notifyInfo.setPriority(PriorityConstant.TYPE_INFO);
			} else {
				m_log.warn("job run session size is too high. (dataSource = " + dataSourceName + ", jndi=" + jndiName
						+ ", count=" + count + ", threshold=" + thresholdOrig);

				notifyInfo = new OutputBasicInfo();

				String[] msgAttr1 = { new Long(count).toString(), new Long(thresholdOrig).toString() };
				String[] msgAttr2 = { new Long(count).toString(), new Long(thresholdOrig).toString() };

				notifyInfo.setPluginId(PLUGIN_ID);
				notifyInfo.setMonitorId(monitorId);
				notifyInfo.setApplication(application);
				notifyInfo.setMessageId("001");
				notifyInfo.setMessage(Messages.getString("message.selfcheck.notify.job.runsession.failure.msg", msgAttr1));
				notifyInfo.setMessageOrg(Messages.getString("message.selfcheck.notify.job.runsession.failure.origmsg", msgAttr2));
				notifyInfo.setFacilityId(FACILITY_ID);
				notifyInfo.setScopeText(FACILITY_TEXT);
				notifyInfo.setPriority(PriorityConstant.TYPE_WARNING);
				notifyInfo.setGenerationDate(new Date().getTime());
			}
		} catch (Exception e) {
			m_log.warn("monitoring job run session failure. (dataSource = " + dataSourceName + ", jndi=" + jndiName
					+ ", threshold=" + threshold);
		} finally {
			try {
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e) {
				m_log.warn("closing connection failure. (dataSource = " + dataSourceName + ", jndi=" + jndiName + ")", e);
			}
		}

		return notifyInfo;
	}


	/**
	 * 実行中のジョブセッションレコード数を返すメソッド
	 * @param conn 問い合わせに利用するコネクション
	 * @return レコード数
	 */
	private static long getJobRunSessionCount(Connection conn) {
		// ローカル変数
		Statement stmt = null;
		ResultSet res = null;

		String query = "SELECT count(*) " +
		"FROM cc_job_session s, cc_job_session_job j " +
		"WHERE s.session_id = j.session_id AND s.jobunit_id = j.jobunit_id " +
		"AND s.job_id = j.job_id AND status NOT IN (300,301)";
		long count = -1;

		// メイン処理
		try {
			stmt = conn.createStatement();
			res = stmt.executeQuery(query);
			if (res.next()) {
				count = res.getLong("count");
			}
		} catch (Exception e) {
			m_log.warn("database query execution failure. (" + query + ")", e);
		} finally {
			try {
				if (stmt != null) {
					stmt.close();
				}
			} catch (Exception e) {
				m_log.warn("closing statement failure.", e);
			}
		}

		return count;
	}

}
