/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.selfcheck.monitor;

import java.util.Date;

import javax.naming.InitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.Scheduler;
import org.quartz.Trigger;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.util.Messages;

/**
 * Quartzスケジューラの稼動状態を確認する処理の実装クラス
 */
public class QuartzMonitor extends SelfCheckMonitorBase {

	private static Log m_log = LogFactory.getLog( QuartzMonitor.class );

	private String jndiName = null;
	private String triggerName = null;
	private String triggerGroupName = null;
	private int thresholdSec = 0;

	private static final String monitorIdPrefix = "SYS_SCHEDULE";
	private String monitorId = "";
	private String application = "SELFCHECK (Scheduler)";

	/**
	 * コンストラクタ
	 * @param jndiName QuartzのJNDI名（QuartzServiceなど）
	 * @param jobName 確認の対象とするジョブ名
	 * @param thresholdSec 最大遅延時間（この時間より長い場合、スケジュール実行時刻が遅延している場合に非稼動とする）
	 */
	public QuartzMonitor(String jndiName, String triggerName, String triggerGroupName, int thresholdSec) {
		this.jndiName = jndiName;
		this.triggerName = triggerName;
		this.triggerGroupName = triggerGroupName;
		this.thresholdSec = thresholdSec;

		this.monitorId = monitorIdPrefix + "_" + jndiName;
	}

	/**
	 * セルフチェック処理名
	 */
	@Override
	public String toString() {
		return "monitoring quartz scheduler - " + jndiName;
	}

	/**
	 * 監視項目ID
	 */
	@Override
	public String getMonitorId() {
		return monitorId;
	}

	/**
	 * Quartzスケジューラの稼動状態を確認する処理
	 * @return 通知情報（アプリケーション名は未格納）
	 */
	@Override
	public OutputBasicInfo execute() {
		/** ローカル変数 */
		InitialContext initCtx = null;
		Scheduler scheduler = null;
		Trigger trigger = null;

		Date now = null;
		Date nextFireTime = null;
		String nextFireTimeStr = null;

		OutputBasicInfo notifyInfo = null;

		/** メイン処理 */
		if (m_log.isDebugEnabled()) m_log.debug("monitoring a quartz scheduler. (jndiName = " + jndiName + ", triggerName = " + triggerName + ", triggerGroupName = " + triggerGroupName + ")");

		try {
			initCtx = new InitialContext();
			scheduler = (Scheduler)initCtx.lookup(jndiName);
			trigger = scheduler.getTrigger(triggerName, triggerGroupName);

			now = new Date();
			nextFireTime = trigger.getNextFireTime();
			if (nextFireTime != null) {
				nextFireTimeStr = String.format("%1$tY-%1$tm-%1$td %1$tH:%1$tM:%1$tS", nextFireTime);
			}
		} catch (Exception e) {
			m_log.warn("quartz scheduler access failure. (jndiName = " + jndiName + ", triggerName = " + triggerName + ", triggerGroupName = " + triggerGroupName + ")", e);
		}

		if (nextFireTime != null) {
			if (now.getTime() - nextFireTime.getTime() <= thresholdSec * 1000) {
				if (m_log.isDebugEnabled())
					m_log.debug("quartz scheduler is running without delay. (jndiName = " + jndiName + ", triggerName = " + triggerName + ", triggerGroupName = " + triggerGroupName + ", nextFireTime = " + nextFireTimeStr + ")");

				// set result, but do not notify (priority == info)
				notifyInfo = new OutputBasicInfo();
				notifyInfo.setPluginId(PLUGIN_ID);
				notifyInfo.setMonitorId(monitorId);
				notifyInfo.setPriority(PriorityConstant.TYPE_INFO);
			} else {
				m_log.warn("quartz scheduler may be not running. (jndiName = " + jndiName + ", triggerName = " + triggerName + ", triggerGroupName = " + triggerGroupName + ", nextFireTime = " + nextFireTimeStr + ")");
				notifyInfo = new OutputBasicInfo();

				String[] msgAttr1 = { jndiName, new Integer(thresholdSec).toString() };
				String[] msgAttr2 = { jndiName, nextFireTimeStr, new Integer(thresholdSec).toString() };

				notifyInfo.setPluginId(PLUGIN_ID);
				notifyInfo.setMonitorId(monitorId);
				notifyInfo.setApplication(application);
				notifyInfo.setMessageId("001");
				notifyInfo.setMessage(Messages.getString("message.selfcheck.notify.quartz.failure.msg", msgAttr1));
				notifyInfo.setMessageOrg(Messages.getString("message.selfcheck.notify.quartz.failure.origmsg", msgAttr2));
				notifyInfo.setFacilityId(FACILITY_ID);
				notifyInfo.setScopeText(FACILITY_TEXT);
				notifyInfo.setPriority(PriorityConstant.TYPE_WARNING);
				notifyInfo.setGenerationDate(new Date().getTime());
			}
		} else {
			if (m_log.isInfoEnabled())
				m_log.info("skipped monitoring a quartz scheduler. (jndiName = " + jndiName + ", triggerName = " + triggerName + ", triggerGroupName = " + triggerGroupName + ")");
		}

		return notifyInfo;
	}

}
