/*

 Copyright (C) 2011 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 */

package com.clustercontrol.systemlog.jmx;

import java.nio.charset.Charset;

import javax.naming.CompositeName;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NamingException;

import org.jboss.system.ServiceMBeanSupport;
import org.jboss.util.naming.NonSerializableFactory;

import com.clustercontrol.systemlog.SyslogReceiverConfig;
import com.clustercontrol.systemlog.service.MatcherTask;
import com.clustercontrol.systemlog.service.SyslogReceiver;
import com.clustercontrol.systemlog.service.SyslogReceiver.Mode;

/**
 * syslogを受信するmBean <BR>
 * 
 * @jmx.mbean
 *     name="user:service=SyslogReceiveService,name=SyslogReceiveService"
 *     description="SnmpTrapService MBean"
 *     extends="org.jboss.system.ServiceMBean"
 * 
 * @jboss.service servicefile="SyslogReceive"
 */
public class SyslogReceiveService extends ServiceMBeanSupport implements SyslogReceiveServiceMBean{

	private String jndiName = "SyslogReceiveService";  // JNDIに登録する際の名前

	private final SyslogReceiverConfig config = new SyslogReceiverConfig();

	private SyslogReceiver _receiver;

	public SyslogReceiveService(){ }

	/**
	 * @jmx.managed-attribute

	 * @see com.clustercontrol.troubledetection.jmx.SyslogReceiveServiceMBean#getJndiName()
	 */
	@Override
	public String getJndiName() {
		return jndiName;
	}

	/**
	 * @jmx.managed-attribute
	 * 
	 * @see com.clustercontrol.troubledetection.jmx.SyslogReceiveServiceMBean#setJndiName(java.lang.String)
	 * 
	 * @param jndiName
	 * @throws NamingException
	 */
	@Override
	public void setJndiName(String jndiName) throws NamingException {
		log.info("setJndiName() : jndiName = " + jndiName);

		String oldName = this.jndiName;
		this.jndiName = jndiName;

		if (super.getState() == STARTED) {
			try {
				unbind(oldName);
			} catch (NamingException e){
				NamingException ne = new NamingException("Failed to unbind");
				ne.setRootCause(e);
				log.error(ne);
			}

			try{
				rebind();
			} catch (NamingException e){
				NamingException ne = new NamingException("Failed to rebind");
				ne.setRootCause(e);
				log.error(ne);
			}
		}
	}

	/**
	 * @jmx.managed-attribute
	 * 
	 * @see org.jboss.system.ServiceMBean#getName()
	 */
	@Override
	public String getName() {
		return "SyslogReceiveService (" + jndiName + " : " + config + ")";
	}

	/* (non-Javadoc)
	 * @see org.jboss.system.ServiceMBeanSupport#createService()
	 */
	@Override
	public void createService() {
		log.info("Create SyslogReceiveService(" + jndiName + ")");
		log.info("Created SyslogReceiveService(" + jndiName + ")");
	}

	/* (non-Javadoc)
	 * @see org.jboss.system.ServiceMBeanSupport#startService()
	 */
	@Override
	public void startService() throws NamingException {
		log.info("Start SyslogReceiveService(" + jndiName + ")");

		try {
			_receiver = new SyslogReceiver(config);
			_receiver.start();
		} catch (Exception e) {
			log.error(e.getMessage(), e);
		}

		rebind();

		log.info("Started SyslogReceiveService(" + jndiName + ")");
	}

	/* (non-Javadoc)
	 * @see org.jboss.system.ServiceMBeanSupport#stopService()
	 */
	@Override
	public void stopService() throws NamingException {
		log.info("Stop SyslogReceiveService(" + jndiName + ")");

		// 監視終了
		_receiver.stop();

		unbind(jndiName);

		log.info("Stoped SyslogReceiveService(" + jndiName + ")");
	}

	/* (non-Javadoc)
	 * @see org.jboss.system.ServiceMBeanSupport#destroyService()
	 */
	@Override
	public void destroyService() throws Exception {
		log.info("Destroy SyslogReceiveService(" + jndiName + ")");

		//監視終了
		if(_receiver != null)
			_receiver.stop();

		_receiver = null;

		log.info("Destroyed SyslogReceiveService(" + jndiName + ")");
	}

	private static Context createContext(Context rootCtx, Name name) throws NamingException {
		Context subctx = rootCtx;

		for (int n = 0; n < name.size(); n++) {
			String atom = name.get(n);

			try {
				Object obj = subctx.lookup(atom);
				subctx = (Context) obj;
			} catch (NamingException e) {
				// 存在しない場合は、サブコンテキストを生成
				subctx = subctx.createSubcontext(atom);
			}
		}

		return subctx;
	}

	private void rebind() throws NamingException {
		InitialContext rootCtx = new InitialContext();

		Name fullName = rootCtx.getNameParser("").parse(jndiName);
		Name parentName = fullName;
		if(fullName.size() > 1){
			parentName = fullName.getPrefix(fullName.size()-1);
		} else {
			parentName = new CompositeName();
		}

		Context parentCtx = createContext(rootCtx, parentName);
		Name atomName = fullName.getSuffix(fullName.size()-1);
		String atomStirng = atomName.get(0);

		NonSerializableFactory.rebind(parentCtx, atomStirng, _receiver);
	}

	private void unbind(String jndiName) throws NamingException {
		InitialContext rootCtx = null;

		try {
			rootCtx = new InitialContext();

			Name fullName = rootCtx.getNameParser("").parse(jndiName);
			Name atomName = fullName.getSuffix(fullName.size() - 1);
			String atom = atomName.get(0);

			rootCtx.unbind(jndiName);
			NonSerializableFactory.unbind(atom);
		} finally {
			if(rootCtx != null) {
				rootCtx.close();
			}
		}
	}

	/**
	 * @jmx.managed-attribute
	 * ホスト名を返します。
	 * 
	 * @return ホスト名
	 */
	@Override
	public String getHost() {
		return config.getListenAddress();
	}

	/**
	 * @jmx.managed-attribute
	 * ホスト名を設定します。
	 * 
	 * @param host ホスト名
	 */
	@Override
	public void setHost(String host) {
		config.setListenAddress(host);
	}

	/**
	 * @jmx.managed-attribute
	 * 待ち受けポート番号を返します。
	 * 
	 * @return 待ち受けポート番号
	 */
	@Override
	public int getPort() {
		return config.getListenPort();
	}

	/**
	 * @jmx.managed-attribute
	 *
	 * 待ち受けポート番号を設定します。
	 * @param port 待ち受けポート番号
	 */
	@Override
	public void setPort(int port) {
		config.setLisntenPort(port);
	}



	/**
	 * @jmx.managed-attribute
	 * マッチング処理の同時実行スレッド数を返します。
	 * 
	 * @return マッチング処理の同時実行スレッド数
	 */
	@Override
	public int getThreadPoolSize() {
		return config.getThreadPoolSize();
	}

	/**
	 * @jmx.managed-attribute
	 * マッチング処理の同時実行スレッド数を設定します。
	 * 
	 * @param threadPoolSize マッチング処理の同時実行スレッド数
	 */
	@Override
	public void setThreadPoolSize(int threadPoolSize) {
		config.setThreadPoolSize(threadPoolSize);
	}

	/**
	 * @jmx.managed-attribute
	 * Queuingできるマッチング処理の最大数を返します。
	 * 
	 * @return Queuingできるマッチング処理の最大数
	 */
	@Override
	public int getFilterTaskQueueSize() {
		return config.getFilterTaskQueueSize();
	}

	/**
	 * @jmx.managed-attribute
	 * Queuingできるマッチング処理の最大数を設定します。
	 * 
	 * @param threadPoolSize Queuingできるマッチング処理の最大数
	 */
	@Override
	public void setFilterTaskQueueSize(int queueSize) {
		config.setFilterTaskQueueSize(queueSize);
	}

	/**
	 * @jmx.managed-attribute
	 * バイト配列を復号化する際の文字セットを返します。
	 * 
	 * @return バイト配列を復号化する際の文字セット
	 */
	@Override
	public String getCharsetName() {
		return config.getCharset().name();
	}

	/**
	 * @jmx.managed-attribute
	 * バイト配列を復号化する際の文字セットを設定します。
	 * 
	 * @param charsetName バイト配列を復号化する際の文字セット
	 */
	@Override
	public void setCharsetName(String charsetName) {
		Charset charset = Charset.forName("UTF-8");;
		try {
			charset = Charset.forName(charsetName);
		} catch (Exception e) {
			charset = Charset.forName("UTF-8");
		} finally {
			config.setCharset(charset);
		}
	}

	/**
	 * @jmx.managed-attribute
	 * 
	 * 動作モードを返す。
	 */
	@Override
	public String getMode() {
		return config.getQueueMode().name();
	}

	/**
	 * @jmx.managed-attribute
	 * 動作モードを設定します。
	 * 
	 * @param multiId 動作モード
	 */
	@Override
	public void setMode(String mode) {
		Mode queueMode = Mode.POOLED_MULTI_THREAD;
		try {
			queueMode = Mode.valueOf(mode);
		} catch (Exception e) {
			queueMode = Mode.POOLED_MULTI_THREAD;
		} finally {
			config.setQueueMode(queueMode);
		}
	}

	/**
	 * @jmx.managed-attribute
	 * 通知用クラス名を返す。
	 */
	@Override
	public String getLogNotifierClass() {
		return config.getLogNotifierClass();
	}

	/**
	 * @jmx.managed-attribute
	 * ログ通知用クラス名を設定します。
	 */
	@Override
	public void setLogNotifierClass(String className) {
		config.setLogNotifierClass(className);
	}

	/**
	 * @jmx.managed-attribute
	 * 
	 * マネージャ起動時から受信したsyslogメッセージ数を返す。
	 */
	@Override
	public long getMessageCount() {
		return _receiver.getMessageCount();
	}

	/**
	 * @jmx.managed-attribute
	 * 
	 * syslogでパースに失敗したメッセージの数を返す。
	 */
	@Override
	public long getInvalidFormatMessageCount() {
		return MatcherTask.getInvalidFormatMessageCount();
	}

	/**
	 * @jmx.managed-attribute
	 * 
	 * 通知した件数を返す。
	 */
	@Override
	public long getNotifyCount() {
		return MatcherTask.getNotifyCount();
	}
}
