/*
Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.endpoint;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.activation.DataHandler;
import javax.annotation.Resource;
import javax.ejb.CreateException;
import javax.naming.NamingException;
import javax.xml.bind.annotation.XmlMimeType;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.fault.EventLogNotFound;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.commons.bean.ViewListInfo;
import com.clustercontrol.monitor.bean.ConfirmConstant;
import com.clustercontrol.monitor.bean.EventBatchConfirmInfo;
import com.clustercontrol.monitor.bean.EventDataInfo;
import com.clustercontrol.monitor.bean.EventFilterInfo;
import com.clustercontrol.monitor.bean.ScopeDataInfo;
import com.clustercontrol.monitor.bean.StatusDataInfo;
import com.clustercontrol.monitor.bean.StatusFilterInfo;
import com.clustercontrol.monitor.ejb.session.MonitorControllerLocal;
import com.clustercontrol.monitor.ejb.session.MonitorControllerLocalHome;
import com.clustercontrol.repository.bean.FacilityTargetConstant;

/**
 * 監視用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://monitor.ws.clustercontrol.com")
@XmlSeeAlso({
	EventDataInfo.class
})
public class MonitorEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( MonitorEndpoint.class );
	private static Log m_opelog = LogFactory.getLog("HinemosOperation");

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<String> roleList = new ArrayList<String>();
		HttpAuthenticator.authCheck(wsctx, roleList);

		return str + ", " + str;
	}

	/**
	 * 監視用のSessionBeanの取得
	 * 
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 */
	private MonitorControllerLocal getMonitorControllerLocal() throws HinemosUnknown, InvalidUserPass {
		MonitorControllerLocal local = null;
		try {
			String account = HttpAuthenticator.getAccount(wsctx);
			NamingContext namingContext = HttpAuthenticator.getContext(account);
			MonitorControllerLocalHome localHome =
				(MonitorControllerLocalHome)
				namingContext.lookup(MonitorControllerLocalHome.JNDI_NAME);
			local = localHome.create();
		} catch (CreateException e) {
			m_log.error("getMonitorControllerLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getMonitorControllerLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return local;
	}

	/**
	 * 引数で指定された条件に一致するイベント一覧情報を取得します。(クライアントview用)<BR><BR>
	 * 
	 * 引数のpropertyには、com.clustercontrol.monitor.factory.EventFilterInfoの属性が１つ以上含まれます。<BR>
	 * 各イベント情報は、EventDataInfoインスタンスとして保持されます。<BR>
	 * 戻り値のViewListInfoは、クライアントにて表示用の形式に変換されます。
	 * 
	 * MonitorResultRead権限が必要
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param filter 検索条件
	 * @param messages 表示イベント数
	 * @return ビュー一覧情報
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws MonitorNotFound
	 * 
	 * @see com.clustercontrol.monitor.bean.EventDataInfo
	 * @see com.clustercontrol.monitor.factory.SelectEvent#getEventList(String, EventFilterInfo, int)
	 */
	public ViewListInfo getEventList(String facilityId, EventFilterInfo filter, int messages) throws InvalidUserPass, InvalidRole, HinemosUnknown, MonitorNotFound {
		m_log.debug("getEventList");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_RESULT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		if(filter != null){
			SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
			msg.append(", Priority=");
			msg.append(filter.getPriority());
			msg.append(", OutputDateFrom=");
			msg.append(filter.getOutputDateFrom()==null?null:sdf.format(new Date(filter.getOutputDateFrom())));
			msg.append(", OutputDateTo=");
			msg.append(filter.getOutputDateTo()==null?null:sdf.format(new Date(filter.getOutputDateTo())));
			msg.append(", GenerationDateFrom=");
			msg.append(filter.getGenerationDateFrom()==null?null:sdf.format(new Date(filter.getGenerationDateFrom())));
			msg.append(", GenerationDateTo=");
			msg.append(filter.getGenerationDateTo()==null?null:sdf.format(new Date(filter.getGenerationDateTo())));
			msg.append(", FacilityType=");
			msg.append(filter.getFacilityType());
			msg.append(", Application=");
			msg.append(filter.getApplication());
			msg.append(", Message=");
			msg.append(filter.getMessage());
			msg.append(", ConfirmFlgType=");
			msg.append(filter.getConfirmFlgType());
			msg.append(", OutputDate=");
			msg.append(filter.getOutputDate()==null?null:sdf.format(new Date(filter.getOutputDate())));
			msg.append(", GenerationDate=");
			msg.append(filter.getGenerationDate()==null?null:sdf.format(new Date(filter.getGenerationDate())));
			msg.append(", ConfirmedUser=");
			msg.append(filter.getConfirmedUser());
			msg.append(", Comment=");
			msg.append(filter.getComment());
			msg.append(", CommentDate=");
			msg.append(filter.getCommentDate()==null?null:sdf.format(new Date(filter.getCommentDate())));
			msg.append(", CommentUser=");
			msg.append(filter.getCommentUser());
		}
		msg.append(", Count=");
		msg.append(messages);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MONITOR + " Get, Method=getEventList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getMonitorControllerLocal().getEventList(facilityId, filter, messages);
	}

	/**
	 * スコープ情報一覧を取得します。<BR><BR>
	 * 引数で指定されたファシリティの配下全てのファシリティのスコープ情報一覧を返します。<BR>
	 * 各スコープ情報は、ScopeDataInfoのインスタンスとして保持されます。<BR>
	 * 
	 * MonitorResultRead権限が必要
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param statusFlag
	 * @param eventFlag
	 * @param orderFlg
	 * @return スコープ情報一覧（ScopeDataInfoが格納されたArrayList）
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws FacilityNotFound
	 * @throws MonitorNotFound
	 * 
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<ScopeDataInfo> getScopeList(String facilityId, boolean statusFlag, boolean eventFlag, boolean orderFlg) throws InvalidUserPass, InvalidRole, HinemosUnknown, MonitorNotFound, FacilityNotFound {
		m_log.debug("getScopeList");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_RESULT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		msg.append(", StatusFlag=");
		msg.append(statusFlag);
		msg.append(", EventFlag=");
		msg.append(eventFlag);
		msg.append(", OrderFlag=");
		msg.append(orderFlg);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MONITOR + " Get, Method=getScopeList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getMonitorControllerLocal().getScopeList(facilityId, statusFlag, eventFlag, orderFlg);
	}

	/**
	 * 引数で指定された条件に一致するステータス情報一覧を取得します。<BR>
	 * 各ステータス情報は、StatusDataInfoのインスタンスとして保持されます。<BR>
	 * 
	 * MonitorResultRead権限が必要
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param property 検索条件
	 * @return ステータス情報一覧（StatusDataInfoが格納されたArrayList）
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws MonitorNotFound
	 * 
	 * @see com.clustercontrol.monitor.bean.StatusDataInfo
	 * @see com.clustercontrol.monitor.factory.SelectStatus#getStatusList(String, StatusFilterInfo)
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<StatusDataInfo> getStatusList(String facilityId, StatusFilterInfo filter) throws InvalidUserPass, InvalidRole, HinemosUnknown, MonitorNotFound
	{
		m_log.debug("getStatusList");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_RESULT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		if(filter != null){
			SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
			msg.append(", Priority=");
			msg.append(filter.getPriority());
			msg.append(", OutputDateFrom=");
			msg.append(filter.getOutputDateFrom()==null?null:sdf.format(new Date(filter.getOutputDateFrom())));
			msg.append(", OutputDateTo=");
			msg.append(filter.getOutputDateTo()==null?null:sdf.format(new Date(filter.getOutputDateTo())));
			msg.append(", GenerationDateFrom=");
			msg.append(filter.getGenerationDateFrom()==null?null:sdf.format(new Date(filter.getGenerationDateFrom())));
			msg.append(", GenerationDateTo=");
			msg.append(filter.getGenerationDateTo()==null?null:sdf.format(new Date(filter.getGenerationDateTo())));
			msg.append(", FacilityType=");
			msg.append(filter.getFacilityType());
			msg.append(", Application=");
			msg.append(filter.getApplication());
			msg.append(", Message=");
			msg.append(filter.getMessage());
			msg.append(", OutputDate=");
			msg.append(filter.getOutputDate()==null?null:sdf.format(new Date(filter.getOutputDate())));
			msg.append(", GenerationDate=");
			msg.append(filter.getGenerationDate()==null?null:sdf.format(new Date(filter.getGenerationDate())));
		}
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MONITOR + " Get, Method=getStatusList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getMonitorControllerLocal().getStatusList(facilityId, filter);
	}

	/**
	 * 引数で指定されたステータス情報を削除します。<BR>
	 * 引数のlistは、StatusDataInfoが格納されたListとして渡されます。<BR>
	 * 
	 * MonitorResultWrite権限が必要
	 * 
	 * @param list 削除対象のステータス情報一覧（StatusDataInfoが格納されたList）
	 * @return 削除に成功した場合、</code> true </code>
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws MonitorNotFound
	 * 
	 * @see com.clustercontrol.monitor.bean.StatusDataInfo
	 * @see com.clustercontrol.monitor.factory.DeleteStatus#delete(List)
	 */
	public boolean deleteStatus(ArrayList<StatusDataInfo> list) throws InvalidUserPass, InvalidRole, HinemosUnknown, MonitorNotFound {
		m_log.debug("deleteStatus");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_RESULT_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(list != null && list.size() > 0){
			StringBuffer msg = new StringBuffer();
			for (int i=0; i<list.size(); i++) {
				msg.append(", " + (i + 1) + "=(");
				msg.append("PluginID=");
				msg.append(list.get(i).getPluginId());
				msg.append(", MonitorID=");
				msg.append(list.get(i).getMonitorId());
				msg.append(", FacilityID=");
				msg.append(list.get(i).getFacilityId());
				msg.append(")");
			}

			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR + " Delete Status, Method=deleteStatus, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		return getMonitorControllerLocal().deleteStatus(list);
	}

	/**
	 * 引数で指定された条件に一致する帳票出力用イベント情報一覧を返します。<BR><BR>
	 * 
	 * MonitorResultRead権限が必要
	 * 	 *
	 * @param facilityId 取得対象の親ファシリティID
	 * @param filter 検索条件
	 * @return 帳票出力用イベント情報一覧（{@link com.clustercontrol.monitor.bean.ReportEventInfo}のリスト）
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @since 2.1.0
	 * 
	 * @see com.clustercontrol.monitor.factory.SelectEvent#getEventListForReport(String, EventFilterInfo)
	 */
	@XmlMimeType("application/octet-stream")
	public DataHandler downloadEventFile(String facilityId, EventFilterInfo filter, String filename) throws MonitorNotFound, HinemosUnknown, InvalidUserPass, InvalidRole
	{
		m_log.debug("downloadEventFile");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_RESULT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		msg.append(", FileName=");
		msg.append(filename);
		if(filter != null){
			SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
			msg.append(", Priority=");
			msg.append(filter.getPriority());
			msg.append(", OutputDateFrom=");
			msg.append(filter.getOutputDateFrom()==null?null:sdf.format(new Date(filter.getOutputDateFrom())));
			msg.append(", OutputDateTo=");
			msg.append(filter.getOutputDateTo()==null?null:sdf.format(new Date(filter.getOutputDateTo())));
			msg.append(", GenerationDateFrom=");
			msg.append(filter.getGenerationDateFrom()==null?null:sdf.format(new Date(filter.getGenerationDateFrom())));
			msg.append(", GenerationDateTo=");
			msg.append(filter.getGenerationDateTo()==null?null:sdf.format(new Date(filter.getGenerationDateTo())));
			msg.append(", FacilityType=");
			msg.append(filter.getFacilityType());
			msg.append(", Application=");
			msg.append(filter.getApplication());
			msg.append(", Message=");
			msg.append(filter.getMessage());
			msg.append(", ConfirmFlgType=");
			msg.append(filter.getConfirmFlgType());
			msg.append(", OutputDate=");
			msg.append(filter.getOutputDate()==null?null:sdf.format(new Date(filter.getOutputDate())));
			msg.append(", GenerationDate=");
			msg.append(filter.getGenerationDate()==null?null:sdf.format(new Date(filter.getGenerationDate())));
			msg.append(", ConfirmedUser=");
			msg.append(filter.getConfirmedUser());
			msg.append(", Comment=");
			msg.append(filter.getComment());
			msg.append(", CommentDate=");
			msg.append(filter.getCommentDate()==null?null:sdf.format(new Date(filter.getCommentDate())));
			msg.append(", CommentUser=");
			msg.append(filter.getCommentUser());
		}
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR + " Download, Method=downloadEventFile, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getMonitorControllerLocal().downloadEventFile(facilityId, filter, filename);
	}

	public void deleteEventFile(String filename) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("deleteEventFile");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_RESULT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FileName=");
		msg.append(filename);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR + " Delete, Method=deleteEventFile, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		getMonitorControllerLocal().deleteEventFile(filename);
	}

	/**
	 * イベント詳細情報を取得します。<BR><BR>
	 * 
	 * MonitorResultRead権限が必要
	 * 
	 * @param monitorId 取得対象の監視項目ID
	 * @param pluginId 取得対象のプラグインID
	 * @param facilityId 取得対象のファシリティID
	 * @param outputDate 取得対象の受信日時
	 * @return イベント詳細情報
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 */
	public EventDataInfo getEventInfo(String monitorId, String monitorDetailId, String pluginId, String facilityId, Long outputDate) throws MonitorNotFound, HinemosUnknown, InvalidUserPass, InvalidRole
	{
		m_log.debug("getEventInfo");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_RESULT_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
		StringBuffer msg = new StringBuffer();
		msg.append(", MonitorID=");
		msg.append(monitorId);
		msg.append(", MonitorDetailID=");
		msg.append(monitorDetailId);
		msg.append(", PluginID=");
		msg.append(pluginId);
		msg.append(", FacilityID=");
		msg.append(facilityId);
		msg.append(", OutputDate=");
		msg.append(outputDate==null?null:sdf.format(new Date(outputDate)));
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MONITOR + " Get, Method=getEventInfo, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getMonitorControllerLocal().getEventInfo(monitorId, monitorDetailId, pluginId, facilityId, outputDate);
	}

	/**
	 * 引数で指定されたイベント情報のコメントを更新します。<BR><BR>
	 * コメント追記ユーザとして、コメントユーザを設定します。
	 * 
	 * MonitorResultWrite権限が必要
	 * 
	 * @param monitorId 更新対象の監視項目ID
	 * @param pluginId 更新対象のプラグインID
	 * @param facilityId 更新対象のファシリティID
	 * @param outputDate 更新対象の受信日時ID
	 * @param confirmDate 確認済み日時（更新値）
	 * @param confirmType 確認タイプ（未／済）（更新値）
	 * @throws EventLogNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.factory.ModifyEventComment#modifyComment(String, String, String, Long, String, Long, String)
	 */
	public void modifyComment(String monitorId, String monitorDetailId, String pluginId, String facilityId, Long outputDate, String comment, Long commentDate, String commentUser)
	throws HinemosUnknown, EventLogNotFound, InvalidUserPass, InvalidSetting, InvalidRole
	{
		m_log.debug("modifyComment");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_RESULT_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
		StringBuffer msg = new StringBuffer();
		msg.append(", MonitorID=");
		msg.append(monitorId);
		msg.append(", PluginID=");
		msg.append(pluginId);
		msg.append(", FacilityID=");
		msg.append(facilityId);
		msg.append(", OutputDate=");
		msg.append(outputDate==null?null:sdf.format(new Date(outputDate)));
		msg.append(", Comment=");
		msg.append(comment);
		msg.append(", CommentDate=");
		msg.append(commentDate==null?null:sdf.format(new Date(commentDate)));
		msg.append(", CommentUser=");
		msg.append(commentUser);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR + " Change, Method=modifyComment, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		getMonitorControllerLocal().modifyComment(monitorId, monitorDetailId, pluginId, facilityId, outputDate, comment, commentDate, commentUser);
	}


	/**
	 * 引数で指定されたイベント情報一覧の確認を更新します。<BR><BR>
	 * 確認ユーザとして、操作を実施したユーザを設定します。<BR>
	 * 複数のイベント情報を更新します。
	 * 
	 * MonitorResultWrite権限が必要
	 * 
	 * @param list 更新対象のイベント情報一覧（EventDataInfoが格納されたArrayList）
	 * @param confirmType 確認タイプ（未／済）（更新値）
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws MonitorNotFound
	 * 
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.factory.ModifyEventConfirm#modifyConfirm(List, int)
	 */
	public void modifyConfirm(ArrayList<EventDataInfo> list, int confirmType) throws InvalidUserPass, InvalidRole, HinemosUnknown, MonitorNotFound {
		m_log.debug("modifyConfirmMultiple");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_RESULT_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(list != null && list.size() > 0){
			SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
			String confirmTypeStr = "";
			StringBuffer msg = new StringBuffer();
			if(confirmType == ConfirmConstant.TYPE_CONFIRMED) {
				confirmTypeStr = "Confirmed";
			} else {
				confirmTypeStr = "Unonfirmed";
			}

			for (int i=0; i<list.size(); i++) {
				msg.append(", " + (i + 1) + "=(");
				msg.append("PluginID=");
				msg.append(list.get(i).getPluginId());
				msg.append(", FacilityID=");
				msg.append(list.get(i).getFacilityId());
				msg.append(", Priority=");
				msg.append((list.get(i).getPriority()==null?null:PriorityConstant.typeToStringEN(list.get(i).getPriority())));
				msg.append(", Time Received=");
				msg.append((list.get(i).getOutputDate()==null?null:sdf.format(new Date(list.get(i).getOutputDate()))));
				msg.append(", Time Created=");
				msg.append((list.get(i).getGenerationDate()==null?null:sdf.format(new Date(list.get(i).getGenerationDate()))));
				msg.append(", Comfirm=");
				msg.append(confirmTypeStr);
				msg.append(")");
			}

			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR + " Confirm, Method=modifyConfirm, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		getMonitorControllerLocal().modifyConfirm(list, confirmType);
	}

	/**
	 * 引数で指定された条件に一致するイベント情報の確認を一括更新します。<BR><BR>
	 * 確認ユーザとして、操作を実施したユーザを設定します。<BR>
	 * 
	 * MonitorResultWrite権限が必要
	 * 
	 * @param confirmType 確認タイプ（未／済）（更新値）
	 * @param facilityId 更新対象の親ファシリティID
	 * @param property 更新条件
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.bean.ConfirmConstant
	 * @see com.clustercontrol.monitor.factory.ModifyEventConfirm#modifyBatchConfirm(int, String, EventBatchConfirmInfo)
	 */
	public void modifyBatchConfirm(int confirmType, String facilityId, EventBatchConfirmInfo info) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("modifyBatchConfirm");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_RESULT_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", FacilityID=");
			msg.append(facilityId);
			msg.append(", Priority=");
			msg.append(info.getPriority()==null?null:PriorityConstant.typeToStringEN(info.getPriority()));
			msg.append(", Target Facility=");
			if(FacilityTargetConstant.STRING_BENEATH.equals(info.getFacilityType())) {
				msg.append("Sub-scope Facilities Only");
			} else {
				msg.append("ALL Facilities");
			}
			msg.append(", Application=");
			msg.append(info.getApplication());
			msg.append(", Message=");
			msg.append(info.getMessage());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR + " Confirm All, Method=modifyBatchConfirm, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		getMonitorControllerLocal().modifyBatchConfirm(confirmType, facilityId, info);
	}
}