/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.aws.presenter;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.net.URL;
import java.util.List;

import javax.naming.NamingException;
import javax.xml.namespace.QName;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.WebServiceException;
import javax.xml.ws.soap.SOAPBinding;

import org.apache.log4j.Logger;
import org.eclipse.jface.preference.IPreferenceStore;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.util.EndpointManager.EndpointSetting;
import com.clustercontrol.util.LoginManager;
import com.clustercontrol.ws.access.AccessEndpoint;
import com.clustercontrol.ws.cloud.CloudEndpoint;
import com.clustercontrol.ws.cloud.CloudEndpointService;
import com.clustercontrol.ws.jobmanagement.JobEndpoint;
import com.clustercontrol.ws.repository.RepositoryEndpoint;


public class EndpointManager implements IEndpointManager {
	// スレッドローカルとして、エンドポイントを作成。
	// VM オプションでは、都度作成しているように見受けられた。
	// 都度作成すると、開発環境では、0.5 秒ほど時間がかかるので、スレッドローカルな情報としてエンドポイントを使いまわす。
	private static final ThreadLocal<CloudEndpoint> cloudEndpoint = 
        new ThreadLocal<CloudEndpoint>() {
            @Override
            protected CloudEndpoint initialValue() {
    			InvocationHandler h = new InvocationHandler() {
    				private String currentUserId;
    				private String currentPassword;
    				private String currentUrl;

    				private CloudEndpoint endpoint;
    				
    				@Override
    				public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
    					// ログアウトしたタイミングがわからないので、アクセスする前に、前回のログイン情報と同じか確認する。
    					LoginManager loginManager = LoginManager.getContextManager();
    					try {
    						loginManager.getNamingContext();
    					}
    					catch (NamingException e) {
    						throw new IllegalStateException(e);
    					}

    					// 作成済みのエンドポイントが存在するか確認。
    					if (endpoint == null) {
    						//　エンドポイントが未作成なので、新規に作成。
	    					String userId = loginManager.getUserId();
	    					String password = loginManager.getPassword();
	    					String url = LoginManager.getUrl();
    						
	    					endpoint = createCloudEndpoint(userId, password, url);

	    					currentUserId = userId;
	    					currentPassword = password;
	    					currentUrl = url;
    					}
    					else {
	    					String userId = loginManager.getUserId();
	    					String password = loginManager.getPassword();
	    					String url = LoginManager.getUrl();

    						//　過去に作成したエンドポイントのログイン情報と、現在のログイン情報が一致するか確認。
	    					if (!(currentUserId.equals(userId) && currentPassword.equals(password) && currentUrl.equals(url))) {
	    						// 一致しないので新規に作成。
		    					endpoint = createCloudEndpoint(userId, password, url);
		    					currentUserId = userId;
		    					currentPassword = password;
		    					currentUrl = url;
	    					}
    					}
    					
    					try {
							return method.invoke(endpoint, args);
    					}
						catch (InvocationTargetException e) {
							throw e.getCause();
    					}
    				}
    			};
    			return (CloudEndpoint)Proxy.newProxyInstance(this.getClass().getClassLoader(), new Class<?>[]{CloudEndpoint.class}, h);
            }
    	};
	
	private List<EndpointSetting> getEndpointSetting(Class<?> clazz) {
		if (AccessEndpoint.class == clazz) {
			return com.clustercontrol.util.EndpointManager.getAccessEndpoint();
		}
		else if (RepositoryEndpoint.class == clazz) {
			return com.clustercontrol.util.EndpointManager.getRepositoryEndpoint();
		}
		else if (JobEndpoint.class == clazz) {
			return com.clustercontrol.util.EndpointManager.getJobEndpoint();
		}
		
		throw new UnsupportedOperationException(); 
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public synchronized <T> T getEndpoint(final Class<T> clazz) {
		T endpoint = null;
		
		if (clazz == CloudEndpoint.class) {
			endpoint = (T)cloudEndpoint.get();
		}
		else {
			InvocationHandler h = new InvocationHandler() {
				@Override
				public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
					// 以下の処理は、Hinemos クライアントから引用。
					// 以下の作りこみは、getEndpoint を クラウド クライアントから、エンドポイントを取得する統一的な方法にするため。
					// 基本的には、Hinemos クライアントで、エンドポイントを取得する方法と同じになるはず。
					WebServiceException wse = null;
					for (EndpointSetting endpointSetting : getEndpointSetting(clazz)) {
						try {
							Object endpoint = endpointSetting.getEndpoint();
							return method.invoke(endpoint, args);
						}
						catch (InvocationTargetException e) {
							if (!(e.getCause() instanceof WebServiceException)) {
								throw e.getCause();
							}
							
							Logger logger = Logger.getLogger(EndpointManager.class);
							logger.warn("calling " + method.getName(), e.getCause());

							com.clustercontrol.util.EndpointManager.changeEndpoint();

							wse = (WebServiceException)e.getCause();
						}
					}
					throw wse;
				}
			};
			endpoint = (T)Proxy.newProxyInstance(this.getClass().getClassLoader(), new Class<?>[]{clazz}, h);
		}

		return endpoint;
	}

	private static CloudEndpoint createCloudEndpoint(String userId, String password, String url) {
		CloudEndpoint endpoint = null;

		try {
			String urlCloudEndpoint = LoginManager.getUrl() + CloudEndpointService.class.getSimpleName();

			CloudEndpointService service = new CloudEndpointService(new URL(urlCloudEndpoint + "?wsdl"),
				new QName("http://cloud.ws.clustercontrol.com", CloudEndpointService.class.getSimpleName()));
			
			endpoint = service.getCloudEndpointPort();

			BindingProvider bp = (BindingProvider)endpoint;
			bp.getRequestContext().put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, urlCloudEndpoint);
			bp.getRequestContext().put(BindingProvider.USERNAME_PROPERTY, userId);
			bp.getRequestContext().put(BindingProvider.PASSWORD_PROPERTY, password);

			IPreferenceStore store = ClusterControlPlugin.getDefault().getPreferenceStore();
			int httpConnectTimeout = store.getInt(LoginManager.KEY_HTTP_CONNECT_TIMEOUT);
			int httpRequestTimeout = store.getInt(LoginManager.KEY_HTTP_REQUEST_TIMEOUT);
			
			bp.getRequestContext().put("com.sun.xml.internal.ws.connect.timeout", httpConnectTimeout * 1000);
			bp.getRequestContext().put("com.sun.xml.internal.ws.request.timeout", httpRequestTimeout * 1000);

			((SOAPBinding)bp.getBinding()).setMTOMEnabled(true);
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}

		return endpoint;
	}
	
	@Override
	public String getAccountName() {
		LoginManager loginManager = LoginManager.getContextManager();
		return loginManager.getUserId();
	}
}