/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.aws.presenter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;

import com.clustercontrol.action.FacilityTree;
import com.clustercontrol.cloud.commons.util.MessageManager;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.ws.repository.FacilityInfo;
import com.clustercontrol.ws.repository.FacilityTreeItem;

public class FacilityRoot extends Element implements IFacilityRoot {
	private MessageManager bundle_messages = MessageManager.getInstance("messages");
	
	private HinemosService hinemosService;
	private RootScope rootScope;

	public FacilityRoot(HinemosService hinemosService) {
		this.hinemosService = hinemosService;

		try {
			FacilityTreeItem fti = new FacilityTree().getTreeItem();
			
			List<FacilityTreeItem> roots = collectTopScope(fti);
			for (FacilityTreeItem root: roots) {
				if ("AmazonWebService".equals(root.getData().getFacilityName())) {
					rootScope = new RootScope(this);
					
					rootScope.setName(root.getData().getFacilityName());
					rootScope.setFacilityId(root.getData().getFacilityId());
					rootScope.setScopeType(IScope.ScopeType.root);
					
					convertChildren(rootScope, root);
					break;
				}
			}
			
			if (rootScope == null) {
				throw new CloudModelException(bundle_messages.getString("model.warning.not_found_root_scope"), ErrorCodeConstants.FACILITYROOT_INVALID_ROOT_SCOPE_NOT_FOUND);
			}
		}
		catch (Exception e) {
			throw new CloudModelException("faild to get repositry tree.", e);
		}
	}

	@Override
	public void update() throws CloudModelException {
		FacilityTreeItem fti = CacheService.getSingleton().get(this, FacilityTreeItem.class);
		if (fti == null) {
			try {
				fti = new FacilityTree().getTreeItem();
			}
			catch (Exception e) {
				Logger logger = Logger.getLogger(this.getClass());
				logger.error("faild to get repositry tree.", e);
			}
		}
		else {
			CacheService.getSingleton().remove(this);
		}
		
		try {
			List<FacilityTreeItem> roots = collectTopScope(fti);
			FacilityTreeItem awsRoot = null;
			for (FacilityTreeItem root: roots) {
				if ("AmazonWebService".equals(root.getData().getFacilityName())) {
					awsRoot = root;
					break;
				}
			}

			if (awsRoot == null) {
				throw new CloudModelException(bundle_messages.getString("model.warning.not_found_root_scope"), ErrorCodeConstants.FACILITYROOT_INVALID_ROOT_SCOPE_NOT_FOUND);
			}
			
			List<Scope> regionScopes = new ArrayList<Scope>(Arrays.asList(rootScope.getScopes()));
			List<FacilityTreeItem> treeItems = new ArrayList<FacilityTreeItem>(awsRoot.getChildren());
			
			Iterator<Scope> regionScopesIter = regionScopes.iterator();
			while (regionScopesIter.hasNext()) {
				Scope regionScope = regionScopesIter.next();
				
				Iterator<FacilityTreeItem> treeItemIter = treeItems.iterator();
				while (treeItemIter.hasNext()) {
					FacilityTreeItem treeItem = treeItemIter.next();
					FacilityInfo fi = treeItem.getData();

					if (fi.getFacilityType() == FacilityConstant.TYPE_SCOPE) {
						if (fi.getFacilityId().equals(regionScope.getFacilityId())) {
							// 名前の更新。
							regionScope.setName(fi.getFacilityName());

							List<Scope> zoneScopes = new ArrayList<Scope>(Arrays.asList(regionScope.getScopes()));
							List<FacilityTreeItem> treeItems2 = new ArrayList<FacilityTreeItem>(treeItem.getChildren());

							// アベイラビリティゾーンについてチェック。
							Iterator<Scope> zoneScopesIter = zoneScopes.iterator();
							while (zoneScopesIter.hasNext()) {
								Scope zoneScope = zoneScopesIter.next();
								
								Iterator<FacilityTreeItem> treeItemIter2 = treeItems2.iterator();
								while (treeItemIter2.hasNext()) {
									FacilityTreeItem treeItem2 = treeItemIter2.next();
									FacilityInfo fi2 = treeItem2.getData();

									if (fi2.getFacilityType() == FacilityConstant.TYPE_SCOPE) {
										if (fi2.getFacilityId().equals(zoneScope.getFacilityId())) {
											// 名前の更新。
											zoneScope.setName(fi2.getFacilityName());

											List<Node> nodes = new ArrayList<Node>(Arrays.asList(zoneScope.getNodes()));
											List<FacilityTreeItem> treeItems3 = new ArrayList<FacilityTreeItem>(treeItem2.getChildren());

											// ノードの構成についてチェック。
											Iterator<Node> nodesIter = nodes.iterator();
											while (nodesIter.hasNext()) {
												Node node = nodesIter.next();
												
												Iterator<FacilityTreeItem> treeItemIter3 = treeItems3.iterator();
												while (treeItemIter3.hasNext()) {
													FacilityTreeItem treeItem3 = treeItemIter3.next();
													FacilityInfo fi3 = treeItem3.getData();
											
													if (fi3.getFacilityType() == FacilityConstant.TYPE_NODE) {
														if (fi3.getFacilityId().equals(node.getFacilityId())) {
															node.setName(fi3.getFacilityName());

															nodesIter.remove();
															treeItemIter3.remove();
														}
													}
													else {
														// ノードでない要素は捨てる。
														treeItemIter3.remove();
													}
												}
											}
											
											// ノードの追加、削除分を調整。
											for (Node node: nodes) {
												zoneScope.removeChildNode(node);
											}
											for (FacilityTreeItem treeItem3: treeItems3) {
												convertChild(zoneScope, treeItem3);
											}
											
											zoneScopesIter.remove();
											treeItemIter2.remove();
										}
									}
									else {
										// スコープでない要素は捨てる。
										treeItemIter2.remove();
									}
								}
							}

							
							// アベイラビリティゾーンの追加、削除分を調整。
							for (Scope zone: zoneScopes) {
								regionScope.removeChildScope(zone);
							}
							for (FacilityTreeItem treeItem2: treeItems2) {
								convertChild(regionScope, treeItem2);
							}

							regionScopesIter.remove();
							treeItemIter.remove();

							break;
						}
					}
					else {
						// スコープでない要素は捨てる。
						treeItemIter.remove();
					}
				}
			}
			
			// リージョンの追加、削除分を調整。
			for (Scope region: regionScopes) {
				rootScope.removeChildScope(region);
			}
			for (FacilityTreeItem treeItem: treeItems) {
				convertChild(rootScope, treeItem);
			}
		}
		catch (Exception e) {
			Logger logger = Logger.getLogger(this.getClass());
			logger.error("faild to get repositry tree.", e);
		}
		
		super.update();
	}
	
	@Override
	public Scope[] getScopes() {
		if (rootScope != null) {
			return new Scope[]{rootScope};
		}
		else {
			return new Scope[0];
		}
	}

	public CloudResourceManager getCloudResourceManager() {
		return getHinemosService().getCloudResourceManager();
	}

	@Override
	public HinemosService getHinemosService() {
		return hinemosService;
	}

	/**
	 * FacilityInfo に該当する LNode を新規作成する。
	 * 
	 * @param fi
	 * @return
	 */
	private static Node convertNode(Scope parent, FacilityTreeItem fti) {
		Node node = new Node();
		node.setParent(parent);
		node.setName(fti.getData().getFacilityName());
		node.setFacilityId(fti.getData().getFacilityId());

		return node;
	}

	/**
	 * Hinemos のツリー情報を元に LScope を新規作成する。
	 * 
	 * @param fti
	 * @return
	 */
	private Scope convertScope(Scope parent, IScope.ScopeType type, FacilityTreeItem fti) {
		try {
			FacilityInfo fi = fti.getData();
			assert fi.getFacilityType() == FacilityConstant.TYPE_SCOPE: "unexpected";
			
			Scope scope = new Scope();
			scope.setParent(parent);
			scope.setName(fi.getFacilityName());
			scope.setFacilityId(fti.getData().getFacilityId());
			scope.setScopeType(type);
	
			return scope;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	private void convertChildren(Scope scope, FacilityTreeItem fti) {
		for (FacilityTreeItem child: fti.getChildren()) {
			convertChild(scope, child);
		}
	}
	
	private void convertChild(Scope parent, FacilityTreeItem child) {
		switch(child.getData().getFacilityType()) {
		case FacilityConstant.TYPE_SCOPE:
			IScope.ScopeType nextType = null;
			switch (parent.getScopeType()) {
			case root:
				nextType = IScope.ScopeType.region;
				break;
			case region:
				nextType = IScope.ScopeType.availabilityZone;
				break;
			}
			
			if (nextType != null) {
				Scope childScope = convertScope(parent, nextType, child);
				parent.addChildScope(childScope);
				convertChildren(childScope, child);
			}
			break;
		case FacilityConstant.TYPE_NODE:
			// アベイラビリティーゾーンのスコープでない場合は、スキップ
			if (parent.getScopeType() == IScope.ScopeType.availabilityZone) {
				Node node = convertNode(parent, child);
				parent.addChildNode(node);
			}
			break;
		}
	}

	/**
	 * 指定した Facility が、Hinemos 内部のスコープか確認。
	 * 
	 * @param facility
	 * @return
	 */
	private static boolean isInternalScope(FacilityInfo facility) {
		if (
			facility.getFacilityType() == FacilityConstant.TYPE_SCOPE &&
			(FacilityTreeAttributeConstant.REGISTEREFD_SCOPE.equals(facility.getFacilityId()) ||
			FacilityTreeAttributeConstant.INTERNAL_SCOPE.equals(facility.getFacilityId()) ||
			FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(facility.getFacilityId()))
			) {
			return true;
		}
		return false;
	}
	
	/**
	 * Hinemos のモデルから、Top レベルのスコープ情報を検出する。
	 * 
	 * @param treeItem
	 * @param buf
	 * @return
	 */
	private List<FacilityTreeItem> collectTopScope(FacilityTreeItem treeItem) {
		List<FacilityTreeItem> buf = new ArrayList<FacilityTreeItem>();
		return recursiveCollectTopScope(treeItem, buf);
	}

	private List<FacilityTreeItem> recursiveCollectTopScope(FacilityTreeItem treeItem, List<FacilityTreeItem> buf) {
		if (!isScope(treeItem.getData())) {
			for (FacilityTreeItem fti: treeItem.getChildren()) {
				recursiveCollectTopScope(fti, buf);
			}
		}
		else {
			buf.add(treeItem);
		}
		
		return buf;
	}
	
	/**
	 * 指定した FacilityInfo が、有効なスコープ情報か確認。
	 * 
	 * @param treeItem
	 * @param buf
	 * @return
	 */
	private boolean isScope(FacilityInfo facility) {
		switch(facility.getFacilityType()) {
		case FacilityConstant.TYPE_SCOPE:
			// Hinemos のインターナルのスコープの場合、対象外。
			if (!isInternalScope(facility)) {
				return true;
			}
			break;
		}
		
		return false;
	}
}
