/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.aws.presenter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import com.clustercontrol.ws.cloud.CloudEndpoint;
import com.clustercontrol.ws.cloud.CloudStorage;

public class StorageRoot extends Element implements IStorageRoot {
	private Region region;
	private List<Storage> storages;
	
	public StorageRoot(Region region) {
		this.region = region;
	}

	@Override
	public void update() throws CloudModelException {
		CloudStorage[] cloudStorageArray = CacheService.getSingleton().get(this, CloudStorage[].class);

		List<CloudStorage> cloudStorages = Collections.emptyList();
		if (cloudStorageArray == null) {
			try {
				CloudEndpoint endpoint = getCloudResourceManager().getEndpoint(CloudEndpoint.class);
				cloudStorages = endpoint.getStoragesByRegion(getRegion().getRegionName());
			}
			catch (Exception e) {
				throw new CloudModelException(e);
			}
		}
		else {
			cloudStorages = Arrays.asList(cloudStorageArray);
			CacheService.getSingleton().remove(this);
		}

		internalUpdate(cloudStorages);

		super.update();
	}

	public void internalUpdate(List<CloudStorage> cloudStorages) throws CloudModelException {
		if (isInitialized()) {
			List<Storage> storagesTemp = new ArrayList<Storage>(storages);
			List<CloudStorage> cloudStoragesTemp = new ArrayList<CloudStorage>(cloudStorages);
			Iterator<CloudStorage> cloudStoragesIter = cloudStoragesTemp.iterator();
			while (cloudStoragesIter.hasNext()) {
				CloudStorage cs = cloudStoragesIter.next();
				
				Iterator<Storage> storagesIter = storagesTemp.iterator();
				while (storagesIter.hasNext()) {
					Storage s = storagesIter.next();
					
					if (cs.getRegion().equals(s.getRegion().getRegionName()) && cs.getStorageId().equals(s.getStorageId())) {
						s.internalUpdate(cs);

						storagesIter.remove();
						cloudStoragesIter.remove();
						break;
					}
				}
			}

			for (Storage s: storagesTemp) {
				getInstancesList().remove(s);
				fireElementRemoved(p2.storages, s);
			}

			for (CloudStorage cs: cloudStoragesTemp) {
				Storage newInstance = Storage.createStorage(this, cs);
				getInstancesList().add(newInstance);
				this.fireElementAdded(p2.storages, newInstance);
			}
		}
		else {
			List<Storage> tempStorages = new ArrayList<Storage>();
			for (CloudStorage instance: cloudStorages) {
				tempStorages.add(Storage.createStorage(this, instance));
			}
			storages = tempStorages;
		}
	}

	@Override
	public Storage[] getStorages() {
		return getInstancesList().toArray(new Storage[0]);
	}

	@Override
	public CloudResourceManager getCloudResourceManager() {
		return region.getCloudResourceManager();
	}

	@Override
	public Region getRegion() {
		return region;
	}

	@Override
	public Storage createStorage(CreateStorageRequest request) {
		// 未初期化の場合、とりあえず初期化。
		if (!isInitialized()) {
			getInstancesList();
		}
		com.clustercontrol.ws.cloud.CreateStorageRequest csr = new com.clustercontrol.ws.cloud.CreateStorageRequest();
		csr.setRegion(getRegion().getRegionName());
		csr.setAvailabilityZone(request.getAvailabilityZone());
		csr.setSnapshotId(request.getSnapshotId());
		csr.setStorageName(request.getStorageName());
		csr.setStorageSize(request.getStorageSize());
		csr.setStorageKind(request.getStorageKind());
		csr.setIops(request.getIops());
		
		Storage newStorage = null;
		try {
			CloudEndpoint endpoint = getCloudResourceManager().getEndpoint(CloudEndpoint.class);
			CloudStorage cloudStorage = endpoint.addStorage(csr);
			newStorage = Storage.createStorage(this, cloudStorage);
			storages.add(newStorage);
			
			this.fireElementAdded(IStorageRoot.p2.storages, newStorage);
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
		
		return newStorage;
	}

	@Override
	public 	IStorage restore(CreateStorageRequest request) {
		// 未初期化の場合、とりあえず初期化。
		if (!isInitialized()) {
			getInstancesList();
		}
		com.clustercontrol.ws.cloud.CreateStorageRequest csr = new com.clustercontrol.ws.cloud.CreateStorageRequest();
		csr.setRegion(getRegion().getRegionName());
		csr.setSnapshotId(request.getSnapshotId());
		csr.setAvailabilityZone(request.getAvailabilityZone());
		csr.setStorageName(request.getStorageName());
		csr.setStorageSize(request.getStorageSize());
		csr.setStorageKind(request.getStorageKind());
		csr.setIops(request.getIops());
		
		Storage newStorage = null;
		try {
			CloudEndpoint endpoint = getCloudResourceManager().getEndpoint(CloudEndpoint.class);
			CloudStorage cloudStorage = endpoint.restoreStorage(csr);
			newStorage = Storage.createStorage(this, cloudStorage);
			storages.add(newStorage);
			
			this.fireElementAdded(IStorageRoot.p2.storages, newStorage);
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
		
		return newStorage;
	}
	
	@Override
	public void deleteStorage(String storageId) {
		// 未初期化の場合、とりあえず初期化。
		if (!isInitialized()) {
			getInstancesList();
		}
		// Web サービス経由で新規追加。
		CloudEndpoint ce = getCloudResourceManager().getEndpoint(CloudEndpoint.class);

		try {
			ce.removeStorage(getRegion().getRegionName(), storageId);

			for (IStorage storage: storages) {
				if (storage.getStorageId().equals(storageId)) {
					storages.remove(storage);
					this.fireElementRemoved(IStorageRoot.p2.storages, storage);
					break;
				}
			}
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
	}

	public boolean isInitialized() {
		return storages != null;
	}
	
	private List<Storage> getInstancesList() {
		if (!isInitialized()) {
			update();
		}
		return storages;
	}
}
