/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.aws.presenter;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.clustercontrol.ws.cloud.CloudEndpoint;
import com.clustercontrol.ws.cloud.CloudTemplate;


public class TemplateManagement extends Element implements ITemplateManagement {

	private Inventory inventory;
	private static List<Template> templates;
	
	public TemplateManagement(Inventory inventory){
		this.inventory = inventory;
	}

	@Override
	public CloudResourceManager getCloudResourceManager() {
		return inventory.getCloudResourceManager();
	}

	public Inventory getInventory() {
		return inventory;
	}

	@Override
	public ITemplate[] getTemplates() {
		if (templates == null) {
			templates = new ArrayList<Template>();

			CloudEndpoint endpoint = getCloudResourceManager().getEndpoint(CloudEndpoint.class);
			try {
				List<CloudTemplate> cloudTemplates = endpoint.getAllTemplates();
				for (CloudTemplate cloudTemplate: cloudTemplates) {
					Template template = new Template(this, cloudTemplate);
					templates.add(template);
				}
			}
			catch (Exception e) {
				throw new CloudModelException(e);
			}
		}
		
		return templates.toArray(new Template[0]);
	}
	
	@Override
	public void update() {
		try {
			CloudEndpoint endpoint = getCloudResourceManager().getEndpoint(CloudEndpoint.class);
			List<CloudTemplate> cloudTemplates = endpoint.getAllTemplates();
			internalUpdate(cloudTemplates);
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
	}

	public void internalUpdate(List<CloudTemplate> cloudTemplates) throws CloudModelException {
		List<Template> tmpTemplates = new ArrayList<Template>(templates);
		List<CloudTemplate> tmpCloudTemplates = new ArrayList<CloudTemplate>(cloudTemplates);
		Iterator<CloudTemplate> itrCloudTemplates = tmpCloudTemplates.iterator();
		while (itrCloudTemplates.hasNext()) {
			CloudTemplate cloudTemplate = itrCloudTemplates.next();
			
			Iterator<Template> itrTemplates = tmpTemplates.iterator();
			while (itrTemplates.hasNext()) {
				Template template = itrTemplates.next();
				
				if (cloudTemplate.getTemplateId().equals(template.getTemplateId())) {
					template.internalUpdate(cloudTemplate);

					itrTemplates.remove();
					itrCloudTemplates.remove();
					break;
				}
			}
		}

		for (Template template: tmpTemplates) {
			templates.remove(template);
			fireElementRemoved(p2.templates, template);
		}

		for (CloudTemplate cloudTemplate: tmpCloudTemplates) {
			Template template = new Template(this, cloudTemplate);
			templates.add(template);
			this.fireElementAdded(p2.templates, template);
		}
	}
	
	@Override
	public Template createTemplate(CreateTemplateRequest request){
		Template template = null;
		try {
			CloudTemplate cloudTemplate = new CloudTemplate();
			cloudTemplate.setTemplateId(request.getTemplateId());
			cloudTemplate.setTemplateName(request.getTemplateName());
			cloudTemplate.setImageId(request.getImageId());
			cloudTemplate.setRegion(request.getRegion());
			cloudTemplate.setLaunchJobId(request.getLaunchJobId());
			cloudTemplate.setStartJobId(request.getStartJobId());
			cloudTemplate.setStopJobId(request.getStopJobId());
			
			CloudEndpoint endpoint = getCloudResourceManager().getEndpoint(CloudEndpoint.class);
			CloudTemplate addedTemplate = endpoint.addTemplate(cloudTemplate);

			template = new Template(this, addedTemplate);
			
			templates.add(template);
			fireElementAdded(p2.templates, template);
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
		
		return template;
	}

	@Override
	public Template modifyTemplate(ITemplate argTemplate, CreateTemplateRequest request){
		Template template = (Template) argTemplate;
		try {
			CloudTemplate cloudTemplate = template.getTemplate();
			cloudTemplate.setTemplateName(request.getTemplateName());
			cloudTemplate.setImageId(request.getImageId());
			cloudTemplate.setRegion(request.getRegion());
			cloudTemplate.setLaunchJobId(request.getLaunchJobId());
			cloudTemplate.setStartJobId(request.getStartJobId());
			cloudTemplate.setStopJobId(request.getStopJobId());
			
			CloudEndpoint endpoint = getCloudResourceManager().getEndpoint(CloudEndpoint.class);
			endpoint.modifyTemplate(cloudTemplate);

			CloudTemplate modifiedTemplate = endpoint.getTemplate(template.getTemplateId());
			
			template.internalUpdate(modifiedTemplate);
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
		
		return template;
	}

	@Override
	public void deleteTemplate(String templateId){
		try {
			CloudEndpoint endpoint = getCloudResourceManager().getEndpoint(CloudEndpoint.class);
			endpoint.removeTemplate(templateId);

			for (Template template: templates) {
				if (templateId.equals(template.getTemplateId())) {
					templates.remove(template);
					fireElementRemoved(p2.templates, template);
					break;
				}
			}
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
	}

	@Override
	public ITemplate getTemplate(String templateId) {
		for(Template template: templates){
			if(template.getTemplateId().equals(templateId)){
				return template;
			}
		}

		CloudEndpoint endpoint = getCloudResourceManager().getEndpoint(CloudEndpoint.class);
		try {
			CloudTemplate cloudTemplate = endpoint.getTemplate(templateId);
			if(cloudTemplate != null){

				Template template = new Template(this, cloudTemplate);
				
				templates.add(template);
				fireElementAdded(p2.templates, template);
				
				return template;
			}
		} catch (Exception e) {
			throw new CloudModelException(e);
		}
		
		return null;
	}
}