/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.aws.resources;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.clustercontrol.ws.cloud.CloudEndpoint;
import com.clustercontrol.ws.cloud.KeyPair;
import com.clustercontrol.ws.cloud.Region;

public class CloudResourceProviderImpl implements CloudResourceProvider {
	private CloudEndpoint endpoint;

	public CloudResourceProviderImpl(CloudEndpoint endpoint) {
		this.endpoint = endpoint;
	}
	
	@Override
	public List<String> getAllRegion() {
		try {
			List<String> regions = new ArrayList<String>();
			for (Region r: endpoint.getAllRegion()) {
				regions.add(r.getRegionName());
			}
			return regions;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<AvailabilityZone> getAvailabilityZones(String region) {
		try {
			List<AvailabilityZone> azs = new ArrayList<AvailabilityZone>();
			for (com.clustercontrol.ws.cloud.AvailabilityZone wsAZ: endpoint.getAvailabilityZones(region)) {
				AvailabilityZone az = new AvailabilityZone();
				az.setZoneName(wsAZ.getZoneName());
				azs.add(az);
			}
			return azs;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<Image> getImages(String region, List<Filter> filters) {
		try {
			List<com.clustercontrol.ws.cloud.Filter> wsFilters = new ArrayList<com.clustercontrol.ws.cloud.Filter>();
			for (Filter filter: filters) {
				com.clustercontrol.ws.cloud.Filter wsFilter = new com.clustercontrol.ws.cloud.Filter();
				wsFilter.setName(filter.getName());
				wsFilter.getValues().addAll(filter.getValues());
				wsFilters.add(wsFilter);
			}

			List<Image> images = new ArrayList<Image>();
			for (com.clustercontrol.ws.cloud.Image wsImage: endpoint.getImages(region, wsFilters)) {
				Image image = new Image();
				image.setImageId(wsImage.getImageId());
				image.setName(wsImage.getName());
				images.add(image);
			}
			return images;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<GroupIdentifier> getSecurityGroupsByRegion(String regionName) {
		List<Filter> filters = Collections.emptyList();
		return getSecurityGroups(regionName, filters);
	}
	
	@Override
	public List<GroupIdentifier> getSecurityGroups(String regionName, List<Filter> filters) {
		try {
			List<com.clustercontrol.ws.cloud.SecurityGroup> wsSGs = Collections.emptyList();
			if (!filters.isEmpty()) {
				List<com.clustercontrol.ws.cloud.Filter> wsFilters = new ArrayList<com.clustercontrol.ws.cloud.Filter>();
				for (Filter filter: filters) {
					com.clustercontrol.ws.cloud.Filter wsFilter = new com.clustercontrol.ws.cloud.Filter();
					wsFilter.setName(filter.getName());
					wsFilter.getValues().addAll(filter.getValues());
					wsFilters.add(wsFilter);
				}
				wsSGs = endpoint.getSecurityGroups(regionName, wsFilters);
			}
			else {
				wsSGs = endpoint.getSecurityGroupsByRegion(regionName);
			}

			List<GroupIdentifier> sgs = new ArrayList<GroupIdentifier>();
			for (com.clustercontrol.ws.cloud.SecurityGroup wsSG: wsSGs) {
				GroupIdentifier gi = new GroupIdentifier();
				gi.setGroupId(wsSG.getGroupId());
				gi.setGroupName(wsSG.getGroupName());
				gi.setVpcId(wsSG.getVpcId());
				sgs.add(gi);
			}
			return sgs;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<String> getAllInstanceType() {
		try {
			return endpoint.getAllInstanceType();
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<String> getKeyNames(String region) {
		try {
			List<String> ketNames = new ArrayList<String>();
			for (KeyPair k: endpoint.getKeyNames(region)) {
				ketNames.add(k.getKeyName());
			}
			return ketNames;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<Snapshot> getSnapshots(String region, List<Filter> filters) {
		try {
			List<com.clustercontrol.ws.cloud.Filter> wsFilters = new ArrayList<com.clustercontrol.ws.cloud.Filter>();
			for (Filter filter: filters) {
				com.clustercontrol.ws.cloud.Filter wsFilter = new com.clustercontrol.ws.cloud.Filter();
				wsFilter.setName(filter.getName());
				wsFilter.getValues().addAll(filter.getValues());
				wsFilters.add(wsFilter);
			}

			List<Snapshot> snapshots = new ArrayList<Snapshot>();
			for (com.clustercontrol.ws.cloud.Snapshot ws: endpoint.getSnapshots(region, wsFilters)) {
				Snapshot s = new Snapshot();
				s.setSnapshotId(ws.getSnapshotId());
				s.setDescription(ws.getDescription());
				s.setRegion(ws.getRegion());
				snapshots.add(s);
			}
			return snapshots;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<String> getAllVolumeType() {
		try {
			List<String> volumeTypes = new ArrayList<String>();
			for (String v: endpoint.getAllVolumeType()) {
				volumeTypes.add(v);
			}
			return volumeTypes;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<String> getAllServiceName() {
		try {
			return endpoint.getAllServiceName();
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<Subnet> getSubnets(String regionName, List<Filter> filters) {
		try {
			List<com.clustercontrol.ws.cloud.Filter> wsFilters = new ArrayList<com.clustercontrol.ws.cloud.Filter>();
			for (Filter filter: filters) {
				com.clustercontrol.ws.cloud.Filter wsFilter = new com.clustercontrol.ws.cloud.Filter();
				wsFilter.setName(filter.getName());
				wsFilter.getValues().addAll(filter.getValues());
				wsFilters.add(wsFilter);
			}

			List<Subnet> subnets = new ArrayList<Subnet>();
			for (com.clustercontrol.ws.cloud.Subnet wsSubnet: endpoint.getSubnets(regionName, wsFilters)) {
				Subnet subnet = new Subnet();
				subnet.setSubnetId(wsSubnet.getSubnetId());
				subnet.setVpcId(wsSubnet.getVpcId());
				subnet.setAvailabilityZone(wsSubnet.getAvailabilityZone());
				subnet.setCidrBlock(wsSubnet.getCidrBlock());
				subnets.add(subnet);
			}
			return subnets;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<Subnet> getSubnetsByRegion(String regionName) {
		try {
			List<com.clustercontrol.ws.cloud.Filter> emptyList = Collections.emptyList();
			List<Subnet> subnets = new ArrayList<Subnet>();
			for (com.clustercontrol.ws.cloud.Subnet wsSubnet: endpoint.getSubnets(regionName, emptyList)) {
				Subnet subnet = new Subnet();
				subnet.setSubnetId(wsSubnet.getSubnetId());
				subnet.setVpcId(wsSubnet.getVpcId());
				subnet.setAvailabilityZone(wsSubnet.getAvailabilityZone());
				subnet.setCidrBlock(wsSubnet.getCidrBlock());
				subnets.add(subnet);
			}
			return subnets;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}
}