/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.ui.views;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.jface.layout.TreeColumnLayout;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.part.ViewPart;

import com.clustercontrol.action.FacilityTree;
import com.clustercontrol.bean.FacilityImageConstant;
import com.clustercontrol.cloud.Activator;
import com.clustercontrol.cloud.aws.presenter.AllPropertyObserver;
import com.clustercontrol.cloud.aws.presenter.CloudModelException;
import com.clustercontrol.cloud.aws.presenter.ErrorCodeConstants;
import com.clustercontrol.cloud.aws.presenter.IElement;
import com.clustercontrol.cloud.aws.presenter.IFacility;
import com.clustercontrol.cloud.aws.presenter.IFacilityRoot;
import com.clustercontrol.cloud.aws.presenter.INode;
import com.clustercontrol.cloud.aws.presenter.IScope;
import com.clustercontrol.composite.FacilityTreeComposite;
import com.clustercontrol.repository.bean.FacilityConstant;

public class ScopeView extends ViewPart {
	public static final String Id = "com.clustercontrol.cloud.ui.views.ScopeView";
	
	private class FacilityRootUpdateService {
		private FacilityTreeComposite listener;
		
		public FacilityRootUpdateService() {
			listener = new FacilityTreeComposite(composite, SWT.None) {
				@Override
				public void update() {
					parent.getDisplay().asyncExec(new Runnable() {
						@Override
						public void run() {
							if (input == null) {
								return;
							}
							ScopeView.this.update();
						}
					});
				}
				public boolean isDisposed () {
					return false;
				}
			};
			listener.dispose();
			new FacilityTree().addComposite(listener);
		}
		
		public void dispose() {
			new FacilityTree().delComposite(listener);
		}
	}

	private AllPropertyObserver observer = new AllPropertyObserver() {
		@Override
		public void elementAdded(ElementAddedEvent event) {
			if (event.getAddedElement() instanceof IElement) {
				((IElement)event.getAddedElement()).addPropertyObserver2(IElement.allProperty, this);
			}
			
			ScopeView.this.getSite().getShell().getDisplay().asyncExec(new Runnable() {
				@Override
				public void run() {
					treeViewer.refresh(true);
				}
			});
		}

		@Override
		public void elementRemoved(ElementRemovedEvent event) {
			if (event.getRemovedElement() instanceof IElement) {
				((IElement)event.getRemovedElement()).removePropertyObserver2(IElement.allProperty, this);
			}

			ScopeView.this.getSite().getShell().getDisplay().asyncExec(new Runnable() {
				@Override
				public void run() {
					treeViewer.refresh(true);
				}
			});
		}

		@Override
		public void propertyChanged(ValueChangedEvent event) {
			ScopeView.this.getSite().getShell().getDisplay().asyncExec(new Runnable() {
				@Override
				public void run() {
					treeViewer.refresh(true);
				}
			});
		}
	};
	

	private FacilityRootUpdateService service;
	
	private class TreeContentProvider implements IStructuredContentProvider, ITreeContentProvider{
		
		public Object[] getChildren(Object parentElement) {
			if(parentElement instanceof IScope){
				IScope scope = (IScope)parentElement;
				List<IFacility> facilities = new ArrayList<IFacility>();
				facilities.addAll(Arrays.asList(scope.getScopes()));
				facilities.addAll(Arrays.asList(scope.getNodes()));
				
				return facilities.toArray();
			}
			return null;
		}

		public Object getParent(Object element) {
			if(element instanceof IFacility){
				IFacility facility = (IFacility)element;
				return facility.getParent() != null ? facility.getParent(): facility.getFacilityRoot();
			}
			return null;
		}

		public boolean hasChildren(Object element) {
			if (element instanceof IScope){
				IScope scope = (IScope)element;
				return scope.getScopes().length != 0 || scope.getNodes().length != 0;
			}
			return false;	
		}

		public Object[] getElements(Object inputElement) {
			if (inputElement instanceof IFacilityRoot) {
				return ((IFacilityRoot)inputElement).getScopes();
			}
			return null;
		}

		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
	}

	private TreeViewer treeViewer;
	private IFacilityRoot input;
	private Composite composite;
	
	public ScopeView() {
	}

	@Override
	public void createPartControl(Composite arg0) {
		
		composite = new Composite(arg0, SWT.NONE);
		composite.setLayout(new FillLayout(SWT.HORIZONTAL));

		Composite composite_1 = new Composite(composite, SWT.NONE);
		TreeColumnLayout tcl_composite = new TreeColumnLayout();
		composite_1.setLayout(tcl_composite);

		treeViewer = new TreeViewer(composite_1, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);

		treeViewer.setContentProvider(new TreeContentProvider());
		treeViewer.setLabelProvider(new LabelProvider(){
			@Override
			public Image getImage(Object element) {
				if (element instanceof IScope) {
					return FacilityImageConstant.typeToImage(FacilityConstant.TYPE_SCOPE);
				}
				else if (element instanceof INode) {
					return FacilityImageConstant.typeToImage(FacilityConstant.TYPE_NODE);					
				}
				else {
					return FacilityImageConstant.typeToImage(FacilityConstant.TYPE_NODE);					
				}
			}

			@Override
			public String getText(Object element) {
				if (element instanceof IFacility){
					IFacility facility = (IFacility)element;
					return facility.getName() + "(" + facility.getFacilityId() + ")";
				}
				return null;
			}
		});
		
		this.getSite().setSelectionProvider(treeViewer);

		treeViewer.setContentProvider(new TreeContentProvider());

		update();
	}

	private void setUp(IFacilityRoot faciiltyRoot) {
		faciiltyRoot.addPropertyObserver2(IElement.allProperty, observer);
		for (IScope scope: faciiltyRoot.getScopes()) {
			recursiveSetUp(scope);
		}
	}
	
	private void recursiveSetUp(IScope scope) {
		scope.addPropertyObserver2(IElement.allProperty, observer);
		for (IScope child: scope.getScopes()) {
			recursiveSetUp(child);
		}
		for (INode node: scope.getNodes()) {
			node.addPropertyObserver2(IElement.allProperty, observer);
		}
	}
	
	private void setDown(IFacilityRoot faciiltyRoot) {
		faciiltyRoot.removePropertyObserver2(IElement.allProperty, observer);
		for (IScope scope: faciiltyRoot.getScopes()) {
			recursiveSetDown(scope);
		}
	}
	
	private void recursiveSetDown(IScope scope) {
		scope.removePropertyObserver2(IElement.allProperty, observer);
		for (IScope child: scope.getScopes()) {
			recursiveSetDown(child);
		}
		for (INode node: scope.getNodes()) {
			node.removePropertyObserver2(IElement.allProperty, observer);
		}
	}

	@Override
	public void setFocus() {
		treeViewer.getTree().setFocus();
	}

	@Override
	public void dispose() {
		if (service != null) {
			service.dispose();
		}
		if (input != null) {
			setDown(input);
		}
		
		// ログオフ時の動作としてモデルのリリースをこのビューで行う。
		Activator.getDefault().releaseCloudManager();
		
		super.dispose();
	}

	public void update() {
		if (input == null) {
			try {
				input = Activator.getDefault().getCloudManager().getHinemosService().getFacilityRoot();
				if (input != null) {
					setUp(input);
					treeViewer.setInput(input);
					treeViewer.expandToLevel(2);
					service = new FacilityRootUpdateService();
				}
			}
			catch (CloudModelException e) {
				Logger logger = Logger.getLogger(ScopeView.class);
				logger.error(e.getMessage(), e);
			}
		}
		else {
			try {
				input.update();
			}
			catch (CloudModelException e) {
				if (ErrorCodeConstants.FACILITYROOT_INVALID_ROOT_SCOPE_NOT_FOUND.equals(e.getErrorCode())) {
					//　ビュー表示内容の消去。
					setDown(input);
					input = null;
					treeViewer.setInput(null);
					service.dispose();
					service = null;
				}
				else {
					Logger logger = Logger.getLogger(ScopeView.class);
					logger.warn(e.getMessage(), e);
				}
			}
		}
	}
}
