/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.aws.base.commons.util;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import com.clustercontrol.cloud.commons.util.MessageManager;

public class MessageManagerExt {
	private static final Map<String, MessageManagerExt> managers = new HashMap<String, MessageManagerExt>();
	private MessageManager messages = null;
	
	private String fileName;
	private ResourceBundle bundle;

	private MessageManagerExt(String fileName) {
		this.fileName = fileName;
		bundle = ResourceBundle.getBundle(fileName);
		try{
			messages = MessageManager.getInstance(fileName);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public static MessageManagerExt getInstance(String fileName){
		MessageManagerExt manager = null;
		synchronized(managers){
			if(managers.containsKey(fileName)){
				manager = managers.get(fileName);
			} else { 
				manager = new MessageManagerExt(fileName);
				managers.put(fileName, manager);
			}
		}
		return manager;
	}


	/**
	 * Returns the formatted message for the given key in the resource bundle.
	 * 
	 * @param key
	 *            the resource name
	 * @param args
	 *            the message arguments
	 * @return the string
	 */
	public String getString(String key, Object[] args) {
		List<Object> objs = new LinkedList<Object>(); 
		for(Object obj: args){
			objs.add(getString((String)obj));
		}
		MessageFormat messageFormat = new MessageFormat(getString(key));
		return messageFormat.format(objs.toArray());
	}

	/**
	 * Returns the resource object with the given key in the resource bundle. If
	 * there isn't any value under the given key, the key is returned.
	 * 
	 * @param key
	 *            the resource name
	 * @return the string
	 */
	public String getString(String key) {
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			if(messages != null){
				return messages.getString(key);
			}
			return key;
		}
	}

	/**
	 * Returns the resource object with the given key in the resource bundle. If
	 * there isn't any value under the given key, the default value is returned.
	 * 
	 * @param key
	 *            the resource name
	 * @param def
	 *            the default value
	 * @return the string
	 */
	public String getString(String key, String def) {
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return def;
		}
	}

	/**
	 * Returns the formatted message for the given key in the resource bundle.
	 * 
	 * @param key
	 *            the resource name
	 * @param args
	 *            the message arguments
	 * @param locale
	 * @return the string
	 */
	public String getString(String key, Object[] args, Locale locale) {
		MessageFormat messageFormat = new MessageFormat(getString(key, locale));
		return messageFormat.format(args);
	}

	/**
	 * Returns the resource object with the given key in the resource bundle. If
	 * there isn't any value under the given key, the key is returned.
	 * 
	 * @param key
	 *            the resource name
	 * @param locale
	 * @return the string
	 */
	public String getString(String key, Locale locale) {
		ResourceBundle bundle = ResourceBundle.getBundle(fileName,
				locale);
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return key;
		}
	}

	/**
	 * Returns the resource object with the given key in the resource bundle. If
	 * there isn't any value under the given key, the default value is returned.
	 * 
	 * @param key
	 *            the resource name
	 * @param def
	 *            the default value
	 * @param locale
	 * @return the string
	 */
	public String getString(String key, String def, Locale locale) {
		ResourceBundle bundle = ResourceBundle.getBundle(fileName,
				locale);
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return def;
		}
	}
}
