/*
Copyright (C) 2013 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
*/
package com.clustercontrol.cloud;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.ws.WebServiceContext;

import org.apache.log4j.Logger;

import com.clustercontrol.cloud.SessionService.ISession;
import com.clustercontrol.cloud.bean.CloudUser;
import com.clustercontrol.cloud.commons.CloudPropertyConstants;
import com.clustercontrol.cloud.dao.CloudUserDao.CloudUserType;
import com.clustercontrol.cloud.factory.AccountResourceOperator;
import com.clustercontrol.cloud.factory.CloudServiceOperator;
import com.clustercontrol.cloud.factory.CloudUserOperator;
import com.clustercontrol.cloud.factory.IAccountResourceOperator;
import com.clustercontrol.cloud.factory.ICloudServiceOperator;
import com.clustercontrol.cloud.factory.ICloudUserOperator;
import com.clustercontrol.cloud.factory.IInstanceOperator;
import com.clustercontrol.cloud.factory.IStorageOperator;
import com.clustercontrol.cloud.factory.ITemplateOperator;
import com.clustercontrol.cloud.factory.InstanceOperator;
import com.clustercontrol.cloud.factory.StorageOperator;
import com.clustercontrol.cloud.factory.TemplateOperator;
import com.clustercontrol.cloud.factory.monitors.AutoDetectionManager;
import com.clustercontrol.cloud.factory.monitors.CloudServiceBillingMonitor;
import com.clustercontrol.cloud.registry.AbstractObjectChangedListener;
import com.clustercontrol.cloud.registry.IObjectChangedService;
import com.clustercontrol.cloud.registry.ObjectChangedService;
import com.clustercontrol.cloud.registry.ObjectRegistryService;
import com.clustercontrol.cloud.util.AccessControllerBeanWrapper;
import com.clustercontrol.cloud.util.AccountResourceUtil;
import com.clustercontrol.cloud.validation.ValidationUtil;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.commons.util.ObjectSharingService;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.JobMasterNotFound;
import com.clustercontrol.fault.PrivilegeDuplicate;
import com.clustercontrol.fault.UsedObjectPrivilege;
import com.clustercontrol.monitor.run.factory.RunMonitor;
import com.clustercontrol.ws.cloud.CloudEndpointImpl;
import com.clustercontrol.ws.cloud.IWebServiceBase;
import com.clustercontrol.ws.cloud.Publisher;
import com.clustercontrol.ws.util.HttpAuthenticator;

public class CloudManagerBaseService extends CloudPlugin {
	public final static String id = "com.clustercontrol.cloud.CloudService";

	public final static String Key_MainEndpoint = "main_endpoint";
	
	private static CloudManagerBaseService singleton;
	
	private Publisher publisher = new Publisher();

	/**
	 * CloudServiceのコンストラクタ
	 */
	public CloudManagerBaseService() {
		super();
		singleton = this;
	}

	public synchronized void publish (String postAddress, IWebServiceBase implementor) {
		String address = HinemosProperties.getProperty("common.ws.address" , "http://0.0.0.0:8080") + postAddress;
		Logger.getLogger(this.getClass()).info("publish " + address);
		publisher.publish(address, implementor);
	}

	@Override
	public Set<String> getDependency() {
		return Collections.emptySet();
	}

	@Override
	public void create() {
		Logger logger = Logger.getLogger(this.getClass());

		// メイン処理
		logger.info("creating " + this.getClass().getSimpleName() + "...");

		//　プロパティ一覧の表示
		for (CloudPropertyConstants value: CloudPropertyConstants.values()) {
			logger.info(value.id + " = " + value.value() + " (" + value.defaultValue() + ")");
		}

		logger.info("successful in creating " + this.getClass().getSimpleName() + "...");
	}

	@Override
	public void activate() {
		Logger logger = Logger.getLogger(this.getClass());

		// メイン処理
		logger.info("activate " + this.getClass().getSimpleName() + "...");

		// クラウド管理プラグインの初期化
		final Map<String, CloudPlugin> pluginMap = new HashMap<>();
		List<CloudPlugin> pluginList = new ArrayList<>(getPluginMap().values());
		for (CloudPlugin plugin: getPluginMap().values()) {
			pluginMap.put(plugin.getClass().getName(), plugin);
		}
		
		Collections.sort(pluginList, new Comparator<CloudPlugin>() {
				public int compare(CloudPlugin o1, CloudPlugin o2) {
					if (checkNegative(o1, o2)) {
						return -1;
					}
					else if (checkPositive(o1, o2)) {
						return 1;
					}
					return 0;
				}

				private boolean checkNegative(CloudPlugin o1, CloudPlugin dependencyPlugin) {
					for (String dependency: dependencyPlugin.getDependency()) {
						if (!dependency.equals(o1.getClass().getName())) {
							CloudPlugin nextDependencyPlugin = pluginMap.get(dependency);
							if (checkNegative(o1, nextDependencyPlugin)) {
								return true;
							}
						}
						else {
							return true;
						}
					}
					return false;
				}
				
				private boolean checkPositive(CloudPlugin o1, CloudPlugin dependencyPlugin) {
					for (String dependency: o1.getDependency()) {
						if (!dependency.equals(dependencyPlugin.getClass().getName())) {
							CloudPlugin nextDependencyPlugin = pluginMap.get(dependency);
							if (checkNegative(nextDependencyPlugin, dependencyPlugin)) {
								return true;
							}
						}
						else {
							return true;
						}
					}
					return false;
				}
			});
		
		for (CloudPlugin cloudPlugin: pluginList) {
			cloudPlugin.initialize();
		}
		
		// メインのエンドポイントを登録
		IWebServiceBase mainEndpoint = ObjectRegistryService.registry().get(IWebServiceBase.class, Key_MainEndpoint);
		
		// Webサービスの起動処理
		publish("/HinemosWS/CloudEndpoint", mainEndpoint == null ? new CloudEndpointImpl(): mainEndpoint);

		AutoDetectionManager.getService().start();

		logger.info("successful in activating " + this.getClass().getSimpleName() + "...");
	}

	@Override
	public void deactivate() {
		Logger logger = Logger.getLogger(this.getClass());

		// メイン処理
		logger.info("stopping " + this.getClass().getSimpleName() + "...");

		AutoDetectionManager.getService().stop();

		// webサービスの停止
		try {
			synchronized (this) {
				publisher.close();
			}
		}
		catch (Exception e) {
			logger.error(e.getMessage(), e);
		}

		logger.info("successful in stopping " + this.getClass().getSimpleName() + "...");
	}

	@Override
	public void destroy() {
		Logger logger = Logger.getLogger(this.getClass());

		// メイン処理
		logger.info("destroying " + this.getClass().getSimpleName() + "...");

		logger.info("successful in destroying " + this.getClass().getSimpleName() + "...");
	}

	public static CloudManagerBaseService getSingleton() {
		return singleton;
	}

	@Override
	public String getPluginId() {
		return id;
	}

	@Override
	public void initialize() {
		Logger logger = Logger.getLogger(this.getClass());
		logger.info("initializing " + this.getClass().getSimpleName() + "...");

		ValidationUtil.setMessages(new MessagesHolder("com.clustercontrol.cloud.validation.cloud_validation_messages"));
		
		// Web サービスのリクエスト実行中に発生したエラーの最終処理を設定。
		publisher.setUncaughtExceptionHandler(new Publisher.UncaughtExceptionHandler() {
			@Override
			public CloudManagerFault uncaughtException(Throwable e) {
				if (e instanceof CloudManagerFault) {
					return (CloudManagerFault)e;
				}
				else if (e instanceof PluginFault) {
					return new CloudManagerFault(e);
				}
				else {
					return null;
				}
			}
		});
		
		ObjectRegistryService.registry().put(ICloudServiceOperator.class, CloudServiceOperator.class);
		ObjectRegistryService.registry().put(IAccountResourceOperator.class, AccountResourceOperator.class);
		ObjectRegistryService.registry().put(ICloudUserOperator.class, CloudUserOperator.class);
		ObjectRegistryService.registry().put(IInstanceOperator.class, InstanceOperator.class);
		ObjectRegistryService.registry().put(IStorageOperator.class, StorageOperator.class);
		ObjectRegistryService.registry().put(ITemplateOperator.class, TemplateOperator.class);
		ObjectRegistryService.registry().put(IObjectChangedService.class, new ObjectChangedService());

		// クラウド課金監視を追加
		ObjectSharingService.objectRegistry().put(RunMonitor.class, CloudServiceBillingMonitor.monitorTypeId + "." + CloudServiceBillingMonitor.monitorType, CloudServiceBillingMonitor.class);
		
		IObjectChangedService service = ObjectRegistryService.registry().get(IObjectChangedService.class);
		service.addObjectChangedListener(null, CloudUser.class, new AbstractObjectChangedListener<CloudUser>() {
			@Override
			public void postAdded(String type, CloudUser object) throws PluginFault {
				// 追加されたアカウントが制限ユーザーの場合、オブジェクト権限をアカウントリソース関連リソースに設定する。
				if (object.getCloudUserType() == CloudUserType.user) {
					try {
						AccountResourceUtil.addAccountResourceRightToRole(AccessControllerBeanWrapper.bean(), object.getRoleId(), object.getAccountResourceId());
					}
					catch (PrivilegeDuplicate | UsedObjectPrivilege | HinemosUnknown | InvalidSetting | InvalidRole | JobMasterNotFound e) {
						throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
					}
				}
			}

			@Override
			public void preRemoved(String type, CloudUser object) throws PluginFault {
				// 制限ユーザの削除は、オブジェクト権限も消す。
				if (object.getCloudUserType() == CloudUserType.user) {
					try {
						AccountResourceUtil.removeAccountResourceRightToRole(AccessControllerBeanWrapper.bean(), object.getRoleId(), object.getAccountResourceId());
					}
					catch (PrivilegeDuplicate | UsedObjectPrivilege | HinemosUnknown | InvalidSetting | InvalidRole | JobMasterNotFound e) {
						throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
					}
				}
			}
		});

		// セッション情報の初期化処理を追加。
		SessionService.addInitializer(new SessionService.ISessionInitializer() {
			@Override
			public void initialize(ISession context) {
				try {
					// Web サービスの認証情報から、ログオンユーザーをセッション情報に設定。
					WebServiceContext wsContext = Publisher.getCurrentContext();
					if (wsContext != null) {
						String accountName = HttpAuthenticator.getAccount(Publisher.getCurrentContext()).split(":")[0];
						context.setHinemosCredential(new HinemosCredential(accountName));
					}
				}
				catch (InvalidUserPass e) {
					throw new InternalManagerError(e);
				}
			}

			@Override
			public void close(ISession context) {
			}
		});
		logger.info("successful in initializing " + this.getClass().getSimpleName() + "...");
	}
}