/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.MissingResourceException;

import javax.xml.bind.annotation.XmlTransient;

import com.clustercontrol.cloud.bean.Image;
import com.clustercontrol.cloud.bean.InstanceStateChange;
import com.clustercontrol.cloud.bean.InstanceStateKind;
import com.clustercontrol.cloud.bean.PlatformKind;
import com.clustercontrol.cloud.bean.Snapshot;
import com.clustercontrol.cloud.bean.StorageAttachmentStateKind;
import com.clustercontrol.cloud.bean.StorageStateKind;
import com.clustercontrol.cloud.bean.Tag;
import com.clustercontrol.cloud.bean.Zone;

public interface IResourceManagement {
	public enum ErrorCode {
		Resource_InvalidStorage_NotFound("message.error.resource.invalidstorage.not_found"),
		Resource_InvalidInstanceID_NotFound("message.error.resource.invalidinstanceid.not_found"),
		Resource_InvalidImageID_NotFound("message.error.resource.invalidimage.not_found"),
		Resource_InvalidSnapshot_NotFound("message.error.resource.invalidsnapshot.not_found"),
		Resource_Instance_Backup_Image_State_Failed("message.error.resource.instance_backup_image_state_failed");
		
		private String messageId;

		private ErrorCode(String messageId) {
			this.messageId = messageId;
		}
		
		public String getMessage(Object... args) {
			MessageFormat messageFormat = new MessageFormat(getMessage());
			return messageFormat.format(args);
		}

		public String getMessage() {
			try {
				return Messages.messages().getString(messageId);
			}
			catch (MissingResourceException e) {
				return messageId;
			}
		}

		public CloudManagerFault cloudManagerFault(Object... args) {
			return new CloudManagerFault(getMessage(args), this.name());
		}

	}
	
	public interface ICredential {
	    String getAccessKey() throws CloudManagerFault ;
	    String getSecretKey() throws CloudManagerFault ;
	}

	public interface IRegion {
		public String getName();
		public IEndpoint getEndpoint(String epName);
	}

	public interface IStore {
		public static class StoreValue {
			private String resourceType;
			private String resourceId;
			private String value;

			public StoreValue(String resourceType, String resourceId,
					String value) {
				super();
				this.resourceType = resourceType;
				this.resourceId = resourceId;
				this.value = value;
			}
			public String getResourceType() {
				return resourceType;
			}
			public void setResourceType(String resourceType) {
				this.resourceType = resourceType;
			}
			public String getResourceId() {
				return resourceId;
			}
			public void setResourceId(String resourceId) {
				this.resourceId = resourceId;
			}
			public String getValue() {
				return value;
			}
			public void setValue(String value) {
				this.value = value;
			}
		}
		
		public List<StoreValue> getValues(String resourceType) throws CloudManagerFault;
		public String get(String resourceType, String resourceId) throws CloudManagerFault;
		public void put(String resourceType, String resourceId, String value) throws CloudManagerFault;
		public void remove(String resourceType, String resourceId) throws CloudManagerFault;
		public List<String> getIds(String resourceType) throws CloudManagerFault;
	}

	public interface IEndpoint {
		public String getEndpointType();
		public String getLocation();
	}
	
	public static abstract class Resource {
		private String resourceType;
		private Object actualResource;
		
		public Resource() {
		}

		public Resource(String resourceType) {
			this.resourceType = resourceType;
		}
		
		public String getResourceType() {
			return resourceType;
		}
		public void setResourceType(String resourceType) {
			this.resourceType = resourceType;
		}
		
		@XmlTransient
		public Object getActualResource() {
			return actualResource;
		}
		public void setActualResource(Object actualResource) {
			this.actualResource = actualResource;
		}
	}
	
	
	public static class Instance extends Resource {
		public static class BlockDeviceMapping {

			private String deviceName;

			private String storageId;

			private String status;

			public BlockDeviceMapping() {
				
			}
			public BlockDeviceMapping(
				String storageId,
				String deviceName,
				String status
				) {
				this.storageId = storageId;
				this.deviceName = deviceName;
				this.status = status;
			}

			public String getDeviceName() {
				return deviceName;
			}

			public void setDeviceName(String deviceName) {
				this.deviceName = deviceName;
			}

			public String getStorageId() {
				return storageId;
			}

			public void setStorageId(String storageId) {
				this.storageId = storageId;
			}

			public String getStatus() {
				return status;
			}

			public void setStatus(String status) {
				this.status = status;
			}
		}

		private String instanceId;
		private String name;
		private String flavor;
		private String zone;
		private InstanceStateKind state;
		private String imageId;
		private PlatformKind platform;
		private String hostName;
		private String ipAddress;
		private List<BlockDeviceMapping> blockDeviceMappings = new ArrayList<>();
		private List<Tag> tags = new ArrayList<>();

		public Instance() {
			super();
		}
		
		public Instance(String resourceType, String instanceId, String name, String flavor, String zone, InstanceStateKind state, String imageId, PlatformKind platform, String hostName, String ipAddress, String detail, List<BlockDeviceMapping> blockDeviceMappings, List<Tag> tags) {
			super(resourceType);
			this.instanceId = instanceId;
			this.name = name;
			this.flavor = flavor;
			this.zone = zone;
			this.state = state;
			this.imageId = imageId;
			this.platform = platform;
			this.hostName = hostName;
			this.ipAddress = ipAddress;
			this.blockDeviceMappings = blockDeviceMappings;
			this.tags = tags;
		}
		
		public String getInstanceId() {
			return instanceId;
		}
		public void setInstanceId(String instanceId) {
			this.instanceId = instanceId;
		}

		public String getName() {
			return name;
		}
		public void setName(String name) {
			this.name = name;
		}

		public String getFlavor() {
			return flavor;
		}
		public void setFlavor(String flavor) {
			this.flavor = flavor;
		}

		public InstanceStateKind getState() {
			return state;
		}
		public void setState(InstanceStateKind state) {
			this.state = state;
		}

		public String getImageId() {
			return imageId;
		}
		public void setImageId(String imageId) {
			this.imageId = imageId;
		}

		public PlatformKind getPlatform() {
			return platform;
		}
		public void setPlatform(PlatformKind platform) {
			this.platform = platform;
		}

		public String getZone() {
			return zone;
		}
		public void setZone(String zone) {
			this.zone = zone;
		}

		public String getIpAddress() {
			return ipAddress;
		}
		public void setIpAddress(String ipAddress) {
			this.ipAddress = ipAddress;
		}

		public String getHostName() {
			return hostName;
		}
		public void setHostName(String hostName) {
			this.hostName = hostName;
		}

		public List<BlockDeviceMapping> getBlockDeviceMappings() {
			return blockDeviceMappings;
		}
		public void setBlockDeviceMappings(List<BlockDeviceMapping> blockDeviceMappings) {
			this.blockDeviceMappings = blockDeviceMappings;
		}

		public List<Tag> getTags() {
			return tags;
		}
		public void setTags(List<Tag> tags) {
			this.tags = tags;
		}
	}
	
	public static class StorageBackup extends Resource {
		public static class BackupedData {
			private String name;
			private String flavor;
			private String zone;
			private Integer size;
			private String storageDetail;

			public BackupedData() {
			}

			public String getName() {
				return name;
			}

			public void setName(String name) {
				this.name = name;
			}

			public String getFlavor() {
				return flavor;
			}

			public void setFlavor(String flavor) {
				this.flavor = flavor;
			}

			public String getZone() {
				return zone;
			}

			public void setZone(String zone) {
				this.zone = zone;
			}

			public int getSize() {
				return size == null ? 0: size;
			}
			public void setSize(int size) {
				this.size = size;
			}

			public String getStorageDetail() {
				return storageDetail;
			}

			public void setStorageDetail(String storageDetail) {
				this.storageDetail = storageDetail;
			}
		}

		private String storageBackupId;
		private String name;
		private String description;
		private String storageId;
		private BackupedData backupedData;
		private Date createTime;

		public String getStorageBackupId() {
			return storageBackupId;
		}
		public void setStorageBackupId(String storageBackupId) {
			this.storageBackupId = storageBackupId;
		}
		public String getName() {
			return name;
		}
		public void setName(String name) {
			this.name = name;
		}
		public String getDescription() {
			return description;
		}
		public void setDescription(String description) {
			this.description = description;
		}
		public String getStorageId() {
			return storageId;
		}
		public void setStorageId(String storageId) {
			this.storageId = storageId;
		}
		public Date getCreateTime() {
			return createTime;
		}
		public void setCreateTime(Date createTime) {
			this.createTime = createTime;
		}
		public BackupedData getBackupedData() {
			return backupedData;
		}
		public void setBackupedData(BackupedData backupedData) {
			this.backupedData = backupedData;
		}
	}
	
	public static class InstanceBackup extends Resource {
		public static class BackupedData {
			private String name;
			private String flavor;
			private String zone;
			private List<Tag> tags;
			private String instanceDetail;

			public BackupedData() {
			}

			public String getName() {
				return name;
			}

			public void setName(String name) {
				this.name = name;
			}

			public String getFlavor() {
				return flavor;
			}

			public void setFlavor(String flavor) {
				this.flavor = flavor;
			}

			public String getZone() {
				return zone;
			}

			public void setZone(String zone) {
				this.zone = zone;
			}

			public List<Tag> getTags() {
				return tags;
			}

			public void setTags(List<Tag> tags) {
				this.tags = tags;
			}

			public String getInstanceDetail() {
				return instanceDetail;
			}

			public void setInstanceDetail(String instanceDetail) {
				this.instanceDetail = instanceDetail;
			}
		}
		
		private String instanceBackupId;
		private String name;
		private String description;
		private String instanceId;
		private PlatformKind platform;
		private BackupedData backupedData;
		private Date createTime;

		public String getInstanceBackupId() {
			return instanceBackupId;
		}
		public void setInstanceBackupId(String instanceBackupId) {
			this.instanceBackupId = instanceBackupId;
		}
		public String getName() {
			return name;
		}
		public void setName(String name) {
			this.name = name;
		}
		public PlatformKind getPlatform() {
			return platform;
		}
		public void setPlatform(PlatformKind platform) {
			this.platform = platform;
		}
		public Date getCreateTime() {
			return createTime;
		}
		public void setCreateTime(Date createTime) {
			this.createTime = createTime;
		}
		public String getDescription() {
			return description;
		}
		public void setDescription(String description) {
			this.description = description;
		}
		public BackupedData getBackupedData() {
			return backupedData;
		}
		public void setBackupedData(BackupedData backupedData) {
			this.backupedData = backupedData;
		}
		public String getInstanceId() {
			return instanceId;
		}
		public void setInstanceId(String instanceId) {
			this.instanceId = instanceId;
		}
	}
	
	public static class Storage extends Resource {
		public static class StorageAttachment {
			private String instanceId;
			private String device;
			private StorageAttachmentStateKind state;
		    private Date attachTime;

			public StorageAttachment() {
			}
			
			public StorageAttachment(String instanceId, String device, StorageAttachmentStateKind state, Date attachTime) {
				this.instanceId = instanceId;
				this.device = device;
				this.state = state;
				this.attachTime = attachTime;
			}
			
			public String getInstanceId() {
				return instanceId;
			}
			public void setInstanceId(String instanceId) {
				this.instanceId = instanceId;
			}

			public String getDevice() {
				return device;
			}
			public void setDevice(String device) {
				this.device = device;
			}

			public StorageAttachmentStateKind getState() {
				return state;
			}
			public void setState(StorageAttachmentStateKind state) {
				this.state = state;
			}

			public Date getAttachTime() {
				return attachTime;
			}

			public void setAttachTime(Date attachTime) {
				this.attachTime = attachTime;
			}
		}
		
		private String storageId;
		private String name;
		private String flavor;
		private int size;
		private String zone;
		private String snapshotId;
		private StorageStateKind state;
	    private Date createTime;

		private StorageAttachment sa;

		public String getStorageId() {
			return storageId;
		}
		public void setStorageId(String storageId) {
			this.storageId = storageId;
		}
		public String getName() {
			return name;
		}
		public void setName(String name) {
			this.name = name;
		}
		public String getFlavor() {
			return flavor;
		}
		public void setFlavor(String flavor) {
			this.flavor = flavor;
		}
		public String getSnapshotId() {
			return snapshotId;
		}
		public void setSnapshotId(String snapshotId) {
			this.snapshotId = snapshotId;
		}
		public String getZone() {
			return zone;
		}
		public void setZone(String zone) {
			this.zone = zone;
		}
		public StorageAttachment getStorageAttachment() {
			return sa;
		}
		public void setStorageAttachment(StorageAttachment sa) {
			this.sa = sa;
		}
		public int getSize() {
			return size;
		}
		public void setSize(int size) {
			this.size = size;
		}
		public StorageStateKind getState() {
			return state;
		}
		public void setState(StorageStateKind state) {
			this.state = state;
		}
		public Date getCreateTime() {
			return createTime;
		}
		public void setCreateTime(Date createTime) {
			this.createTime = createTime;
		}
	}
	
	public static class LoadBalancer extends Resource {
		private String lbId;
		private List<String> registerdInstanceIds;
		private List<String> availableInstanceIds;

		public String getLbId() {
			return lbId;
		}
		public void setLbId(String lbId) {
			this.lbId = lbId;
		}
		public List<String> getRegisterdInstanceIds() {
			return registerdInstanceIds;
		}
		public void setRegisterdInstanceIds(List<String> registerdInstanceIds) {
			this.registerdInstanceIds = registerdInstanceIds;
		}
		public List<String> getAvailableInstanceIds() {
			return availableInstanceIds;
		}
		public void setAvailableInstanceIds(List<String> availableInstanceIds) {
			this.availableInstanceIds = availableInstanceIds;
		}
	}
	
	void setAccessDestination(ICredential credential, IRegion region);
	void setStore(IStore store);
	ICredential getCledential();
	IRegion getRegion();
	void disconnect();
	
// インスタンス
	Instance createInstance(String name,  String flavor, String imageId, String zone, String instanceDetail, List<Tag> tags) throws CloudManagerFault;
	Instance restoreInstance(String instanceBackupId, String name,  String flavor, String zone, String instanceDetail, List<Tag> tags) throws CloudManagerFault;
	void deleteInstance(String instanceId) throws CloudManagerFault;
	Instance getInstance(String instanceId) throws CloudManagerFault;
	List<Instance> getInstances(String...instanceIds) throws CloudManagerFault;
	List<Instance> getInstances(List<String> instanceIds) throws CloudManagerFault;
	InstanceStateChange startInstance(String instanceId) throws CloudManagerFault;
	InstanceStateChange stopInstance(String instanceId) throws CloudManagerFault;

// インスタンス作成時の種別
	List<String> getInstanceFlavors() throws CloudManagerFault;
// ストレージ作成時の種別 (iops1、standard)
	List<String> getStorageFlavors() throws CloudManagerFault;
	
// アベイラビリティゾーン一覧の取得
	List<Zone> getZones() throws CloudManagerFault;

// ストレージ関連
	Storage createStorage(String name, String flavor, int size, String snapshotId, String zone, String storageDetail) throws CloudManagerFault;
	Storage restoreStorage(String storageBackupId, String name, String flavor, Integer size, String zone, String storageDetail) throws CloudManagerFault;
	void attachStorage(String instanceId, String storageId, String deviceName) throws CloudManagerFault;
	void detachStorage(String instanceId, String storageId) throws CloudManagerFault;
	void deleteStorage(String storageId) throws CloudManagerFault;
	Storage getStorage(String storageId) throws CloudManagerFault;
	List<Storage> getStorages(String...storageIds) throws CloudManagerFault;
	List<Storage> getStorages(List<String> storageIds) throws CloudManagerFault;
	
// ストレージバックアップ
	StorageBackup createStorageBackup(String storageId, String name, String description, String backupOption) throws CloudManagerFault;
	void deleteStorageBackup(String storageBackupId) throws CloudManagerFault;
	StorageBackup getStorageBackup(String storageBackupId) throws CloudManagerFault;
	List<StorageBackup> getStorageBackups(String...storageBackupId) throws CloudManagerFault;
	List<StorageBackup> getStorageBackups(List<String> storageBackupId) throws CloudManagerFault;

// インスタンスバックアップ
	InstanceBackup createInstanceBackup(String instanceId, String name, String description, Boolean noReboot, String backupOption) throws CloudManagerFault;
	InstanceBackup createInstanceBackup(String instanceId, String name, String description, Boolean noReboot, List<String> strageIds, String backupOption) throws CloudManagerFault;
	void deleteInstanceBackup(String instanceBackupId) throws CloudManagerFault;
	InstanceBackup getInstanceBackup(String instanceBackupId) throws CloudManagerFault;
	List<InstanceBackup> getInstanceBackups(String...instanceBackupIds) throws CloudManagerFault;
	List<InstanceBackup> getInstanceBackups(List<String> instanceBackupIds) throws CloudManagerFault;

// スナップショット
	Snapshot getSnapshot(String snapshotId) throws CloudManagerFault;
	List<Snapshot> getSnapshots(String...snapshotIds) throws CloudManagerFault;
	List<Snapshot> getSnapshots(List<String> snapshotIds) throws CloudManagerFault;
	List<Snapshot> getSnapshotsWithFilter(Filter...filters) throws CloudManagerFault;
	List<Snapshot> getSnapshotsWithFilter(List<Filter> filters) throws CloudManagerFault;
		
// イメージ
	Image getImage(String imageId) throws CloudManagerFault;
	List<Image> getImages(String...imageIds) throws CloudManagerFault;
	List<Image> getImages(List<String> imageIds) throws CloudManagerFault;
	List<Image> getImagesWithFilter(Filter...filters) throws CloudManagerFault;
	List<Image> getImagesWithFilter(List<Filter> filters) throws CloudManagerFault;

// ロードバランサー
	List<LoadBalancer> getLoadBalancers() throws CloudManagerFault;
	void registerInstanceToLoadBalancer(String lbId, String instanceId) throws CloudManagerFault;
	void unregisterInstanceToLoadBalancer(String lbId, String instanceId) throws CloudManagerFault;
}