/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.ui.views;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;
import org.eclipse.swt.widgets.Display;

import com.clustercontrol.cloud.Activator;
import com.clustercontrol.cloud.presenter.ICloudInstance;
import com.clustercontrol.cloud.presenter.ICloudInstance.StateType;
import com.clustercontrol.ws.cloud.CloudEndpoint;
import com.clustercontrol.ws.cloud.CloudInstance;

/**
 * AWS へポーリングし、AWS のインスタンスを追跡する。
 * 
 * @author torun
 *
 */
public class InstanceMonitorService {
	private static class InstanceIdentity {
		public InstanceIdentity(String regionName, String instanceId, ICloudInstance instance, StateType[] stateTypes) {
			this.regionName = regionName;
			this.instanceId = instanceId;
			this.instance = instance;
			this.stateTypes = stateTypes;
		}
		
		public ICloudInstance instance;
		public String regionName;
		public String instanceId;
		public StateType[] stateTypes;
	}
	
	private final ScheduledThreadPoolExecutor executor = new ScheduledThreadPoolExecutor(1) {
		@Override
	    protected void afterExecute(Runnable r, Throwable t) {
	    	if (t != null) {
	    		Logger logger = Logger.getLogger(InstanceMonitorService.class);
	    		logger.debug(t.getMessage(), t);
	    	}
	    }
	};

	/**
	 * 同時する必要あり。
	 */
	private List<InstanceIdentity> stoppedStatusMap = new ArrayList<InstanceIdentity>();
	/**
	 * instanceMap に合わせて同期される。
	 */
	private ScheduledFuture<?> sf;
	
	private CloudEndpoint endpoint;
	
	private Display display;
	
    private static final ThreadLocal<InstanceMonitorService> instanceMonitorService = 
        new ThreadLocal<InstanceMonitorService>() {
            @Override
            protected InstanceMonitorService initialValue() {
                return new InstanceMonitorService();
            }
    	};
	
	private InstanceMonitorService() {
		display = Display.getCurrent();
		endpoint = Activator.getEndpointManager().getEndpoint(CloudEndpoint.class);
	}
	
	/**
	 * 追跡開始。
	 * 
	 * @param indtanecId
	 * @param listener
	 */
	public void startMonitor(final ICloudInstance instance, StateType... stoppedStatus) {
		synchronized (stoppedStatusMap) {
			stoppedStatusMap.add(new InstanceIdentity(instance.getRegion(), instance.getInstanceId(), instance, stoppedStatus));
			
			if (sf == null) {
				sf = scheduleAtFixedRate(
					new Runnable() {
						@Override
						public void run() {
							Logger logger = Logger.getLogger(InstanceMonitorService.class);
							try {
								List<InstanceIdentity> tempStoppedStatus = new ArrayList<InstanceIdentity>();
								synchronized(stoppedStatusMap) {
									tempStoppedStatus.addAll(stoppedStatusMap);
								}
														
								// 既存のインスタンスの情報が取得できている場合、更新。
								Iterator<InstanceIdentity> iter = tempStoppedStatus.iterator();
								while (iter.hasNext()) {
									final InstanceIdentity instanceIdentity = iter.next();
									if (instanceIdentity.stateTypes == null) {
										// このパターンはないはず。
										continue;
									}
		
									logger.debug("region : " + instanceIdentity.regionName + ", instanceid : " + instanceIdentity.instanceId + ", condition : " + instanceIdentity.stateTypes.toString());
									
									try {
										final CloudInstance cloudInstance = endpoint.getInstance(instance.getCloudInstanceManager().getRegion().getAccountResource().getActiveUser().getRoleId(), instanceIdentity.regionName, instanceIdentity.instanceId);
										display.asyncExec(new Runnable() {
											@Override
											public void run() {
												((com.clustercontrol.cloud.presenter.CloudInstance)instanceIdentity.instance).internalUpdate(cloudInstance);
											}
										});
										
										for (StateType state: instanceIdentity.stateTypes) {
											if (state.label().equals(cloudInstance.getState().value())) {
												Collections.synchronizedList(stoppedStatusMap).remove(instanceIdentity);
												iter.remove();
												break;
											}
										}
									}
									catch (Exception e) {
										logger.error(e.getMessage(), e);
										Collections.synchronizedList(stoppedStatusMap).remove(instanceIdentity);
										iter.remove();
									}
								}
								
								mustStop();
							}
							catch (Exception e) {
								logger.error(e.getMessage(), e);
							}
						}
					},
					10,
					10,
					TimeUnit.SECONDS); 
			}
		}
	}
	
	/**
	 * 追跡停止。
	 * 
	 * @param indtanecId
	 */
	public void stopMonitor(String indtanecId) {
		synchronized (stoppedStatusMap) {
			stoppedStatusMap.remove(indtanecId);
			mustStop();
		}
	}
	
	private void mustStop() {
		synchronized (stoppedStatusMap) {
			if (sf != null && stoppedStatusMap.isEmpty()) {
				sf.cancel(true);
				sf = null;
			}
		}
	}

	private ScheduledFuture<?> scheduleAtFixedRate(Runnable command, long initialDelay, long period, TimeUnit unit) {
		return executor.scheduleAtFixedRate(command, initialDelay, period, unit);
	}
	
	public void shutdown() {
		executor.shutdown();
		stoppedStatusMap.clear();
	}
	
	public static InstanceMonitorService getInstanceMonitorService() {
		return instanceMonitorService.get();
	}
}