/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.persistence;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.persistence.EntityExistsException;
import javax.persistence.EntityGraph;
import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.EntityTransaction;
import javax.persistence.FlushModeType;
import javax.persistence.LockModeType;
import javax.persistence.Query;
import javax.persistence.StoredProcedureQuery;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaDelete;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.CriteriaUpdate;
import javax.persistence.metamodel.Metamodel;

import org.apache.log4j.Logger;

import com.clustercontrol.cloud.Filter;
import com.clustercontrol.commons.util.JpaTransactionManager;


/**
 * PersistenceManager の内部実装。
 * 
 * @author torun
 *
 */
public class StdEntityManagerEx implements EntityManagerEx {
	private EntityManager entityManager;

	public StdEntityManagerEx() {
		JpaTransactionManager tm = new JpaTransactionManager(); 
		this.entityManager = tm.getEntityManager();
	}

	public StdEntityManagerEx(EntityManager entityManager) {
		this.entityManager = entityManager;
	}
	
	public StdEntityManagerEx(JpaTransactionManager tm) {
		this.entityManager = tm.getEntityManager();
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T> List<T> findAll(Class<T> type) {
		Logger logger = Logger.getLogger(StdEntityManagerEx.class);
		logger.debug("findAll() start");
		
		try {
			Query q = entityManager.createQuery("select c from " + type.getSimpleName() + " c");
			return (List<T>)q.getResultList();
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}	
	
	@SuppressWarnings("unchecked")
	@Override
	public <T> List<T> findByFilter(Class<T> type, Filter... filters) {
		Logger logger = Logger.getLogger(StdEntityManagerEx.class);
		logger.debug("findByFilter() start");
		
		StringBuilder sb = new StringBuilder();
		sb.append("select c from ");
		sb.append(type.getSimpleName());
		sb.append(" c where ");
		
		boolean subsequent = false;
		for (Filter f: filters) {
			if (subsequent) {
				sb.append(" and ");
			}
			
			if (f.getValues().size() > 1) {
				sb.append('(');
			}
			boolean subsequent2 = false;
			for (String v: f.getValues()) {
				if (subsequent2) {
					sb.append(" or ");
				}
				sb.append("c.")
					.append(f.getName())
					.append(" like ")
					.append("'" + v + "'");
				subsequent2 = true;
			}
			if (f.getValues().size() > 1) {
				sb.append(')');
			}
			
			subsequent = true;
		}
		
		Query q = entityManager.createQuery(sb.toString());
		
		List<?> result = q.getResultList();
		if (result != null) {
			return (List<T>)result;
		}
		else {
			return Collections.emptyList();
		}
	}

	@Override
	public void persist(Object entity) {
		entityManager.persist(entity);
	}

	@Override
	public void persist(IDHolder entity) {
		Object managed = this.find(entity.getClass(), entity.getId());
		if (managed == null) {
			entityManager.persist(entity);
		}
		else {
			// 重複エラー
			throw new EntityExistsException("{" + entity.getClass().getSimpleName() + ", " + entity.getId().toString() + "}");
		}
	}

	public <T> T merge(T entity) {
		return entityManager.merge(entity);
	}

	public void remove(Object entity) {
		entityManager.remove(entity);
	}

	public <T> T find(Class<T> entityClass, Object primaryKey) {
		return entityManager.find(entityClass, primaryKey);
	}

	public <T> T find(Class<T> entityClass, Object primaryKey,
			Map<String, Object> properties) {
		return entityManager.find(entityClass, primaryKey, properties);
	}

	public <T> T find(Class<T> entityClass, Object primaryKey,
			LockModeType lockMode) {
		return entityManager.find(entityClass, primaryKey, lockMode);
	}

	public <T> T find(Class<T> entityClass, Object primaryKey,
			LockModeType lockMode, Map<String, Object> properties) {
		return entityManager
				.find(entityClass, primaryKey, lockMode, properties);
	}

	public <T> T getReference(Class<T> entityClass, Object primaryKey) {
		return entityManager.getReference(entityClass, primaryKey);
	}

	public void flush() {
		entityManager.flush();
	}

	public void setFlushMode(FlushModeType flushMode) {
		entityManager.setFlushMode(flushMode);
	}

	public FlushModeType getFlushMode() {
		return entityManager.getFlushMode();
	}

	public void lock(Object entity, LockModeType lockMode) {
		entityManager.lock(entity, lockMode);
	}

	public void lock(Object entity, LockModeType lockMode,
			Map<String, Object> properties) {
		entityManager.lock(entity, lockMode, properties);
	}

	public void refresh(Object entity) {
		entityManager.refresh(entity);
	}

	public void refresh(Object entity, Map<String, Object> properties) {
		entityManager.refresh(entity, properties);
	}

	public void refresh(Object entity, LockModeType lockMode) {
		entityManager.refresh(entity, lockMode);
	}

	public void refresh(Object entity, LockModeType lockMode,
			Map<String, Object> properties) {
		entityManager.refresh(entity, lockMode, properties);
	}

	public void clear() {
		entityManager.clear();
	}

	public void detach(Object entity) {
		entityManager.detach(entity);
	}

	public boolean contains(Object entity) {
		return entityManager.contains(entity);
	}

	public LockModeType getLockMode(Object entity) {
		return entityManager.getLockMode(entity);
	}

	public void setProperty(String propertyName, Object value) {
		entityManager.setProperty(propertyName, value);
	}

	public Map<String, Object> getProperties() {
		return entityManager.getProperties();
	}

	public Query createQuery(String qlString) {
		return entityManager.createQuery(qlString);
	}

	public <T> TypedQuery<T> createQuery(CriteriaQuery<T> criteriaQuery) {
		return entityManager.createQuery(criteriaQuery);
	}

	public <T> TypedQuery<T> createQuery(String qlString, Class<T> resultClass) {
		return entityManager.createQuery(qlString, resultClass);
	}

	public Query createNamedQuery(String name) {
		return entityManager.createNamedQuery(name);
	}

	public <T> TypedQuery<T> createNamedQuery(String name, Class<T> resultClass) {
		return entityManager.createNamedQuery(name, resultClass);
	}

	public Query createNativeQuery(String sqlString) {
		return entityManager.createNativeQuery(sqlString);
	}

	@SuppressWarnings("rawtypes")
	public Query createNativeQuery(String sqlString, Class resultClass) {
		return entityManager.createNativeQuery(sqlString, resultClass);
	}

	public Query createNativeQuery(String sqlString, String resultSetMapping) {
		return entityManager.createNativeQuery(sqlString, resultSetMapping);
	}

	public void joinTransaction() {
		entityManager.joinTransaction();
	}

	public <T> T unwrap(Class<T> cls) {
		return entityManager.unwrap(cls);
	}

	public Object getDelegate() {
		return entityManager.getDelegate();
	}

	public void close() {
		entityManager.close();
	}

	public boolean isOpen() {
		return entityManager.isOpen();
	}

	public EntityTransaction getTransaction() {
		return entityManager.getTransaction();
	}

	public EntityManagerFactory getEntityManagerFactory() {
		return entityManager.getEntityManagerFactory();
	}

	public CriteriaBuilder getCriteriaBuilder() {
		return entityManager.getCriteriaBuilder();
	}

	public Metamodel getMetamodel() {
		return entityManager.getMetamodel();
	}

	@Override
	public Query createQuery(@SuppressWarnings("rawtypes") CriteriaUpdate updateQuery) {
		return entityManager.createQuery(updateQuery);
	}

	@Override
	public Query createQuery(@SuppressWarnings("rawtypes") CriteriaDelete deleteQuery) {
		return entityManager.createQuery(deleteQuery);
	}

	@Override
	public StoredProcedureQuery createNamedStoredProcedureQuery(String name) {
		return entityManager.createNamedStoredProcedureQuery(name);
	}

	@Override
	public StoredProcedureQuery createStoredProcedureQuery(String procedureName) {
		return entityManager.createStoredProcedureQuery(procedureName);
	}

	@Override
	public StoredProcedureQuery createStoredProcedureQuery(String procedureName, @SuppressWarnings("rawtypes") Class... resultClasses) {
		return entityManager.createStoredProcedureQuery(procedureName, resultClasses);
	}

	@Override
	public StoredProcedureQuery createStoredProcedureQuery(String procedureName, String... resultSetMappings) {
		return entityManager.createStoredProcedureQuery(procedureName, resultSetMappings);
	}

	@Override
	public boolean isJoinedToTransaction() {
		return entityManager.isJoinedToTransaction();
	}

	@Override
	public <T> EntityGraph<T> createEntityGraph(Class<T> rootType) {
		return entityManager.createEntityGraph(rootType);
	}

	@Override
	public EntityGraph<?> createEntityGraph(String graphName) {
		return entityManager.createEntityGraph(graphName);
	}

	@Override
	public EntityGraph<?> getEntityGraph(String graphName) {
		return entityManager.getEntityGraph(graphName);
	}

	@Override
	public <T> List<EntityGraph<? super T>> getEntityGraphs(Class<T> entityClass) {
		return entityManager.getEntityGraphs(entityClass);
	}
}