/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.cloudn.rest.api;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import org.apache.commons.codec.binary.Base64;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Collections2;

public class ParamHolder {
	private static final String HMAC_SHA1_ALGORITHM = "HmacSHA1";
	
//	private static final SimpleDateFormat sdf = new SimpleDateFormat("YYYY-MM-DD'T'hh:mm:ssZ");

	public static class Param {
		public Param(String name, String param) {
			Preconditions.checkNotNull(name);
			Preconditions.checkNotNull(param);
			this.name = name;
			this.param = param;
		}
		public final String name;
		public final String param;
		
		public static ParamHolder.Param build(String name, String param) {
			return new Param(name, param);
		}
	}
	
	private List<ParamHolder.Param> params = new ArrayList<>();
	
	private ParamHolder() {
	}
	public ParamHolder with(String name, String value) {
		params.add(new Param(name, value));
		return this;
	}
	public ParamHolder withs(ParamHolder.Param...p) {
		withs(Arrays.asList(p));
		return this;
	}
	public ParamHolder withs(Collection<ParamHolder.Param> ps) {
		params.addAll(ps);
		return this;
	}
	public ParamHolder with(ParamHolder holder) {
		withs(holder.params);
		return this;
	}
	public static ParamHolder create() {
		return new ParamHolder();
	}
	public String toString() {
		return toString(params);
	}
	public String toParameterString(String secretKey) {
		List<ParamHolder.Param> params = new ArrayList<>(this.params);
		
		Calendar c = Calendar.getInstance();
		c.add(Calendar.MINUTE, 5);
		
//		SimpleDateFormat sdf = new SimpleDateFormat("YYYY-MM-DD'T'hh:mm:ssZ");
//		params.add(ParamHolder.Param.build("expires", sdf.format(c.getTime())));

		int zoneOffset = c.get(Calendar.ZONE_OFFSET) / 1000 / 60;
		params.add(ParamHolder.Param.build("expires", String.format("%04d-%02d-%02dT%02d:%02d:%02d%s%02d%02d",
				c.get(Calendar.YEAR),
				c.get(Calendar.MONTH) + 1,
				c.get(Calendar.DAY_OF_MONTH),
				c.get(Calendar.HOUR_OF_DAY),
				c.get(Calendar.MINUTE),
				c.get(Calendar.SECOND),
				zoneOffset >= 0 ? '+': '-',
				Math.abs(zoneOffset / 60),
				zoneOffset % 60
				)));
		params.add(ParamHolder.Param.build("signatureVersion", "3"));
//		params.add(ParamHolder.Param.build("response", "json"));

		Collections.sort(params, new Comparator<ParamHolder.Param>() {
			@Override
			public int compare(ParamHolder.Param o1, ParamHolder.Param o2) {
				return o1.name.toLowerCase().compareTo(o2.name.toLowerCase());
			}
		});
		
		List<ParamHolder.Param> encoded = new ArrayList<>(Collections2.transform(params, new Function<ParamHolder.Param, ParamHolder.Param>() {
			@Override
			public ParamHolder.Param apply(ParamHolder.Param input) {
				try {
					return new Param(input.name, URLEncoder.encode(input.param, "UTF-8"));
				}
				catch (UnsupportedEncodingException e) {
					throw CloudnRestfulApplicationError.Params_FailToMakeParamString.exceptionException(e, input.param);
				}
			}
		}));

		List<ParamHolder.Param> hash = new ArrayList<>(Collections2.transform(encoded, new Function<ParamHolder.Param, ParamHolder.Param>() {
			@Override
			public ParamHolder.Param apply(ParamHolder.Param input) {
				return new Param(input.name.toLowerCase(), input.param.toLowerCase().replace("+", "%20"));
			}
		}));
		return ParamHolder.create().withs(encoded).with(RestApplicationBuilder.COMMON_PARAM_SIGNAURE, createSignature(toString(hash), secretKey)).toString();
	}
	public String toRequest(String endpointUrl, String secretKey) {
		return endpointUrl + "?" + toParameterString(secretKey);
	}
	private static String createSignature(String parameters, String secretKey) {
		try {
			SecretKeySpec signingKey = new SecretKeySpec(secretKey.getBytes(), HMAC_SHA1_ALGORITHM);
			Mac mac = Mac.getInstance(HMAC_SHA1_ALGORITHM);
			mac.init(signingKey);
			return URLEncoder.encode(Base64.encodeBase64String(mac.doFinal(parameters.getBytes())), "UTF-8");
		} catch (NoSuchAlgorithmException | InvalidKeyException | UnsupportedEncodingException | IllegalStateException e) {
			throw CloudnRestfulApplicationError.Signature_FailToMakeSignature.exceptionException(e, parameters);
		}
	}
	private static String toString(List<ParamHolder.Param> params) {
		StringBuffer strBuffer = new StringBuffer();
		boolean first = false;
		for (ParamHolder.Param param: params) {
			if (first) {
				strBuffer.append("&");	
			} else { 
				first = true;
			}
			strBuffer.append(param.name).append("=").append(param.param);
		}
		return strBuffer.toString();
	}
}