/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.cloudn.util;

import java.net.Inet4Address;
import java.net.UnknownHostException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Cider {
	private static final Pattern ciderPattern = Pattern.compile("^(\\d{1,3}+)\\.(\\d{1,3}+)\\.(\\d{1,3}+)\\.(\\d{1,3}+)/(\\d{1,2}+)$");
	private static final Pattern ipPattern = Pattern.compile("^(\\d{1,3}+)\\.(\\d{1,3}+)\\.(\\d{1,3}+)\\.(\\d{1,3}+)$");

	private int network;
	private int mask;

	public Cider(String cider) throws UnknownHostException {
		Matcher m = ciderPattern.matcher(cider);
		if (!m.matches()) 
			throw new UnknownHostException("cider is of illegal format");
		network = ofIpaddress(new byte[]{toByte(m.group(1)), toByte(m.group(2)), toByte(m.group(3)), toByte(m.group(4))});
		mask = ofMask(Byte.parseByte(m.group(5)));
	}
	
	private static int ofIpaddress(Inet4Address ipaddress) {
		return ofIpaddress(ipaddress.getAddress());
	}

	private static int ofIpaddress(byte[] bytes) {
        int address = bytes[3] & 0xFF;
        address |= ((bytes[2] << 8) & 0xFF00);
        address |= ((bytes[1] << 16) & 0xFF0000);
        address |= ((bytes[0] << 24) & 0xFF000000);
		return address;
	}
	
	private static int ofMask(byte mask) {
		byte[] masks = new byte[4];
		for (int i = 0; i < 4; ++i) {
			int check = (mask - 8 * (i + 1));
			masks[i] = (byte)(check > 0 ? 0xFF: makeBitPattern(8 + check));
			if (check < 0) {
				break;
			}
		}
		return ofIpaddress(masks);
	}
	
	private static byte toByte(String value) {
		int intValue = Integer.parseInt(value);
		if (intValue > 127) {
			return (byte)(intValue - 256);
		}
		return (byte)intValue;
	}

	private static byte makeBitPattern(int check) {
		byte result = 0;
		for (int i = 0; i < Math.min(check, 8); ++i) {
			result |= (1 << (8 - i - 1));
		}
		return result;
	}

	public boolean matches(Inet4Address addr) {
		return (ofIpaddress(addr) & mask) == network;
	}

	public boolean matches(String addr) /* throws UnknownHostException */ {
		Matcher m = ipPattern.matcher(addr);
		if (!m.matches()) 
			return false;
//			throw new UnknownHostException("addr is of illegal format");
		return (ofIpaddress(new byte[]{toByte(m.group(1)), toByte(m.group(2)), toByte(m.group(3)), toByte(m.group(4))}) & mask) == network;
	}
}