/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.bean;

import java.io.Serializable;
import java.sql.Timestamp;

import javax.xml.bind.annotation.XmlTransient;

import com.clustercontrol.cloud.dao.CloudUserDao;
import com.clustercontrol.cloud.dao.CloudUserDao.CloudUserType;
import com.clustercontrol.cloud.validation.ValidationConstants;
import com.clustercontrol.cloud.validation.annotation.Identity;
import com.clustercontrol.cloud.validation.annotation.NotNull;
import com.clustercontrol.cloud.validation.annotation.ReadOnly;
import com.clustercontrol.cloud.validation.annotation.Size;

/**
 * クラウドユーザー情報を保持するクラス。 
 * {@link com.clustercontrol.ws.cloud.CloudEndpoint#addCloudUser(CloudUser cloudUser) addCloudUser 関数}、
 * {@link com.clustercontrol.ws.cloud.CloudEndpoint#modifyCloudUser(CloudUser cloudUser) modifyCloudUser 関数} にて使用される。
 *
 */
public class CloudUser implements Serializable {
	/**
	 * 
	 */
	private static final long serialVersionUID = -9161813048730684001L;
	
	private CloudUserDao dao;
	private String accountResourceId;

	public CloudUser() {
		super();
		this.dao = new CloudUserDao();
	}

	public CloudUser(CloudUserDao dao) {
		super();
		this.dao = new CloudUserDao(dao);
		this.accountResourceId = dao.getAccountResource().getAccountResourceId();
	}
	/**
	 * クラウドユーザー Id を取得する。
	 * 
	 * @return クラウドユーザー Id
	 */
	@Identity
	public String getCloudUserId() {
		return dao.getCloudUserId();
	}
	/**
	 * クラウドユーザー Id を指定します。
	 * クラウドユーザーの作成および変更時に、アルファベット、数字、"-"、"_" にて構成される 64 文字以内の文字列を指定する必要がある。
	 * 
	 * @param cloudUserId　クラウドユーザー Id。
	 */
	public void setCloudUserId(String cloudUserId) {
		dao.setCloudUserId(cloudUserId);
	}
	/**
	 * クラウドユーザー名を取得します。
	 * 
	 * @return クラウドユーザー名。
	 */
	@Size(max=128)
	@NotNull(groups=ValidationConstants.group_creation)
	public String getCloudUserName() {
		return dao.getCloudUserName();
	}
	/**
	 * クラウドユーザー名を指定します。
	 * 作成時には128 字以内の文字列を指定する必要があります。編集時には、null を指定した場合は、既存の値が維持されます。
	 * 
	 * @param cloudUserName　クラウドユーザー名。
	 */
	public void setCloudUserName(String cloudUserName) {
		dao.setCloudUserName(cloudUserName);
	}
	/**
	 * AWS アカウントあるいは、IAM ユーザーのシークレットキーを取得します。
	 * 
	 * @return　シークレットキー。
	 */
	@NotNull(groups=ValidationConstants.group_creation)
	public String getSecretKey() {
		return dao.getSecretKey();
	}
	/**
	 * AWS アカウントあるいは、IAM ユーザーのシークレットキーを指定します。
	 * 作成時には必ず指定する必要があります。編集時には、null を指定した場合は、既存の値が維持されます。
	 * 
	 * @param secretKey　シークレットキー。
	 */
	public void setSecretKey(String secretKey) {
		dao.setSecretKey(secretKey);
	}
	/**
	 * AWS アカウントあるいは、IAM ユーザーのアクセスキーを取得します。
	 * 
	 * @return　アクセスキー。
	 */
	@NotNull(groups=ValidationConstants.group_creation)
	public String getAccessKey() {
		return dao.getAccessKey();
	}
	/**
	 * AWS アカウントあるいは、IAM ユーザーのアクセスキーを指定します。
	 * 作成時には必ず指定する必要があります。編集時には、null を指定した場合は、既存の値が維持されます。
	 * 
	 * @param accessKey　アクセスキー。
	 */
	public void setAccessKey(String accessKey) {
		dao.setAccessKey(accessKey);
	}

	/**
	 * クラウド Id を取得します。現在は、"AWS" のみです。
	 * 
	 * @return クラウド Id
	 */
	@Identity(groups=ValidationConstants.group_creation)
	public String getAccountResourceId() {
		return accountResourceId;
	}

	/**
	 * クラウド Id を指定します。参照時のみ有効となるので使用しません。
	 * 
	 * @return
	 */
	public void setAccountResourceId(String accountResourceId) {
		this.accountResourceId = accountResourceId;
	}
	/**
	 * 説明を取得します。
	 * 
	 * @return　説明。
	 */
	@Size(max = 256)
	@NotNull
	public String getDescription() {
		return dao.getDescription();
	}
	/**
	 * 説明を指定します。
	 * 作成時、編集時に 256 文字以内の文字列を指定する必要があります。編集時には、null を指定した場合は、既存の値が維持されます。
	 * 
	 * @param description　説明。
	 */
	public void setDescription(String description) {
		dao.setDescription(description);
	}
	/**
	 * クラウドユーザー種別を取得します。
	 * 
	 * @return クラウドユーザー種別。
	 */
	@NotNull(groups=ValidationConstants.group_creation)
	public CloudUserType getCloudUserType() {
		if (dao.getCloudUserType() != null) {
			return CloudUserType.valueOf(dao.getCloudUserType().name());
		}
		return null;
	}
	/**
	 * クラウドユーザー種別を指定します。
	 * AWS アカウントは親 (CloudUserType.parent)、IAM ユーザーは子(CloudUserType.child)を指定します。
	 * 作成時のみ使用します。
	 * 
	 * @param cloudUserType　クラウドユーザー種別。
	 */
	public void setCloudUserType(CloudUserType cloudUserType) {
		if (cloudUserType != null) {
			dao.setCloudUserType(CloudUserDao.CloudUserType.valueOf(cloudUserType.name()));
		}
		else {
			dao.setCloudUserType(null);
		}
	}
	/**
	 * クラウドユーザーに紐づく Hinemos ユーザーを取得します。
	 * 
	 * @return Hinemos ユーザー。
	 */
	@Identity(groups=ValidationConstants.group_creation)
	public String getRoleId() {
		return dao.getRoleId();
	}
	/**
	 * クラウドユーザーに紐づく Hinemos ユーザーを指定します。
	 * 作成時のみ使用します。
	 * 
	 * @param userId　Hinemos ユーザー。
	 */
	public void setRoleId(String roleId) {
		dao.setRoleId(roleId);
	}
	/**
	 * 登録日を取得します。
	 * 
	 * @return 登録日。
	 */
	@ReadOnly
	public Long getRegDate() {
		return dao.getRegDate() != null ? dao.getRegDate().getTime(): null;
	}

	/**
	 * 登録日を指定します。参照時のみ有効となるので使用しません。
	 * 
	 * @param regDate 登録日。
	 */
	public void setRegDate(Long regDate) {
		if (regDate == null) {
			dao.setRegDate(null);
		}
		else {
			dao.setRegDate(new Timestamp(regDate));
		}
	}

	/**
	 * 更新日を取得します。
	 * 
	 * @return 更新日。
	 */
	@ReadOnly
	public Long getUpdateDate() {
		return dao.getUpdateDate() != null ? dao.getUpdateDate().getTime(): null;

	}

	/**
	 * 更新日を指定します。参照時のみ有効となるので使用しません。
	 * 
	 * @param updateDate 更新日。
	 */
	public void setUpdateDate(Long updateDate) {
		if (updateDate == null) {
			dao.setUpdateDate(null);
		}
		else {
			dao.setUpdateDate(new Timestamp(updateDate));
		}
	}
	/**
	 * 登録ユーザーを取得します。
	 * 
	 * @return 登録ユーザー。
	 */
	@ReadOnly
	public String getRegUser() {
		return dao.getRegUser();
	}

	/**
	 * 登録ユーザーを指定します。参照時のみ有効となるので使用しません。
	 * 
	 * @param regUser 登録ユーザー。
	 */
	public void setRegUser(String regUser) {
		dao.setRegUser(regUser);
	}

	/**
	 * 更新ユーザーを取得します。
	 * 
	 * @return 更新ユーザー。
	 */
	@ReadOnly
	public String getUpdateUser() {
		return dao.getUpdateUser();
	}

	/**
	 * 更新ユーザーを指定します。参照時のみ有効となるので使用しません。
	 * 
	 * @param updateUser 更新ユーザー。
	 */
	public void setUpdateUser(String updateUser) {
		dao.setUpdateUser(updateUser);
	}

	@XmlTransient
	public CloudUserDao getTableData() {
		return dao;
	}
}