/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.factory;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.persistence.EntityExistsException;
import javax.persistence.Query;

import org.apache.log4j.Logger;

import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.ErrorCode;
import com.clustercontrol.cloud.Filter;
import com.clustercontrol.cloud.IResourceManagement;
import com.clustercontrol.cloud.IResourceManagement.Storage;
import com.clustercontrol.cloud.IResourceManagement.StorageBackup;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.bean.CloudInstance;
import com.clustercontrol.cloud.bean.CloudStorage;
import com.clustercontrol.cloud.bean.CloudStorageBackup;
import com.clustercontrol.cloud.bean.CreateStorageBackupRequest;
import com.clustercontrol.cloud.bean.CreateStorageRequest;
import com.clustercontrol.cloud.bean.RestoreStatus;
import com.clustercontrol.cloud.bean.RestoreStorageRequest;
import com.clustercontrol.cloud.bean.StorageStateKind;
import com.clustercontrol.cloud.commons.CloudPropertyConstants;
import com.clustercontrol.cloud.commons.PropertyContract;
import com.clustercontrol.cloud.dao.CloudStorageBackupDao;
import com.clustercontrol.cloud.dao.CloudStorageDao;
import com.clustercontrol.cloud.factory.IStorageOperator.UpdateAllResult.StorageMapping;
import com.clustercontrol.cloud.persistence.EntityManagerEx;
import com.clustercontrol.cloud.persistence.PersistenceUtil.TransactionScope;
import com.clustercontrol.cloud.persistence.Transactional;
import com.clustercontrol.cloud.util.CloudMessageUtil;
import com.clustercontrol.cloud.util.RepositoryControllerBeanWrapper;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.repository.bean.NodeDiskInfo;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.session.RepositoryControllerBean;

@Transactional
public class StorageOperator extends ResourceOperatorBase implements IStorageOperator {
	private boolean update = CloudPropertyConstants.autoupdate_storage.match(PropertyContract.on);
	private boolean regist = CloudPropertyConstants.autoregist_storage.match(PropertyContract.on);
	private boolean mount = CloudPropertyConstants.autoupdate_mount.match(PropertyContract.on);

	@Transactional(Transactional.TransactionType.Supported)
	public void setFlags(boolean update, boolean regist, boolean mount) {
		this.update = update;
		this.regist = regist;
		this.mount = mount;
	}

	@Override
	public CloudStorage createStorage(CreateStorageRequest request) throws CloudManagerFault, InvalidRole {
		Storage storage = getResourceManagement().createStorage(
						request.getStorageName(),
						request.getFlavor(),
						request.getStorageSize(),
						request.getSnapshotId(),
						request.getZone(),
						request.getStorageDetail());

		// この段階では、ノードが保持するディスク情報との関連が無いので、Hinemos DB には登録しない。
		CloudStorageDao cloudStorage = createCloudStorageDao(storage);
		
		return new CloudStorage(cloudStorage, storage, null);
	}
	
	@Override
	public CloudStorageBackup createStorageBackup(CreateStorageBackupRequest request) throws CloudManagerFault {
		StorageBackup storageBackup = getResourceManagement().createStorageBackup(request.getStorageId(), request.getSnapshotName(), request.getDescription(), request.getSnapshotDetail());
		
		CloudStorageBackupDao dao = new CloudStorageBackupDao();
		dao.setStorageBackupId(storageBackup.getStorageBackupId());
		dao.setCloudServiceId(getAccountResource().getCloudServiceId());
		dao.setRegion(getCloudRegion().getRegion());
		dao.setAccountResourceId(getAccountResource().getAccountResourceId());
		dao.setSnapshotName(storageBackup.getName());
		dao.setCloudUserId(getCloudUser().getCloudUserId());
		dao.setDescription(request.getDescription());
		dao.setRestoreStatus(CloudStorageBackupDao.RestoreStatus.valueOf(RestoreStatus.available.name()));

		CloudStorageBackupDao.BackupedData backup = new CloudStorageBackupDao.BackupedData();
		backup.setStorageId(storageBackup.getStorageId());
		
		dao.setBackupedData(backup);
		
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		em.persist(dao);

		return new CloudStorageBackup(dao, storageBackup);
	}
	
	public CloudStorage restoreStorage(RestoreStorageRequest request) throws CloudManagerFault, InvalidRole {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudStorageBackupDao backup = em.find(CloudStorageBackupDao.class, new CloudStorageBackupDao.CloudStorageBackupPK(request.getStorageBackupId(), getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (backup == null) {
			throw ErrorCode.CLOUDSTORAGEBACKUP_INVALID_CLOUDSTORAGEBACKUP_NOT_FOUND.cloudManagerFault(request.getStorageBackupId());
		}
		
		Storage storage = getResourceManagement().restoreStorage(request.getStorageBackupId(), request.getStorageName(), request.getFlavor(), request.getStorageSize(), request.getFlavor(), request.getDetail());

		// この段階では、ノードが保持するディスク情報との関連が無いので、Hinemos DB には登録しない。
		CloudStorageDao cloudStorage = createCloudStorageDao(storage);
		
		return new CloudStorage(cloudStorage, storage, null);
	}

	@Override
	public void removeStorage(final String storageId) throws CloudManagerFault, InvalidRole {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudStorageDao storageDao = em.find(CloudStorageDao.class, new CloudStorageDao.CloudStoragePK(storageId, getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (storageDao == null) {
			try {
				getResourceManagement().deleteStorage(storageId);
			}
			catch (Exception e) {
				getLogger().warn(e.getMessage(), e);
			}
			throw ErrorCode.CLOUDSTORAGE_INVALID_CLOUDSTORAGE_NOT_FOUND.cloudManagerFault(storageId);
		}
		else {
			removeStorageDao(storageDao);

			try {
				removeDiskInfo(storageDao.getFacilityId(), storageId);
			}
			catch (FacilityNotFound e) {
				// facility がない場合は、削除続行。
			}

			switch (storageDao.getRegistStatus()) {
			case exist:
				getResourceManagement().deleteStorage(storageId);
				break;
			case deleted:
				break;
			}
		}
	}

	@Override
	public void removeStorageBackup(String storageBackupId) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudStorageBackupDao storageBackupDao = em.find(CloudStorageBackupDao.class, new CloudStorageBackupDao.CloudStorageBackupPK(storageBackupId, getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (storageBackupDao == null) {
			try {
				getResourceManagement().deleteStorageBackup(storageBackupId);
			}
			catch (Exception e) {
				getLogger().warn(e.getMessage(), e);
			}
			throw ErrorCode.CLOUDSTORAGEBACKUP_INVALID_CLOUDSTORAGEBACKUP_NOT_FOUND.cloudManagerFault(storageBackupId);
		}
		else {
			em.remove(storageBackupDao);

			switch (storageBackupDao.getRestoreStatus()) {
			case available:
				getResourceManagement().deleteStorageBackup(storageBackupId);
				break;
			case unavailable:
				break;
			}
		}
	}
	
	@Override
	public UpdateResult updateStorage(String storageId) throws CloudManagerFault, InvalidRole {
		IResourceManagement rm = getResourceManagement();

		//　まず、Hinemos DB からクラウド情報を取得。
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudStorageDao cloudStorageDao = em.find(CloudStorageDao.class, new CloudStorageDao.CloudStoragePK(storageId, getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));

		Storage s = null;
		if (cloudStorageDao == null || cloudStorageDao.getRegistStatus() != CloudStorageDao.StorageStatus.deleted) {
			s = rm.getStorage(storageId);
			
			if (StorageStateKind.deleting.equals(s.getState())) {
				s = null;
			}
		}

		UpdateResult result = internalUpdate(getCloudRegion().getRegion(), cloudStorageDao, s);
		
		return result;
	}
	
	private UpdateResult transactionalInternalUpdate(String region, CloudStorageDao cloudStorageDao, Storage storage) throws CloudManagerFault, InvalidRole {
		UpdateResult result = null;
		try (TransactionScope scope = new TransactionScope(Transactional.TransactionType.RequiredNew)) {
			EntityManagerEx em = SessionService.current().getEntityManagerEx();
			if (cloudStorageDao != null) {
				//　EntityManager が切り替わっているので、再度取得。
				cloudStorageDao = em.find(CloudStorageDao.class, cloudStorageDao.getId());
			}
			result = internalUpdate(region, cloudStorageDao, storage);
			scope.complete();
		}
		return result;
	}
	
	private UpdateResult internalUpdate(String region, CloudStorageDao cloudStorageDao, Storage storage) throws CloudManagerFault, InvalidRole {
		try {
			if (storage != null) {
				if (cloudStorageDao == null) {
					if (update) {
						try {
							// AWS ボリュームが追加。
							cloudStorageDao = createCloudStorageDao(storage);
						}
						catch (CloudManagerFault e) {
							// 別の処理が作成済みらしい。
							if (ErrorCode.CLOUDSTORAGE_ALREADY_EXIST.match(e)) {
								EntityManagerEx em = SessionService.current().getEntityManagerEx();
								cloudStorageDao = em.find(CloudStorageDao.class, new CloudStorageDao.CloudStoragePK(storage.getStorageId(), region, getCloudUser().getAccountResourceId()));
								if (cloudStorageDao == null) {
									throw e;
								}
								return new UpdateResult(getCloudService().getCloudTypeId(), storage, cloudStorageDao);
							}
							throw e;
						}
	
						// インスタンスに対するノードのようなストレージに対するオブジェクトが存在しないので、以下のフラグは無視。
						if (regist) {
						}
						else {
						}
	
						if (mount) {
							addStorageRelation(region, cloudStorageDao, storage);
						}
						else {
							// マウントに関わる情報は無視。
						}
					}
					else {
					}
				}
				else {
					if (update) {
						// 登録済みの情報を更新。
						updateCloudStorageDao(cloudStorageDao, storage);
					}
					else {
					}
					
					if (mount) {
						updateStorageRelation(region, cloudStorageDao, storage);
					}
					else {
						// マウントに関わる情報は無視。
					}
				}
			}
			else {
				if (cloudStorageDao != null) {
					if (update) {
						if (cloudStorageDao.getRegistStatus() != CloudStorageDao.StorageStatus.deleted) {
							cloudStorageDao.setRegistStatus(CloudStorageDao.StorageStatus.deleted);
						}
					}
					
					if (mount) {
						removeStorageRelation(cloudStorageDao);
					}
					else {
					}
					
					// インスタンスに対するノードのようなストレージに対するオブジェクトが存在しないので、以下のフラグは無視。
					if (regist) {
					}
					else {
					}
	
					// ディスク情報が無い場合は、削除。
					if (cloudStorageDao.getFacilityId() == null) {
						removeStorageDao(cloudStorageDao);
						cloudStorageDao = null;
					}
				}
				else {
					// どちらもないのでスルー
				}
			}
		}
		catch (CloudManagerFault e) {
			CloudMessageUtil.notify_AutoUpadate_Error_StorageOperator(cloudStorageDao.getAccountResourceId(), cloudStorageDao.getStorageId(), e);
		}
		return new UpdateResult(getCloudService().getCloudTypeId(), storage, cloudStorageDao);
	}

	private void updateCloudStorageDao(CloudStorageDao storageDao, Storage storage) throws CloudManagerFault {
		if (storage.getName() != null) {
			if (!storage.getName().equals(storageDao.getStorageName())) {
				storageDao.setStorageName(storage.getName());
			}
		}
		else {
			if (storageDao.getStorageName() != null) {
				storageDao.setStorageName(null);
			}
		}
	}
	
	private void updateStorageRelation(String Region, CloudStorageDao cloudStorageDao, Storage storage) throws CloudManagerFault, InvalidRole {
		if (storage.getStorageAttachment() == null) {
			removeStorageRelation(cloudStorageDao);
		}
		else {
			addStorageRelation(Region, cloudStorageDao, storage);
		}
	}

	private CloudStorageDao createCloudStorageDao(Storage storage) throws CloudManagerFault, InvalidRole {
		// DB に追加する情報を作成。
		CloudStorageDao dao = new CloudStorageDao();
		dao.setStorageId(storage.getStorageId());
		dao.setRegion(getCloudRegion().getRegion());
		dao.setCloudServiceId(getAccountResource().getCloudServiceId());
		dao.setAccountResourceId(getCloudUser().getAccountResourceId());
		dao.setStorageName(storage.getName());
		dao.setZone(storage.getZone());
		dao.setFlavor(storage.getFlavor());
		dao.setRegistStatus(CloudStorageDao.StorageStatus.exist);
		
		CloudStorage cloudStorage = new UpdateResult(null, storage, dao).getCloudStorage();
		try (AddedEventNotifier<CloudStorage> notifier = new AddedEventNotifier<>(null, CloudStorage.class, cloudStorage)) {
			try {
				EntityManagerEx em = SessionService.current().getEntityManagerEx();
				em.persist(dao);
				notifier.setCompleted();
				return dao;
			}
			catch (EntityExistsException e) {
				throw ErrorCode.CLOUDSTORAGE_ALREADY_EXIST.cloudManagerFault(dao.getStorageId());
			}
		}
	}
	
	private void removeStorageDao(CloudStorageDao cloudStorageDao) throws CloudManagerFault, InvalidRole {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudStorage cloudStorage = new UpdateResult(null, null, cloudStorageDao).getCloudStorage();
		try (RemovedEventNotifier<CloudStorage> notifier = new RemovedEventNotifier<>(null, CloudStorage.class, cloudStorage)) {
			em.remove(cloudStorageDao);
			notifier.setCompleted();
		}
	}
	
	@Override
	public UpdateAllResult updateAllStorage() throws CloudManagerFault, InvalidRole {
		getLogger().debug("enter updateAll()");

		IResourceManagement rm = getResourceManagement();
		List<Storage> result = rm.getStorages();
		
		// まず、リージョンでフィルターした情報を取得し、AWS と比較して、最新に更新する。
		Filter filter = new Filter("region", getCloudRegion().getRegion());
		Filter filter2 = new Filter("accountResourceId", getCloudUser().getAccountResourceId());
		List<CloudStorageDao> csList = new ArrayList<CloudStorageDao>(SessionService.current().getEntityManagerEx().findByFilter(CloudStorageDao.class, filter, filter2));
		
		// AWS と Hinemos の情報の差分を取得。
		List<Storage> sList = new ArrayList<>(result);
		Iterator<Storage> sIter = sList.iterator();
		List<StorageMapping> storageMappings = new ArrayList<StorageMapping>();
		while (sIter.hasNext()) {
			Storage s = sIter.next();
			if (StorageStateKind.deleting == s.getState()) {
				sIter.remove();
				continue;
			}
			
			Iterator<CloudStorageDao> csIter = csList.iterator();

			while (csIter.hasNext()) {
				CloudStorageDao cs = csIter.next();
				
				if (s.getStorageId().equals(cs.getStorageId())) {
					storageMappings.add(new StorageMapping(s, cs));
					
					sIter.remove();
					csIter.remove();
					break;
				}
			}
		}

		List<StorageMapping> both = new ArrayList<StorageMapping>();
		List<StorageMapping> onlyAws = new ArrayList<StorageMapping>();
		List<CloudStorageDao> onlyCloud = new ArrayList<CloudStorageDao>();

		// AWS にも Hinemos DB にも登録されている。
		for (StorageMapping storageMapping: storageMappings) {
			UpdateResult updateResult = transactionalInternalUpdate(getCloudRegion().getRegion(), storageMapping.cloudStorage, storageMapping.storage);
			storageMapping.cloudStorage = updateResult.cloudStorage;
		}
		both.addAll(storageMappings);

		// AWS のみに存在する。 
		// インスタンスの自動更新フラグが有効になっているなら、自動登録処理実施。
		for (Storage s: sList) {
			UpdateResult updateResult = transactionalInternalUpdate(getCloudRegion().getRegion(), null, s);
			if (updateResult.cloudStorage != null) {
				// ストレージの自動アタッチ。
				onlyAws.add(new StorageMapping(updateResult.stroage, updateResult.cloudStorage));
			}
			getLogger().debug("AWS : " + "volumeId=" + s.getStorageId());
		}
		
		for (CloudStorageDao cs: csList) {
			UpdateResult updateResult = transactionalInternalUpdate(getCloudRegion().getRegion(), cs, null);
			if (updateResult.cloudStorage != null) {
				onlyCloud.add(updateResult.cloudStorage);
			}
		}
		
		getLogger().debug("exit updateAll()");

		return new UpdateAllResult(both, onlyAws, onlyCloud);
	}
	
	private void addStorageRelation(String Region, CloudStorageDao CloudStorageDao, Storage storage) throws CloudManagerFault, InvalidRole {
		if (storage.getStorageAttachment() == null) {
			return;
		}

		try {
			// ストレージに紐づいているインスタンスの情報から、ディスク情報を更新。
			IInstanceOperator operator = getResourceOperator(IInstanceOperator.class);
			operator.setFlags(false, false, false);
			CloudInstance cloudInstance = operator.getInstance(storage.getStorageAttachment().getInstanceId());
			// ディスク情報作成。
			updateDiskInfo(cloudInstance, CloudStorageDao, storage);
		}
		catch (CloudManagerFault e) {
			Logger logger = Logger.getLogger(this.getClass());
			logger.warn(e.getMessage(), e);
		}
	}
	
	private void updateDiskInfo(CloudInstance cloudInstance, CloudStorageDao cloudStorageDao, Storage storage) throws CloudManagerFault, InvalidRole {
		assert storage.getStorageAttachment() != null;
		
		// FacilityId に紐づいているか確認。
		if (cloudInstance.getFacilityId() != null) {
			if (!cloudInstance.getFacilityId().equals(cloudStorageDao.getFacilityId())) {
				try {
					removeDiskInfo(cloudStorageDao.getFacilityId(), cloudStorageDao.getStorageId());
				}
				catch (FacilityNotFound e) {
					throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
				}
			}
			
			RepositoryControllerBean bean = RepositoryControllerBeanWrapper.bean();
			
			// ディスク情報を設定。
			NodeInfo nodeInfo;
			try {
				nodeInfo = bean.getNode(cloudInstance.getFacilityId());
			}
			catch (FacilityNotFound | HinemosUnknown e) {
				throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
			}

			// 既に紐づいているか確認。
			NodeDiskInfo nodeDiskInfo = null;
			for (NodeDiskInfo diskInfo: nodeInfo.getNodeDiskInfo()) {
				if (("storageId=" + storage.getStorageId()).equals(diskInfo.getDeviceDescription())) {
					// 既に紐づいているので終了。
					nodeDiskInfo = diskInfo;
					break;
				}
			}

			// まだ、紐づいていないので追加。
			if (nodeDiskInfo == null) {
				nodeDiskInfo = new NodeDiskInfo();
				nodeDiskInfo.setDeviceType("disk");
				nodeDiskInfo.setDeviceName(storage.getStorageAttachment().getDevice());

				Set<Integer> indexs = new HashSet<Integer>();
				for (NodeDiskInfo diskInfo: nodeInfo.getNodeDiskInfo()) {
					indexs.add(diskInfo.getDeviceIndex());
				}
				
				for (int i = 0; i < Integer.MAX_VALUE; ++i) {
					if (!indexs.contains(i)) {
						nodeDiskInfo.setDeviceIndex(i);
						break;
					}
				}
				
				nodeDiskInfo.setDeviceSize(storage.getSize());
				nodeDiskInfo.setDeviceSizeUnit("Gib");
//					nodeDiskInfo.setDiskRpm(volume.getIops() == null ? 0: volume.getIops());
				nodeDiskInfo.setDeviceDescription("storageId=" + storage.getStorageId());
				nodeDiskInfo.setDeviceDisplayName(cloudStorageDao.getStorageName() == null ? storage.getStorageAttachment().getDevice(): cloudStorageDao.getStorageName());
				ArrayList<NodeDiskInfo> disklist = new ArrayList<NodeDiskInfo>(nodeInfo.getNodeDiskInfo());
				disklist.add(nodeDiskInfo);
				nodeInfo.setNodeDiskInfo(disklist);

				if (ActionMode.isAutoDetection()) {
					Logger logger = Logger.getLogger(this.getClass());
					logger.info("Change Node, Method=autoRegist, CloudUser=" + getCloudUser().getCloudUserId() + ", FacilityID=" + nodeInfo.getFacilityId() + ", InstanceId=" + cloudInstance.getInstanceId() + ", AddDisk=" + storage.getStorageAttachment().getDevice() + ", StorageId=" + storage.getStorageId());
				}
				
				try {
					bean.modifyNode(nodeInfo);
				}
				catch (InvalidSetting | InvalidRole | HinemosUnknown e) {
					throw new CloudManagerFault(e.getMessage(), ErrorCode.HINEMOS_MANAGER_ERROR.name(), e);
				}
				
				cloudStorageDao.setDeviceIndex(nodeDiskInfo.getDeviceIndex());
				cloudStorageDao.setDeviceName(nodeDiskInfo.getDeviceName());
				cloudStorageDao.setDeviceType(nodeDiskInfo.getDeviceType());
				cloudStorageDao.setFacilityId(nodeInfo.getFacilityId());
				cloudStorageDao.setRegistStatus(CloudStorageDao.StorageStatus.exist);
			}
			else if (!storage.getStorageAttachment().getDevice().equals(nodeDiskInfo.getDeviceName())) {
				nodeDiskInfo.setDeviceName(storage.getStorageAttachment().getDevice());
				
				try {
					bean.modifyNode(nodeInfo);
				}
				catch (InvalidSetting | InvalidRole | HinemosUnknown e) {
					throw new CloudManagerFault(e.getMessage(), ErrorCode.HINEMOS_MANAGER_ERROR.name(), e);
				}
				
				cloudStorageDao.setDeviceName(nodeDiskInfo.getDeviceName());
			}
		}
		else {
			if (cloudStorageDao.getFacilityId() != null) {
				try {
					removeDiskInfo(cloudInstance.getFacilityId(), cloudStorageDao.getStorageId());
				}
				catch (FacilityNotFound e) {
					throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
				}
			}
			
			cloudStorageDao.setDeviceIndex(null);
			cloudStorageDao.setDeviceName(null);
			cloudStorageDao.setDeviceType(null);
			cloudStorageDao.setFacilityId(null);
			cloudStorageDao.setRegistStatus(CloudStorageDao.StorageStatus.exist);
		}
	}
	
	private void removeStorageRelation(CloudStorageDao cloudStorageDao) throws CloudManagerFault, InvalidRole {
		if (cloudStorageDao.getFacilityId() == null) {
			return;
		}
		
		try {
			removeDiskInfo(cloudStorageDao.getFacilityId(), cloudStorageDao.getStorageId());
		}
		catch (FacilityNotFound e) {
			// ノードがない場合も続行。
			Logger.getLogger(this.getClass()).warn(e.getMessage(), e);
		}

		cloudStorageDao.setFacilityId(null);
		cloudStorageDao.setDeviceIndex(null);
		cloudStorageDao.setDeviceName(null);
		cloudStorageDao.setDeviceType(null);
	}

	private void removeDiskInfo(String facilityId, String storageId) throws CloudManagerFault, InvalidRole, FacilityNotFound {
		NodeInfo nodeInfo = null;
		try {
			nodeInfo = new RepositoryControllerBean().getNode(facilityId);
		}
		catch (HinemosUnknown e) {
			throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
		}
		
		if (nodeInfo != null && nodeInfo.getNodeDiskInfo() != null) {
			for (NodeDiskInfo disk: nodeInfo.getNodeDiskInfo()) {
				if (("storageId=" + storageId).equals(disk.getDeviceDescription())) {
					ArrayList<NodeDiskInfo> disklist = new ArrayList<NodeDiskInfo>(nodeInfo.getNodeDiskInfo());
					disklist.remove(disk);
					
					if (ActionMode.isAutoDetection()) {
						IInstanceOperator instanceOperator = getResourceOperator(IInstanceOperator.class);
						CloudInstance cloudInstance = null;
						try {
							cloudInstance = instanceOperator.getInstanceByFacilityId(facilityId);
						}
						catch (Exception e) {
							Logger.getLogger(this.getClass()).warn(e.toString(), e);
						}

						Logger logger = Logger.getLogger(this.getClass());
						logger.info("Change Node, Method=autoRegist, CloudUser=" + getCloudUser().getCloudUserId() + ", FacilityID=" + nodeInfo.getFacilityId() + ", InstanceId=" + (cloudInstance != null ? cloudInstance.getInstanceId(): null) + ", DelDisk=" + disk.getDeviceName() + ", VolumeId=" + storageId);
					}
					
					nodeInfo.setNodeDiskInfo(disklist);
					break;
				}
			}
		}
	}
	
	@Override
	public void attachStorage(String instanceId, String storageId, String device) throws CloudManagerFault, InvalidRole {
		IResourceManagement rm = getResourceManagement();

		rm.attachStorage(instanceId, storageId, device);
		Storage s = rm.getStorage(storageId);

		//　Hinemos DB からクラウド情報を取得。
		CloudStorageDao cloudStorageDao = SessionService.current().getEntityManagerEx().find(CloudStorageDao.class, new CloudStorageDao.CloudStoragePK(storageId, getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (cloudStorageDao == null) {
			throw ErrorCode.CLOUDSTORAGE_INVALID_CLOUDSTORAGE_NOT_FOUND.cloudManagerFault(storageId);
		}

		internalUpdate(getCloudRegion().getRegion(), cloudStorageDao, s);
	}
	
	@Override
	public void detachStorage(String instanceId, String storageId) throws CloudManagerFault, InvalidRole {
		IResourceManagement rm = getResourceManagement();
		rm.detachStorage(instanceId, storageId);
		Storage after = rm.getStorage(storageId);

		//　Hinemos DB からクラウド情報を取得。
		CloudStorageDao cloudStorageDao = SessionService.current().getEntityManagerEx().find(CloudStorageDao.class, new CloudStorageDao.CloudStoragePK(storageId, getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (cloudStorageDao == null) {
			throw ErrorCode.CLOUDSTORAGE_INVALID_CLOUDSTORAGE_NOT_FOUND.cloudManagerFault(storageId);
		}

		internalUpdate(getCloudRegion().getRegion(), cloudStorageDao, after);
	}

	@Override
	public CloudStorage getStorage(String storageId) throws CloudManagerFault, InvalidRole {
		return updateStorage(storageId).getCloudStorage();
	}

	@Override
	public List<CloudStorage> getAllStorage() throws CloudManagerFault, InvalidRole {
		return updateAllStorage().getCloudStorages();
	}

	@Override
	public CloudStorageBackup getStorageBackup(String storageBackupId) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudStorageBackupDao dao = em.find(CloudStorageBackupDao.class, new CloudStorageBackupDao.CloudStorageBackupPK(storageBackupId, getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (dao == null) {
			throw ErrorCode.CLOUDSTORAGEBACKUP_INVALID_CLOUDSTORAGEBACKUP_NOT_FOUND.cloudManagerFault(storageBackupId);
		}

		IResourceManagement rm = getResourceManagement();
		StorageBackup storageBackup = rm.getStorageBackup(storageBackupId);
		if (dao.getRestoreStatus() == CloudStorageBackupDao.RestoreStatus.available) {
			if (storageBackup == null) {
				dao.setRestoreStatus(CloudStorageBackupDao.RestoreStatus.unavailable);
			}
		}
		return new CloudStorageBackup(dao, storageBackup);
	}

	@Override
	public List<CloudStorageBackup> getAllStorageBuckup() throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		Filter filter = new Filter("accountResourceId", getCloudUser().getAccountResourceId());
		Filter filter2 = new Filter("region", getCloudRegion().getRegion());
		List<CloudStorageBackupDao> daos = new ArrayList<>(em.findByFilter(CloudStorageBackupDao.class, filter, filter2));

		List<String> storageBackupIds = new ArrayList<>();
		for (CloudStorageBackupDao backupDao: daos) {
			if (backupDao.getRestoreStatus() == CloudStorageBackupDao.RestoreStatus.available) {
				storageBackupIds.add(backupDao.getStorageBackupId());
			}
		}
		
		Map<String, StorageBackup> snapshotMap = new HashMap<>();
		if (!storageBackupIds.isEmpty()) {
			IResourceManagement rm = getResourceManagement();
			List<StorageBackup> storageBackups = rm.getStorageBackups(storageBackupIds);
			for (StorageBackup storageBackup: storageBackups) {
				snapshotMap.put(storageBackup.getStorageBackupId(), storageBackup);
			}
		}

		List<CloudStorageBackup> backups = new ArrayList<>();
		for (CloudStorageBackupDao backupDao: daos) {
			StorageBackup storageBackup = snapshotMap.get(backupDao.getStorageBackupId());
			if (backupDao.getRestoreStatus() == CloudStorageBackupDao.RestoreStatus.available) {
				if (storageBackup == null) {
					backupDao.setRestoreStatus(CloudStorageBackupDao.RestoreStatus.unavailable);
				}
			}
			backups.add(new CloudStorageBackup(backupDao, storageBackup));
		}
		
		return backups;
	}

	@Override
	@Transactional(Transactional.TransactionType.Supported)
	public List<com.clustercontrol.cloud.bean.Snapshot> getSnapshotsWithFilter(List<Filter> filters) throws CloudManagerFault, InvalidRole {
		IResourceManagement rm = getResourceManagement();
		return rm.getSnapshotsWithFilter(filters);
	}

	@Override
	public void updateAllStorageBackup() throws CloudManagerFault, InvalidRole {
		getAllStorageBuckup();
	}

	@Override
	public List<CloudStorageBackup> getStorageBuckupsByStorageId(String storageId) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		Query query = em.createQuery("SELECT b FROM CloudStorageBackupDao b WHERE b.backupedData.storageId = :storageId");
		query.setParameter("storageId", storageId);
		@SuppressWarnings("unchecked")
		List<CloudStorageBackupDao> daos = query.getResultList();
		if (daos.isEmpty()) return Collections.emptyList();
		
		List<CloudStorageBackup> backups = new ArrayList<>();
		List<String> backupIds = new ArrayList<>();
		for (CloudStorageBackupDao dao: daos) {
			backupIds.add(dao.getStorageBackupId());
		}
		List<StorageBackup> cloudBackups = getResourceManagement().getStorageBackups(backupIds);
		
		List<CloudStorageBackupDao> tempBackupDaos = new ArrayList<>(daos);
		List<StorageBackup> tempBackups = new ArrayList<>(cloudBackups);
		
		Iterator<CloudStorageBackupDao> i = tempBackupDaos.iterator();
		while (i.hasNext()) {
			CloudStorageBackupDao dao = i.next();
			Iterator<StorageBackup> j = tempBackups.iterator();
			while (j.hasNext()) {
				StorageBackup backup = j.next();
				if (backup.getStorageBackupId().equals(dao.getStorageBackupId())) {
					backups.add(new CloudStorageBackup(dao, backup));
					i.remove();
					j.remove();
					break;
				}
			}
		}
		
		for (CloudStorageBackupDao dao: tempBackupDaos) {
			if (dao.getRestoreStatus() == CloudStorageBackupDao.RestoreStatus.available) {
				dao.setRestoreStatus(CloudStorageBackupDao.RestoreStatus.unavailable);
			}
		}
		return backups;
	}

//	@Override
//	public List<String> getFlavors() throws CloudManagerFault {
//		return getResourceManagement().getStorageFlavors();
//	}
}
