/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.factory.monitors;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.log4j.Logger;
import org.quartz.CronScheduleBuilder;
import org.quartz.CronTrigger;
import org.quartz.JobBuilder;
import org.quartz.JobDetail;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.JobKey;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.TriggerBuilder;
import org.quartz.impl.StdSchedulerFactory;

import com.clustercontrol.accesscontrol.session.AccessControllerBean;
import com.clustercontrol.cloud.HinemosCredential;
import com.clustercontrol.cloud.ICloudContext;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.SessionService.ContextData;
import com.clustercontrol.cloud.bean.CloudAccountResource;
import com.clustercontrol.cloud.bean.CloudRegion;
import com.clustercontrol.cloud.bean.CloudService;
import com.clustercontrol.cloud.bean.CloudUser;
import com.clustercontrol.cloud.commons.CloudPropertyConstants;
import com.clustercontrol.cloud.commons.PropertyStringConstants;
import com.clustercontrol.cloud.factory.ActionMode;
import com.clustercontrol.cloud.factory.IAccountResourceOperator;
import com.clustercontrol.cloud.factory.ICloudServiceOperator;
import com.clustercontrol.cloud.factory.ICloudUserOperator;
import com.clustercontrol.cloud.factory.IInstanceOperator;
import com.clustercontrol.cloud.factory.IStorageOperator;
import com.clustercontrol.cloud.registry.ObjectRegistryService;
import com.clustercontrol.cloud.util.AccessDestination;
import com.clustercontrol.cloud.util.CloudContext;
import com.clustercontrol.cloud.util.CloudMessageUtil;
import com.clustercontrol.cloud.util.ResourceUtil;
import com.clustercontrol.commons.util.HinemosSessionContext;
import com.clustercontrol.fault.HinemosUnknown;

public class AutoDetectionManager {
	public interface AutoDetectionListner {
		void onPreUpdateBranch();
		void onPostUpdateBranch();
		void onPreUpdateRoot();
		void onPostUpdateRoot();
	}
	
	private static final String quartzJobName = "AotoDetection";
	private static final String quartzJobGroupName = "CLOUD_MANAGEMENT";
	
	private static Scheduler scheduler;
	
	private static List<AutoDetectionListner> listeners = Collections.synchronizedList(new ArrayList<AutoDetectionListner>());
	
	private static ExecutorService threadPool = new ThreadPoolExecutor(3, 3, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>(),
			new ThreadFactory() {
				private final AtomicInteger threadNumber = new AtomicInteger(1);
				@Override
				public Thread newThread(final Runnable r) {
					return new Thread(new Runnable() {
						@Override
						public void run() {
							String userId = CloudPropertyConstants.internal_thread_admin_user.value();
							HinemosSessionContext.instance().setProperty(HinemosSessionContext.LOGIN_USER_ID, userId);
							SessionService.current().setHinemosCredential(new HinemosCredential(userId));
							try {
								HinemosSessionContext.instance().setProperty(HinemosSessionContext.IS_ADMINISTRATOR, new AccessControllerBean().isAdministrator());
							}
							catch (HinemosUnknown e) {
								throw new InternalManagerError(e);
							}
							r.run();
						}
						
					}, "AutoDetectionService-thread-" + threadNumber.getAndIncrement());
				}
			}) {
		@Override
		protected void afterExecute(Runnable r, Throwable t) {
			super.afterExecute(r, t);
			SessionService.current().close();
		}
	};
	
	/**
	 * スケジューラーに関わる呼び出しを同期させるようにする。
	 * HinemosManager の Quartz 実装を参考とする。
	 * 
	 * @return
	 * @throws IllegalStateException
	 */
	private static synchronized Scheduler getScheduler() throws IllegalStateException {
		if (scheduler == null) {
			StdSchedulerFactory factory = new StdSchedulerFactory();
			try {
				final Scheduler internalScheduler = factory.getScheduler();
				scheduler = (Scheduler)Proxy.newProxyInstance(Scheduler.class.getClassLoader(), new Class<?>[]{Scheduler.class}, new InvocationHandler() {
					@Override
					public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
						synchronized (this) {
							return method.invoke(internalScheduler, args);
						}
					}
				});
				
				scheduler.start();
			}
			catch (SchedulerException e) {
				throw new IllegalStateException(e);
			}
		}
		return scheduler;
	}
	
	public interface IAutoDetectionService {
		void addListener(AutoDetectionListner listner);
		void removeListener(AutoDetectionListner listner);
		void start();
		void stop();
	}
	
	private static class AutoDetectionService implements IAutoDetectionService {
		@Override
		public void start() {
			Logger logger = Logger.getLogger(AutoDetectionService.class);
			
			Scheduler scheduler = getScheduler();
			try {
				// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
				scheduler.deleteJob(new JobKey(quartzJobName, quartzJobGroupName));
			}
			catch (SchedulerException e) {
				logger.warn("scheduleEjbInvokerJob() deleteJob : " + e.getClass().getSimpleName() + ", " + e.getMessage());
			}
			
			// 新規にジョブを生成
			JobDetail job = JobBuilder.newJob(AutoDetectionJob.class).withIdentity(quartzJobName, quartzJobGroupName).storeDurably(true).requestRecovery(false).build();
			
			String cronString = CloudPropertyConstants.autoupdate_interval.value();
			CronScheduleBuilder scheduleBuilder = null; 
			try {
				scheduleBuilder = CronScheduleBuilder.cronSchedule(cronString).withMisfireHandlingInstructionDoNothing();
			}
			catch (RuntimeException e) {
				logger.warn(e.getMessage(), e); 
				
				//メッセージ情報作成
				CloudMessageUtil.notify_InvalidCronString("AUTO_UPDATE", "AutoUpdate", cronString);
				
				try {
					scheduleBuilder = CronScheduleBuilder.cronSchedule(CloudPropertyConstants.autoupdate_interval.defaultValue()).withMisfireHandlingInstructionDoNothing();
				}
				catch (RuntimeException e1) {
					throw new IllegalStateException(e);
				}
			}
			
			//CronTriggerを作成
			CronTrigger trigger = TriggerBuilder.newTrigger().withIdentity(quartzJobName, quartzJobGroupName).withSchedule(scheduleBuilder).startNow().build();
			
			// ジョブを登録する
			try {
				scheduler.scheduleJob(job, trigger);
			}
			catch (SchedulerException e) {
				throw new IllegalStateException(e);
			}
		}
		
		@Override
		public void stop() {
			Logger logger = Logger.getLogger(AutoDetectionService.class);
			
			try {
				getScheduler().deleteJob(new JobKey(quartzJobName, quartzJobGroupName));
			}
			catch (SchedulerException e) {
				logger.error(e.getMessage(), e);
			}
		}
		@Override
		public void addListener(AutoDetectionListner listner) {
			listeners.add(listner);
		}
		@Override
		public void removeListener(AutoDetectionListner listner) {
			listeners.remove(listner);
		}
	}
	
	public static class AutoDetectionJob extends AbstractCloudJob {
		public AutoDetectionJob() {
		}
		
		@Override
		public void internalExecute(JobExecutionContext arg0) throws JobExecutionException {
			try {
				ActionMode.enterAutoDetection();
				
				Logger logger = Logger.getLogger(this.getClass());
				
				logger.debug("enter AsyncCallAutoDetection.");
				
				final boolean autoDetection = ActionMode.isAutoDetection();
				
				for (AutoDetectionListner listener: listeners) {
					listener.onPreUpdateRoot();
				}
				
				IAccountResourceOperator carOperator = ObjectRegistryService.registry().get(IAccountResourceOperator.class);
				ICloudUserOperator userOperator = ObjectRegistryService.registry().get(ICloudUserOperator.class);
				ICloudServiceOperator serviceOperator = ObjectRegistryService.registry().get(ICloudServiceOperator.class);
				for (CloudAccountResource accountResource: carOperator.findAllCloudAccountResource()) {
					CloudService service = serviceOperator.findCloudService(accountResource.getCloudServiceId());
					List<CloudRegion> regions = serviceOperator.findCloudRegionsByService(accountResource.getCloudServiceId());
					CloudUser account = userOperator.findCloudUser(accountResource.getAccountId());
					
					try {
						for (final CloudRegion region: regions) {
							SessionService.current().set(ICloudContext.class, new CloudContext(new AccessDestination(account, service, accountResource), region));
							final ContextData contextData = SessionService.current().getContext();
							try {
								threadPool.execute(new Runnable() {
									@Override
									public void run() {
										SessionService.changeContext(contextData);
										
										if (autoDetection) {
											ActionMode.enterAutoDetection();
										}
										
										Logger logger = Logger.getLogger(this.getClass());
										try {
											logger.debug("enter AutoDetection : region=" + region.getRegionName());
											
											for (AutoDetectionListner listener: listeners) {
												listener.onPreUpdateBranch();
											}
											
											IInstanceOperator instanceOpearter = ResourceUtil.getResourceOperator(IInstanceOperator.class);
											instanceOpearter.setFlags(
													CloudPropertyConstants.autoupdate_instance.match(PropertyStringConstants.on),
													CloudPropertyConstants.autoregist_instance.match(PropertyStringConstants.on),
													CloudPropertyConstants.autoregist_scope_relation.match(PropertyStringConstants.on));
											instanceOpearter.updateAllInstance();
											
											IStorageOperator storageOpearter = ResourceUtil.getResourceOperator(IStorageOperator.class);
											storageOpearter.setFlags(
												CloudPropertyConstants.autoupdate_storage.match(PropertyStringConstants.on),
												CloudPropertyConstants.autoregist_storage.match(PropertyStringConstants.on),
												CloudPropertyConstants.autoupdate_mount.match(PropertyStringConstants.on)
												);
											storageOpearter.updateAllStorage();
											
											if (CloudPropertyConstants.autoupdate_backup.match(PropertyStringConstants.on)) {
												instanceOpearter.updateAllInstanceBackup();
												storageOpearter.updateAllStorageBackup();
											}
											
											for (AutoDetectionListner listener: listeners) {
												listener.onPostUpdateBranch();
											}
											
											logger.debug("exit AutoDetection : region=" + region.getRegionName());
										}
										catch (Exception e) {
											logger.error(e.getMessage(), e);
										}
										finally {
											if (autoDetection) {
												ActionMode.leaveAutoDetection();
											}
										}
									}
								});
							}
							catch (Exception e) {
								logger.error(e.getMessage(), e);
							}
						}
					}
					catch (Exception e) {
						logger.error(e.getMessage(), e);
					}
				}
				
				for (AutoDetectionListner listener: listeners) {
					listener.onPostUpdateRoot();
				}
				
				logger.debug("leave AsyncCallAutoDetection.");
			}
			catch (Exception e) {
				throw new JobExecutionException(e);
			}
			finally {
				ActionMode.leaveAutoDetection();
			}
		}
	}
	
	private static IAutoDetectionService ad = new AutoDetectionService();
	
	public static IAutoDetectionService getService() {
		return ad; 
	}
}