/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.cloudn.factory;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.apache.log4j.Logger;
import org.eclipse.jdt.annotation.NonNullByDefault;

import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.bean.Tag;
import com.clustercontrol.cloud.cloudn.CloudnOptionPropertyConstants;
import com.clustercontrol.cloud.cloudn.rest.CloudnCompute;
import com.clustercontrol.cloud.cloudn.rest.DeployVirtualMachineRequest;
import com.clustercontrol.cloud.cloudn.rest.SecurityGroup;
import com.clustercontrol.cloud.cloudn.rest.Template;
import com.clustercontrol.cloud.cloudn.rest.VirtualMachine;
import com.clustercontrol.cloud.cloudn.rest.api.RestfulApplicationException;
import com.clustercontrol.cloud.persistence.Transactional;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;

@NonNullByDefault(false)
@Transactional(Transactional.TransactionType.Supported)
public class CloudnResourceManagement_FLAT extends CloudnResourceManagement_VPC {
	@Override
	protected String getComputeId() {
		return EP_TYPE_COMPUTE_FLAT;
	}

	@Override
	protected String getServiceId() {
		return SERVICE_FLAT_ID;
	}

	protected static class BackupedInstanceDetail {
		public String instanceId;
		public String name;
		public String flavor;
		public String zone;
		public List<AttachedStorage> attachedStorages;
		public List<Tag> tags;
		public Date createTime;
		public InstanceDetail instanceDetail;
	}

	protected static class InstanceDetail{
		public List<String> securityGroups = Collections.emptyList();
		public String group = "";
		public Boolean attachingDisk;
	}

	@Override
	protected DeployVirtualMachineRequest getCreateInstanceRequest(String id, String name, String flavor, String imageId, String zone, String instanceDetail, List<Tag> tags) throws CloudManagerFault, JsonProcessingException, IOException{
		ObjectMapper om = new ObjectMapper();
		ObjectReader or = om.reader(InstanceDetail.class);
		InstanceDetail detail = or.readValue(instanceDetail);

		DeployVirtualMachineRequest deployRequest = new DeployVirtualMachineRequest();
		deployRequest.serviceofferingid = getInstanceFlavorInternal(flavor).id; // 必須
		deployRequest.templateid = parseImageId(imageId).get(0, String.class); // 必須
		deployRequest.zoneid = getZoneInternal(zone).id; // 必須
//		request.diskofferingid = getStorageFlavorName(detail.);
		deployRequest.networkids = null;
		deployRequest.displayname = id;
		deployRequest.group = detail.group;
//		deployRequest.name = name + '-' + new Date().getTime();
		String s = "";
		for (String securityGroup: detail.securityGroups) {
			if (!s.isEmpty()) s += ',';
			s += securityGroup;
		}
		deployRequest.securitygroupnames = s; // 必須
		
		return deployRequest;
	}
	
	@Override
	protected void setupIpAddress(Instance instance, VirtualMachine vm) throws CloudManagerFault {
		instance.setIpAddress(vm.nic.get(0).ipaddress);
		instance.setHostName(TEMP_HOSTNAME);
	}

	protected void registStoredInstanceBackupData(VirtualMachine vm, Template template, List<AttachedStorage> snapshotIds) throws JsonProcessingException, CloudManagerFault{
		String backupId = createImageId(template.id, template.zonename);
		BackupedInstanceDetail backupedData = new BackupedInstanceDetail();
		backupedData.instanceId = vm.id;
		backupedData.name = vm.displayname;
		backupedData.flavor = vm.serviceofferingid;
		backupedData.zone = vm.zonename;
		backupedData.createTime = convertDate(template.created);
		backupedData.attachedStorages = new ArrayList<>(snapshotIds);
		backupedData.tags = new ArrayList<>();
		for (com.clustercontrol.cloud.cloudn.rest.Tag t: vm.tags) {
			backupedData.tags.add(new Tag(t.key, t.value));
		}
		backupedData.instanceDetail = new InstanceDetail();
		backupedData.instanceDetail.securityGroups = new ArrayList<>();
		for (SecurityGroup securityGroup: vm.securitygroup) {
			backupedData.instanceDetail.securityGroups.add(securityGroup.name);
		}
		backupedData.instanceDetail.group = vm.group;

		ObjectMapper om = new ObjectMapper();
		ObjectWriter ow = om.writerWithType(BackupedInstanceDetail.class);
		String backupedDetail = ow.writeValueAsString(backupedData);
		getStore().put(RT_InstanceBackup, backupId, backupedDetail);
	}

	@Override
	protected void deleteRelationalSnapshot(String instanceBackupId) throws CloudManagerFault{
		CloudnCompute compute = getCloudnEndpoint();
		String resource = getStore().get(RT_InstanceBackup, instanceBackupId);
		ObjectMapper om = new ObjectMapper();
		ObjectReader or = om.reader(BackupedInstanceDetail.class);
		BackupedInstanceDetail detail;
		try {
			detail = or.readValue(resource);
		} catch (IOException e1) {
			throw new InternalManagerError(e1);
		}
		
		for (AttachedStorage snapshotId: detail.attachedStorages) {
			try {
				compute.deleteSnapshot(snapshotId.snapshotId, null);
			}
			catch (RestfulApplicationException e) {
				// イメージは消しているので、メッセージだけ記録して処理続行。
				Logger logger = Logger.getLogger(this.getClass());
				logger.error(e.getMessage(), e);
			}
		}
	}

	@Override
	protected InstanceBackup createInternalInstanceBackup(String id) throws CloudManagerFault, JsonProcessingException, IOException{
		ObjectMapper om = new ObjectMapper();
		ObjectReader or = om.reader(BackupedInstanceDetail.class);
		ObjectWriter ow = om.writerWithType(InstanceDetail.class);

		String backupDetail = getStore().get(RT_InstanceBackup, id);
		BackupedInstanceDetail backuped = or.readValue(backupDetail);
		
		InstanceBackup instanceBackup = new InstanceBackup();
		instanceBackup.setResourceType(RT_InstanceBackup);
		instanceBackup.setInstanceBackupId(id);
		instanceBackup.setInstanceId(backuped.instanceId);
		instanceBackup.setCreateTime(backuped.createTime);
		instanceBackup.setBackupedData(new InstanceBackup.BackupedData());
		instanceBackup.getBackupedData().setName(backuped.name);
		instanceBackup.getBackupedData().setFlavor(backuped.flavor);
		instanceBackup.getBackupedData().setZone(backuped.zone);
		instanceBackup.getBackupedData().setTags(backuped.tags);
		instanceBackup.getBackupedData().setInstanceDetail(ow.writeValueAsString(backuped.instanceDetail));
		
		return instanceBackup;
	}

	@Override
	protected boolean isInstanceBackupCreating(String instanceBackupId) throws CloudManagerFault, JsonProcessingException, IOException{
		ObjectMapper om = new ObjectMapper();
		ObjectReader or = om.reader(BackupedInstanceDetail.class);
		BackupedInstanceDetail detail = or.readValue(getStore().get(RT_InstanceBackup, instanceBackupId));
		
		return new Date().getTime() - detail.createTime.getTime() < Long.parseLong(CloudnOptionPropertyConstants.cloudn_flat_until_template_created.value());
	}
	
	@Override
	protected RestoreInfo internalRestoreInstance(String facilityId, String instanceBackupId, String name, String flavor, String zone/*この情報は未使用。*/, String instanceDetail, List<Tag> tags) throws CloudManagerFault, JsonProcessingException, IOException{
		String value = getStore().get(RT_InstanceBackup, instanceBackupId);

		ObjectMapper om = new ObjectMapper();
		BackupedInstanceDetail backuped = om.reader(BackupedInstanceDetail.class).readValue(value);

		backuped.name = name != null ? name: backuped.name;
		backuped.flavor = flavor != null ? flavor: backuped.flavor;
//		backuped.zone = zone != null ? zone: backuped.zone;

		if (instanceDetail != null) {
			InstanceDetail override = om.reader(InstanceDetail.class).readValue(instanceDetail);

			if (backuped.instanceDetail == null) {
				backuped.instanceDetail = override;
			}
			else {
				backuped.instanceDetail.group = override.group != null ? override.group: backuped.instanceDetail.group;
				backuped.instanceDetail.securityGroups = override.securityGroups != null ? override.securityGroups: backuped.instanceDetail.securityGroups;
				backuped.instanceDetail.attachingDisk = override.attachingDisk != null ? override.attachingDisk: backuped.instanceDetail.attachingDisk;
			}
		}

		List<Tag> tagsTemp = new ArrayList<>(tags);
		for (Tag t: backuped.tags) {
			if (tagsTemp.isEmpty()) {
				break;
			}
			for (Iterator<Tag> iter = tagsTemp.iterator(); iter.hasNext(); ) {
				Tag override = iter.next();
				if (t.getKey().equals(override.getKey())) {
					t.setValue(override.getValue());
					iter.remove();
					break;
				}
			}
		}
		for (Tag t: tagsTemp) {
			backuped.tags.add(t);
		}
		String backupedDetail = om.writerWithType(InstanceDetail.class).writeValueAsString(backuped.instanceDetail);

		return new RestoreInfo(internalCreateInstance(facilityId, backuped.name, backuped.flavor, instanceBackupId, backuped.zone, backupedDetail, backuped.tags), backuped.attachedStorages, backuped.instanceDetail.attachingDisk);
	}
}