/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.AbstractExecutorService;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.RunnableFuture;
import java.util.concurrent.RunnableScheduledFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;

import javax.persistence.EntityManager;
import javax.persistence.RollbackException;

import org.apache.log4j.Logger;

import com.clustercontrol.cloud.persistence.EntityManagerEx;
import com.clustercontrol.cloud.persistence.StdEntityManagerEx;
import com.clustercontrol.cloud.persistence.TransactionException;
import com.clustercontrol.cloud.util.HinemosUtil;
import com.clustercontrol.commons.util.HinemosSessionContext;
import com.clustercontrol.commons.util.JpaPersistenceConfig;
import com.clustercontrol.commons.util.JpaTransactionManager;

public class SessionService {
	public interface PreCommitAction {
		RolebackAction preCommit() throws TransactionException;
	}

	public interface PostCommitAction {
		void postCommit() throws TransactionException;
	}

	public interface RolebackAction {
		void rollback() throws TransactionException;
	}

	public interface ISession {
		void setState(Object key, boolean state);
		boolean isState(Object key);
		Map<Object, Boolean> getStates();

		Object getProperty(Object key);
		void setProperty(Object key, Object value);
		Map<Object, Object> getProperties();

		<T> T get(Class<T> clazz);
		<T> void set(Class<T> clazz, T object);
		Map<Class<?>, ?> getObjects();

		HinemosCredential getHinemosCredential();
		void setHinemosCredential(HinemosCredential credential);

		EntityManagerEx getEntityManagerEx();

		void beginTransaction();
		void commitTransaction();
		void flushPersistenceContext();
		void rollbackTransaction();
		void close();

		void addPreCommitAction(PreCommitAction action);
		void addPostCommitAction(PostCommitAction action);
		void addRollbackAction(RolebackAction action);

		boolean isTransaction();

		boolean isDebugEnbled();

		ContextData getContext();
	}

	public interface ISessionInitializer {
		void initialize(ISession session);
		void close(ISession session);
	}

	public static class ContextData {
		private Map<Object, Boolean> states;
		private Map<Class<?>, Object> objects;
		private Map<Object, Object> properties;
		private HinemosCredential credential;
		private String accountName;
		private Boolean isAdministrator;
	}

	private static class DelegatedExecutorService extends AbstractExecutorService {
		private final ExecutorService e;
		DelegatedExecutorService(ExecutorService executor) { e = executor; }
		public void execute(Runnable command) { e.execute(command); }
		public void shutdown() { e.shutdown(); }
		public List<Runnable> shutdownNow() { return e.shutdownNow(); }
		public boolean isShutdown() { return e.isShutdown(); }
		public boolean isTerminated() { return e.isTerminated(); }
		public boolean awaitTermination(long timeout, TimeUnit unit)
				throws InterruptedException {
			return e.awaitTermination(timeout, unit);
		}
		public Future<?> submit(Runnable task) {
			return e.submit(task);
		}
		public <T> Future<T> submit(Callable<T> task) {
			return e.submit(task);
		}
		public <T> Future<T> submit(Runnable task, T result) {
			return e.submit(task, result);
		}
		public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks)
				throws InterruptedException {
			return e.invokeAll(tasks);
		}
		public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks,
				long timeout, TimeUnit unit)
						throws InterruptedException {
			return e.invokeAll(tasks, timeout, unit);
		}
		public <T> T invokeAny(Collection<? extends Callable<T>> tasks)
				throws InterruptedException, ExecutionException {
			return e.invokeAny(tasks);
		}
		public <T> T invokeAny(Collection<? extends Callable<T>> tasks,
				long timeout, TimeUnit unit)
						throws InterruptedException, ExecutionException, TimeoutException {
			return e.invokeAny(tasks, timeout, unit);
		}
	}

	private static class DelegatedScheduledExecutorService extends DelegatedExecutorService implements ScheduledExecutorService {
		private final ScheduledExecutorService e;
		DelegatedScheduledExecutorService(ScheduledExecutorService executor) {
			super(executor);
			e = executor;
		}
		public ScheduledFuture<?> schedule(Runnable command, long delay,  TimeUnit unit) {
			return e.schedule(command, delay, unit);
		}
		public <V> ScheduledFuture<V> schedule(Callable<V> callable, long delay, TimeUnit unit) {
			return e.schedule(callable, delay, unit);
		}
		public ScheduledFuture<?> scheduleAtFixedRate(Runnable command, long initialDelay,  long period, TimeUnit unit) {
			return e.scheduleAtFixedRate(command, initialDelay, period, unit);
		}
		public ScheduledFuture<?> scheduleWithFixedDelay(Runnable command, long initialDelay,  long delay, TimeUnit unit) {
			return e.scheduleWithFixedDelay(command, initialDelay, delay, unit);
		}
	}

	private static class FinalizableDelegatedExecutorService extends DelegatedExecutorService {
		FinalizableDelegatedExecutorService(ExecutorService executor) {
			super(executor);
		}
		protected void finalize() {
			super.shutdown();
		}
	}

	private static class FinalizableDelegatedScheduledExecutorService extends DelegatedScheduledExecutorService {
		FinalizableDelegatedScheduledExecutorService(ScheduledExecutorService executor) {
			super(executor);
		}
		protected void finalize() {
			super.shutdown();
		}
	}

	private static class DelegateRunnableFuture<V> implements RunnableFuture<V> {
		// タスクを依頼したスレッドのセキュリティ情報を保存。
		private ContextData context = SessionService.current().getContext();
		private RunnableFuture<V> task;

		public DelegateRunnableFuture(RunnableFuture<V> task) {
			this.task = task;
		}

		public void run() {
			// 新規に立ち上げたスレッドで保存したセキュリティ情報を適用。
			SessionService.changeContext(context);
			task.run();
		}

		public boolean cancel(boolean mayInterruptIfRunning) {
			return task.cancel(mayInterruptIfRunning);
		}

		public boolean isCancelled() {
			return task.isCancelled();
		}

		public boolean isDone() {
			return task.isDone();
		}

		public V get() throws InterruptedException, ExecutionException {
			return task.get();
		}

		public V get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
			return task.get(timeout, unit);
		}

		@Override
		public int hashCode() {
			return task.hashCode();
		}

		@Override
		public boolean equals(Object obj) {
			return task.equals(obj);
		}

		@Override
		public String toString() {
			return task.toString();
		}
	}

	private static class ThreadPoolExecutorEx extends ThreadPoolExecutor {
		@Override
		protected <T> RunnableFuture<T> newTaskFor(Runnable runnable, T value) {
			return new DelegateRunnableFuture<T>(super.newTaskFor(runnable, value));
		}

		@Override
		protected <T> RunnableFuture<T> newTaskFor(Callable<T> callable) {
			return new DelegateRunnableFuture<T>(super.newTaskFor(callable));
		}

		public ThreadPoolExecutorEx(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, RejectedExecutionHandler handler) {
			super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, handler);
		}

		public ThreadPoolExecutorEx(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, ThreadFactory threadFactory, RejectedExecutionHandler handler) {
			super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory, handler);
		}

		public ThreadPoolExecutorEx(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, ThreadFactory threadFactory) {
			super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory);
		}

		public ThreadPoolExecutorEx(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue) {
			super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue);
		}

		@Override
		protected void afterExecute(Runnable r, Throwable t) {
			super.afterExecute(r, t);
			SessionService.current().close();
			new JpaTransactionManager().close();
		}
	}

	private static class DelegateRunnableScheduledFuture<V> extends DelegateRunnableFuture<V> implements RunnableScheduledFuture<V> {
		private RunnableScheduledFuture<V> task;

		public DelegateRunnableScheduledFuture(RunnableScheduledFuture<V> task) {
			super(task);
			this.task = task;
		}

		public long getDelay(TimeUnit unit) {
			return task.getDelay(unit);
		}

		public boolean isPeriodic() {
			return task.isPeriodic();
		}

		public int compareTo(Delayed o) {
			return task.compareTo(o);
		}
	}

	private static class ScheduledExecutorServiceEx extends ScheduledThreadPoolExecutor {
		public ScheduledExecutorServiceEx(int corePoolSize,
				RejectedExecutionHandler handler) {
			super(corePoolSize, handler);
		}

		public ScheduledExecutorServiceEx(int corePoolSize,
				ThreadFactory threadFactory, RejectedExecutionHandler handler) {
			super(corePoolSize, threadFactory, handler);
		}

		public ScheduledExecutorServiceEx(int corePoolSize,
				ThreadFactory threadFactory) {
			super(corePoolSize, threadFactory);
		}

		public ScheduledExecutorServiceEx(int corePoolSize) {
			super(corePoolSize);
		}

		@Override
		protected <V> RunnableScheduledFuture<V> decorateTask(Callable<V> callable, RunnableScheduledFuture<V> task) {
			return new DelegateRunnableScheduledFuture<V>(super.decorateTask(callable, task));
		}

		@Override
		protected <V> RunnableScheduledFuture<V> decorateTask(Runnable runnable, RunnableScheduledFuture<V> task) {
			return new DelegateRunnableScheduledFuture<V>(super.decorateTask(runnable, task));
		}

		@Override
		protected void afterExecute(Runnable r, Throwable t) {
			super.afterExecute(r, t);
			SessionService.current().close();
		}
	}

	private static ExecutorService executorService = newCachedThreadPool("SessionContextPool");
	private static ScheduledExecutorService scheduledExecutorService = newScheduledThreadPool(3, "SessionContextScheduler");

	private static class SessionImpl implements ISession {
		private static Logger logger = Logger.getLogger(SessionImpl.class);

		private Map<Object, Boolean> states = new HashMap<>();
		private Map<Class<?>, Object> objects = new HashMap<>();
		private Map<Object, Object> properties = new HashMap<>();

		private HinemosCredential credential = new HinemosCredential();
//		private JpaTransactionManager jtm;
		private EntityManager em;
		private EntityManagerEx emEx;
		private boolean transaction;

		private List<PreCommitAction> preCommitActions = new ArrayList<>();
		private List<PostCommitAction> postCommitActions = new ArrayList<>();
		private List<RolebackAction> rollbackActions = new ArrayList<>();

//		private JpaTransactionCallback callback = new JpaTransactionCallback() {
//			@Override
//			public void preBegin() {
//			}
//			@Override
//			public void postBegin() {
//			}
//			@Override
//			public void preFlush() {
//			}
//			@Override
//			public void postFlush() {
//			}
//			@Override
//			public void preCommit() {
//			}
//			@Override
//			public void postCommit() {
//			}
//			@Override
//			public void preRollback() {
//			}
//			@Override
//			public void postRollback() {
//			}
//			@Override
//			public void preClose() {
//			}
//			@Override
//			public void postClose() {
//				SessionImpl.this.release();
//			}
//		};

		private SessionImpl() {
//			jtm = new JpaTransactionManager();
//			jtm.addCallback(callback);
//			em = jtm.getEntityManager();
			em = JpaPersistenceConfig.getHinemosEMFactory().createEntityManager();
			emEx = new StdEntityManagerEx(em);
//			objects.put(JpaTransactionManager.class, jtm);
			objects.put(EntityManager.class, em);
		}

		private SessionImpl(ContextData context) {
			this();
			HinemosSessionContext.instance().setProperty(HinemosSessionContext.LOGIN_USER_ID, context.accountName);
			HinemosSessionContext.instance().setProperty(HinemosSessionContext.IS_ADMINISTRATOR, context.isAdministrator);

			this.states.putAll(context.states);
			this.objects.putAll(context.objects);
			this.properties.putAll(context.properties);
			this.credential = context.credential;
		}

		@Override
		public Object getProperty(Object key) {
			return properties.get(key);
		}

		@Override
		public void setProperty(Object key, Object value) {
			properties.put(key, value);
		}

		@Override
		public <T> T get(Class<T> clazz) {
			return clazz.cast(objects.get(clazz));
		}

		@Override
		public <T> void set(Class<T> clazz, T object) {
			objects.put(clazz, object);
		}

		@Override
		public void beginTransaction() {
			if (!(emEx.getTransaction().isActive() || transaction)) {
//				jtm.begin();
				emEx.getTransaction().begin();
				transaction = true;
				preCommitActions.clear();
				postCommitActions.clear();
				rollbackActions.clear();
			}
		}

		@Override
		public void commitTransaction() {
			if (transaction) {
				try {
					try {
						for (PreCommitAction action: preCommitActions) {
							RolebackAction rollback = action.preCommit();
							if (rollback != null) {
								// 最後に実行したコミット操作を先にロールバックするので先頭へ追加。
								rollbackActions.add(0, rollback);
							}
						}
						emEx.getTransaction().commit();
//						jtm.commit();
					}
					catch (Exception e1) {
						if (!(e1 instanceof RollbackException)) {
							try {
								emEx.getTransaction().rollback();
//								jtm.rollback();
							}
							catch (Exception e2) {
								Logger.getLogger(this.getClass()).error(e2.getMessage(), e2);
							}
						}

						for (RolebackAction action: rollbackActions) {
							try {
								action.rollback();
							}
							catch (Exception e2) {
								Logger.getLogger(this.getClass()).error(e2.getMessage(), e2);
							}
						}
						throw e1;
					}

					// 以下の処理で DB への処理を含めると、例外がでた場合、DB の論理整合が壊れる可能性がある。
					for (PostCommitAction action: postCommitActions) {
						action.postCommit();
					}
				}
				finally {
					transaction = false;
					preCommitActions.clear();
					postCommitActions.clear();
					rollbackActions.clear();
				}
			}
		}

		@Override
		public void rollbackTransaction() {
			if (transaction) {
				try {
					emEx.getTransaction().rollback();
//					jtm.rollback();
				}
				finally {
					for (RolebackAction action: rollbackActions) {
						try {
							action.rollback();
						}
						catch (Exception e2) {
							Logger.getLogger(this.getClass()).error(e2.getMessage(), e2);
						}
					}

					transaction = false;
					preCommitActions.clear();
					postCommitActions.clear();
					rollbackActions.clear();
				}
			}
		}

		@Override
		public void flushPersistenceContext() {
			if (transaction) {
				try {
//					for (PreCommitAction action: preCommitActions) {
//						RolebackAction rollback = action.preCommit();
//						if (rollback != null) {
//							// 最後に実行したコミット操作を先にロールバックするので先頭へ追加。
//							rollbackActions.add(0, rollback);
//						}
//					}
					emEx.flush();
//					jtm.flush();
				}
				catch (Exception e1) {
					for (RolebackAction action: rollbackActions) {
						try {
							action.rollback();
						}
						catch (Exception e2) {
							Logger.getLogger(this.getClass()).error(e2.getMessage(), e2);
						}
					}
					throw e1;
				}
				finally {
					transaction = false;
					preCommitActions.clear();
					postCommitActions.clear();
					rollbackActions.clear();
				}
			}
		}

		@Override
		public void close() {
			emEx.close();
//			jtm.close();
			// release は、JpaTransactionManager からのコールバックで postClose のタイミングで呼ばれるはず。
			release();
		}

		@Override
		public HinemosCredential getHinemosCredential() {
			return credential;
		}

		@Override
		public void setHinemosCredential(HinemosCredential credential) {
			this.credential = credential;
		}

		@Override
		public Map<Object, Object> getProperties() {
			return properties;
		}

		@Override
		public Map<Class<?>, ?> getObjects() {
			return objects;
		}

		@Override
		public EntityManagerEx getEntityManagerEx() {
			return emEx;
		}

		@Override
		public boolean isDebugEnbled() {
			return logger.isDebugEnabled();
		}

		@Override
		public boolean isTransaction() {
//			return (jtm != null && jtm.getEntityManager().getTransaction().isActive()) || transaction;
			return (emEx != null && emEx.getTransaction().isActive()) || transaction;
		}

		@Override
		public void addPreCommitAction(PreCommitAction action) {
			preCommitActions.add(action);
		}

		@Override
		public void addRollbackAction(RolebackAction action) {
			// 最後に実行したコミット操作を先にロールバックするので先頭へ追加。
			rollbackActions.add(0, action);
		}

		@Override
		public ContextData getContext() {
			// スレッドローカルの情報を詰め込む可能性があるので、その問題が出たら対処。
			ContextData data = new ContextData();

			data.states = new HashMap<>(getStates());
			data.objects = new HashMap<>(getObjects());
			data.objects.remove(JpaTransactionManager.class);
			data.objects.remove(EntityManager.class);

			data.properties = new HashMap<>(getProperties());
			data.credential = getHinemosCredential();

			data.accountName = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);
			data.isAdministrator = (Boolean)HinemosSessionContext.instance().getProperty(HinemosSessionContext.IS_ADMINISTRATOR);

			return data;
		}

		@Override
		public void addPostCommitAction(PostCommitAction action) {
			postCommitActions.add(action);
		}

		@Override
		public void setState(Object key, boolean state) {
			states.put(key, state);
		}

		@Override
		public boolean isState(Object key) {
			Boolean state = states.get(key);
			return state != null ? state: false;
		}

		@Override
		public Map<Object, Boolean> getStates() {
			return states;
		}

		public void release() {
			synchronized (initializers) {
				for (ISessionInitializer initializer: initializers) {
					initializer.close(this);
				}
			}
			SessionService.deque.get().pollFirst();
			HinemosUtil.shutdown();
		}
	}

	private static ThreadLocal<Deque<SessionImpl>> deque  = new ThreadLocal<Deque<SessionImpl>>() {
		protected Deque<SessionImpl> initialValue() {
			return new LinkedList<SessionImpl>();
		}
	};

	private static List<ISessionInitializer> initializers = Collections.synchronizedList(new ArrayList<ISessionInitializer>());

	public static ISession current() {
		SessionImpl session = deque.get().peekFirst();
		if (session != null) return session;
		offer();
		return deque.get().peekFirst();
	}
	
	public static ISession changeContext(ContextData context) {
		poll();
		offer(context);
		return deque.get().peekFirst();
	}
	
	public static void offer() {
		SessionImpl session = deque.get().peekFirst();
		offer(session != null ? session.getContext(): null);
	}

	public static void offer(ContextData context) {
		SessionImpl session = context == null ? new SessionImpl(): new SessionImpl(context);
		synchronized (initializers) {
			for (ISessionInitializer initializer: initializers) {
				initializer.initialize(session);
			}
		}
		deque.get().offerFirst(session);
	}

	public static void poll() {
		ISession oldSession = deque.get().peekFirst();
		if (oldSession != null) {
			if (oldSession.isTransaction()) throw new InternalManagerError("Transaction is active.");
			oldSession.close();
		}
	}

	public static void addInitializer(ISessionInitializer initializer) {
		initializers.add(initializer);
	}

	public static ExecutorService newFixedThreadPool(int nThreads, final String poolName) {
		return new FinalizableDelegatedExecutorService(new ThreadPoolExecutorEx(nThreads, nThreads, 0L, TimeUnit.MICROSECONDS, new LinkedBlockingQueue<Runnable>(), new ThreadFactory() {
			private final AtomicInteger threadNumber = new AtomicInteger(1);
			@Override
			public Thread newThread(Runnable r) {
				return new Thread(r, poolName + "-thread-" + threadNumber.getAndIncrement());
			}
		}));
	}

	public static ExecutorService newCachedThreadPool(final String poolName) {
		return new FinalizableDelegatedExecutorService(new ThreadPoolExecutorEx(0, Integer.MAX_VALUE, 60L, TimeUnit.SECONDS, new SynchronousQueue<Runnable>(), new ThreadFactory() {
			private final AtomicInteger threadNumber = new AtomicInteger(1);
			@Override
			public Thread newThread(Runnable r) {
				return new Thread(r, poolName + "-thread-" + threadNumber.getAndIncrement());
			}
		}));
	}

	public static ScheduledExecutorService newScheduledThreadPool(int corePoolSize, final String poolName) {
		return  new FinalizableDelegatedScheduledExecutorService(new ScheduledExecutorServiceEx(corePoolSize, new ThreadFactory() {
			private final AtomicInteger threadNumber = new AtomicInteger(1);
			@Override
			public Thread newThread(Runnable r) {
				return new Thread(r, poolName + "-thread-" + threadNumber.getAndIncrement());
			}
		}));
	}

	public static ExecutorService newSingleThreadExecutor(final String poolName) {
		// FinalizableDelegatedExecutorService でラップしないと、finalizer が呼ばれない。
		// 理由は、worker スレッドが、スレッドプールオブジェクト本体の参照を持っているため、ワーカーが行き続けている間は、スレッドプールオブジェクトの参照が切れない。
		// スレッドプールなのでワーカーはプールされるので、本体の参照が永遠に存在し続ける。
		// FinalizableDelegatedExecutorService は、その参照関係とは関係ないので、最終参照者がはずれるタイミングで、gc 対象となる。
		return new FinalizableDelegatedExecutorService(new ThreadPoolExecutorEx(1, 1, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>(), new ThreadFactory() {
			private final AtomicInteger threadNumber = new AtomicInteger(1);
			@Override
			public Thread newThread(Runnable r) {
				return new Thread(r, poolName + "-thread-" + threadNumber.getAndIncrement());
			}
		}));
	}

	public static ScheduledExecutorService newSingleThreadScheduledExecutor(final String poolName) {
		return new FinalizableDelegatedScheduledExecutorService(new ScheduledExecutorServiceEx(1, new ThreadFactory() {
			private final AtomicInteger threadNumber = new AtomicInteger(1);
			@Override
			public Thread newThread(Runnable r) {
				return new Thread(r, poolName + "-thread-" + threadNumber.getAndIncrement());
			}
		}));
	}

	public static void execute(Runnable command) {
		executorService.execute(command);
	}

	public static <T> Future<T> submit(Callable<T> task) {
		return executorService.submit(task);
	}

	public static <T> Future<T> submit(Runnable task, T result) {
		return executorService.submit(task, result);
	}

	public static Future<?> submit(Runnable task) {
		return executorService.submit(task);
	}

	public static ScheduledFuture<?> scheduleWithFixedDelay(Runnable command, long initialDelay, long delay, TimeUnit unit) {
		return scheduledExecutorService.scheduleWithFixedDelay(command, initialDelay, delay, unit);
	}
}
