/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
*/
package com.clustercontrol.ws.azure;

import static com.clustercontrol.cloud.util.ResourceUtil.getResourceOperator;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;
import javax.jws.WebService;
import javax.xml.ws.WebServiceContext;

import org.apache.log4j.Logger;

import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.SystemPrivilegeMode;
import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.ICloudContext;
import com.clustercontrol.cloud.IResourceManagement.Instance;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.azure.AzureOptionPropertyConstants;
import com.clustercontrol.cloud.azure.bean.AzureCreateInstanceRequest;
import com.clustercontrol.cloud.azure.bean.AzureCreateStorageRequest;
import com.clustercontrol.cloud.azure.bean.AzureImage;
import com.clustercontrol.cloud.azure.bean.AzureRestoreStorageRequest;
import com.clustercontrol.cloud.azure.bean.SnapShotInfo;
import com.clustercontrol.cloud.azure.bean.StorageSize;
import com.clustercontrol.cloud.azure.factory.AzureAsyncExecution;
import com.clustercontrol.cloud.azure.factory.AzureAsyncOperationStatus;
import com.clustercontrol.cloud.azure.factory.AzureAttachStorage;
import com.clustercontrol.cloud.azure.factory.AzureCreateInstance;
import com.clustercontrol.cloud.azure.factory.AzureDetachStorage;
import com.clustercontrol.cloud.azure.util.AzureConstants;
import com.clustercontrol.cloud.azure.util.AzureUtil;
import com.clustercontrol.cloud.bean.CloudStorage;
import com.clustercontrol.cloud.factory.CloudRoleConstants;
import com.clustercontrol.cloud.factory.IStorageOperator;
import com.clustercontrol.cloud.validation.AuthorizingValidator_role_region;
import com.clustercontrol.cloud.validation.annotation.CustomMethodValidation;
import com.clustercontrol.cloud.validation.annotation.ElementId;
import com.clustercontrol.cloud.validation.annotation.Identity;
import com.clustercontrol.cloud.validation.annotation.Into;
import com.clustercontrol.cloud.validation.annotation.NotNull;
import com.clustercontrol.cloud.validation.annotation.Size;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.ws.cloud.IWebServiceBase;
import com.clustercontrol.ws.cloud.security.HinemosAccessRight;
import com.microsoft.windowsazure.Configuration;
import com.microsoft.windowsazure.core.OperationResponse;
import com.microsoft.windowsazure.core.OperationStatus;
import com.microsoft.windowsazure.core.OperationStatusResponse;
import com.microsoft.windowsazure.exception.ServiceException;
import com.microsoft.windowsazure.management.AffinityGroupOperations;
import com.microsoft.windowsazure.management.ManagementClient;
import com.microsoft.windowsazure.management.ManagementService;
import com.microsoft.windowsazure.management.compute.ComputeManagementClient;
import com.microsoft.windowsazure.management.compute.ComputeManagementService;
import com.microsoft.windowsazure.management.compute.DeploymentOperations;
import com.microsoft.windowsazure.management.compute.HostedServiceOperations;
import com.microsoft.windowsazure.management.compute.VirtualMachineDiskOperations;
import com.microsoft.windowsazure.management.compute.models.DataVirtualHardDisk;
import com.microsoft.windowsazure.management.compute.models.DeploymentGetResponse;
import com.microsoft.windowsazure.management.compute.models.DeploymentSlot;
import com.microsoft.windowsazure.management.compute.models.HostedServiceListResponse;
import com.microsoft.windowsazure.management.compute.models.VirtualMachineDataDiskCreateParameters;
import com.microsoft.windowsazure.management.compute.models.VirtualMachineDiskCreateParameters;
import com.microsoft.windowsazure.management.compute.models.VirtualMachineDiskGetResponse;
import com.microsoft.windowsazure.management.compute.models.VirtualMachineOSImageListResponse;
import com.microsoft.windowsazure.management.compute.models.HostedServiceListResponse.HostedService;
import com.microsoft.windowsazure.management.compute.models.VirtualMachineDiskListResponse.VirtualMachineDisk;
import com.microsoft.windowsazure.management.compute.models.Role;
import com.microsoft.windowsazure.management.compute.models.RoleInstance;
import com.microsoft.windowsazure.management.models.AffinityGroupListResponse;
import com.microsoft.windowsazure.management.models.AffinityGroupListResponse.AffinityGroup;
import com.microsoft.windowsazure.management.storage.StorageAccountOperations;
import com.microsoft.windowsazure.management.storage.StorageManagementClient;
import com.microsoft.windowsazure.management.storage.StorageManagementService;
import com.microsoft.windowsazure.management.storage.models.StorageAccount;
import com.microsoft.windowsazure.management.storage.models.StorageAccountGetKeysResponse;
import com.microsoft.windowsazure.management.storage.models.StorageAccountListResponse;
import com.microsoft.windowsazure.services.blob.BlobContract;
import com.microsoft.windowsazure.services.blob.BlobService;
import com.microsoft.windowsazure.services.blob.models.GetBlobResult;
import com.microsoft.windowsazure.services.blob.models.ListContainersResult;
import com.microsoft.windowsazure.services.blob.models.ListContainersResult.Container;


/**
 * クラウド管理オプション用のエンドポイント
 *
 */
@WebService(serviceName = "AzureOptionEndpointService", portName = "AzureOptionEndpointPort", targetNamespace = "http://azure.ws.clustercontrol.com", endpointInterface="com.clustercontrol.ws.azure.AzureOptionEndpoint")
public class AzureOptionEndpointImpl implements AzureOptionEndpoint, IWebServiceBase, AzureConstants, CloudRoleConstants {
	@Resource
	private WebServiceContext wsctx;

	private Configuration getConfiguration() throws CloudManagerFault {
		final ICloudContext context = SessionService.current().get(ICloudContext.class);
		return AzureUtil.getConfiguration(context.getAccessDestionation().getCloudUser().getAccessKey(),
											context.getAccessDestionation().getCloudUser().getSecretKey());
	}

	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public String getRegionByCloudService(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("cloudServiceName") @Identity String cloudServiceName
			)throws CloudManagerFault, InvalidUserPass, InvalidRole{

        Configuration config = getConfiguration();
		ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);
		ManagementClient managementClient = ManagementService.create(config);
		Logger logger = Logger.getLogger(this.getClass());

		HostedServiceOperations hostedServicesOperations = computeManagementClient.getHostedServicesOperations();
		AffinityGroupOperations affinityGroupOperations = managementClient.getAffinityGroupsOperations();

		try {
			HostedServiceListResponse hostedServiceListResponse = hostedServicesOperations.list();
			for(HostedService hostedService:hostedServiceListResponse.getHostedServices()){
				if(hostedService.getServiceName().equals(cloudServiceName)){
					if(hostedService.getProperties().getLocation()!=null){
						return hostedService.getProperties().getLocation();
					}else{
						AffinityGroupListResponse affinityGroupListResponse = affinityGroupOperations.list();
						for(AffinityGroup af :affinityGroupListResponse.getAffinityGroups()){
							if(af.getName().equals(hostedService.getProperties().getAffinityGroup())){
								return af.getLocation();
							}
						}
					}
				}
			}
			logger.info("Successful in get Region by Cloud Service");
		} catch (Exception e) {
			throw new CloudManagerFault(e.getMessage(), e);
		}
		throw new InternalManagerError();
	}

	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public List<String> getStorageAccountByRegion(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region
			)throws CloudManagerFault, InvalidUserPass, InvalidRole{
		List<String> storageAccountNames = new ArrayList<>();

		try {
	        Configuration config = getConfiguration();
			ManagementClient managementClient = ManagementService.create(config);
			StorageManagementClient storageManagementClient = StorageManagementService.create(config);

			StorageAccountOperations storageAccountOperations = storageManagementClient.getStorageAccountsOperations();
			AffinityGroupOperations affinityGroupOperations = managementClient.getAffinityGroupsOperations();
	 		AffinityGroupListResponse affinityGroupListResponse = affinityGroupOperations.list();
			StorageAccountListResponse storageServiceListResponse = storageAccountOperations.list();
			for(StorageAccount storageAccount:storageServiceListResponse.getStorageAccounts()){
				if(AzureUtil.compareLocation(region,storageAccount.getProperties().getLocation(),storageAccount.getProperties().getAffinityGroup(),affinityGroupListResponse.getAffinityGroups())){
					storageAccountNames.add(storageAccount.getName());
				}
			}
		} catch (Exception e) {
			throw new CloudManagerFault(e.getMessage(), e);
		}
		return storageAccountNames;
	}


	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public List<String> getStorageAccountByCloudService(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("cloudServiceName") @Identity String cloudServiceName
			)throws CloudManagerFault, InvalidUserPass, InvalidRole{
		List<String> StorageAccountList = getStorageAccountByRegion(roleId,getRegionByCloudService(roleId,region,cloudServiceName));
		return StorageAccountList;
	}

	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public List<DataVirtualHardDisks> getAttacheableDataVirtualHardDisk(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("instanceId") @NotNull String instanceId
			) throws CloudManagerFault, InvalidUserPass, InvalidRole {

		ArrayList<DataVirtualHardDisks> attacheableDataVirtualHardDisks = new ArrayList<>();
		
		try{
	        Configuration config = getConfiguration();
			ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);
        	ArrayList<VirtualMachineDisk> vmds = computeManagementClient.getVirtualMachineDisksOperations().listDisks().getDisks();
	 		
	 		for(VirtualMachineDisk vmd:vmds){
	 			if(vmd.getUsageDetails()==null){
	 				DataVirtualHardDisks dvhds = new DataVirtualHardDisks();
	        		dvhds.setName(vmd.getName());
	        		dvhds.setOSType(vmd.getOperatingSystemType());
	        		dvhds.setMediaLink(vmd.getMediaLinkUri());
	        		dvhds.setRoleSize(Integer.toString(vmd.getLogicalSizeInGB()));
	        		attacheableDataVirtualHardDisks.add(dvhds);
	 			}
        	}
	 		
			return attacheableDataVirtualHardDisks;
		}
		catch(Exception e){
			throw new CloudManagerFault(e.getMessage(), e);
		}
	}
	
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right={SystemPrivilegeMode.READ, SystemPrivilegeMode.ADD})
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public void detachStorage(@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("instanceId") @NotNull String instanceId,
			@ElementId("storageId") @NotNull String storageId) throws CloudManagerFault {
		
		Logger logger = Logger.getLogger(this.getClass());
		try{
			logger.info("createInstance() start");
			AzureDetachStorage azureCreateInstance = new AzureDetachStorage(getConfiguration(),instanceId,storageId);
		    if (AzureOptionPropertyConstants.azure_api_mode.value().equalsIgnoreCase("async")) {
				AzureAsyncExecution.getSingleton().put(azureCreateInstance);		    	
		    }
		    else {
		    	azureCreateInstance.execute();
		    }
		}catch(Exception e){
    		throw new CloudManagerFault(e.getMessage(), e);
    	}
		finally {
			logger.info("createInstance() end");			
		}
	}

	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public List<InstanceInfo> getInstanceByCloudService(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("cloudServiceName") @Identity String cloudServiceName
			)throws CloudManagerFault, InvalidUserPass, InvalidRole{

		Logger logger = Logger.getLogger(this.getClass());

		List<InstanceInfo> instancesInfo = new ArrayList<>();

		try{
	        Configuration config = getConfiguration();
			ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);

			HostedServiceOperations hostedServicesOperations = computeManagementClient.getHostedServicesOperations();
			DeploymentOperations deploymentOperations = computeManagementClient.getDeploymentsOperations();
	 		HostedServiceListResponse hostedServiceListResponse = hostedServicesOperations.list();
	 		for (HostedServiceListResponse.HostedService hostService: hostedServiceListResponse.getHostedServices()) {
	 			if(hostService.getServiceName().equals(cloudServiceName)){
		 			for (DeploymentSlot slot:AzureUtil.getDeploymentSlots()) {
		 				DeploymentGetResponse deploymentGetResponse = null;
			 			try {
			 				deploymentGetResponse = deploymentOperations.getBySlot(hostService.getServiceName(),slot);
			 			}
			 			catch (ServiceException e) {
			 				if (e.getErrorCode().equals("ResourceNotFound"))
			 					continue;
			 				logger.error("getInstanceByCloudService: "+ cloudServiceName + " (" + e.getErrorCode() +") "+ e.getMessage());
			 				throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
			 			}
						for (Role role: deploymentGetResponse.getRoles()) {
							for(RoleInstance roleinstance:deploymentGetResponse.getRoleInstances()){
								if(roleinstance.getRoleName().equalsIgnoreCase(role.getRoleName())){
									InstanceInfo instanceInfo = new InstanceInfo();
									instanceInfo.setInstanceName(roleinstance.getRoleName());
									instanceInfo.setDeploymentName(deploymentGetResponse.getName());
									instanceInfo.setSize(role.getRoleSize());
									instanceInfo.setOSType(role.getOSVirtualHardDisk().getOperatingSystem());
									instanceInfo.setDiskNumber(role.getDataVirtualHardDisks().size());
									instanceInfo.setAttachedLun(new ArrayList<Integer>());
									for(DataVirtualHardDisk dvhd:role.getDataVirtualHardDisks()){
										instanceInfo.addAttachedLun(dvhd.getLogicalUnitNumber());
									}
									instanceInfo.setDataVirtualHardDisks(role.getDataVirtualHardDisks());
									instanceInfo.setDataDiskAttachableNum(StorageSize.getAttachableMaxmum(role.getRoleSize()));
									instancesInfo.add(instanceInfo);
								}
							}
						}
		 			}
	 			}
			}
		}catch(ServiceException e){
			logger.error("getInstanceByCloudService: "+ cloudServiceName + " (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
		}catch(Exception e){
			logger.error("getInstanceByCloudService: "+ cloudServiceName + " "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
		}
		return instancesInfo;
	}

	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public SnapShotInfo getSnapShotInfo(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("snapShotId") @Identity String snapShotId,
			@ElementId("storageId") @Identity String storageId
			)throws CloudManagerFault, InvalidUserPass, InvalidRole{

		Logger logger = Logger.getLogger(this.getClass());
		SnapShotInfo ssinfo = null;
		String storageAccountName = snapShotId.substring(0,snapShotId.indexOf("_"));
 		String containerName=snapShotId.substring(snapShotId.indexOf("_")+1);
 		containerName=containerName.substring(0,containerName.indexOf("_"));
 		String url = "";
		try {
			Configuration config = getConfiguration();
			StorageManagementClient storageManagementClient = StorageManagementService.create(config);
			StorageAccountOperations storageAccountOperations = storageManagementClient.getStorageAccountsOperations();
			
			ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);
			VirtualMachineDiskGetResponse vmd = computeManagementClient.getVirtualMachineDisksOperations().getDisk(storageId);
			
			Configuration blobconfig = null;
	 		blobconfig = AzureUtil.getBlobConfiguration(storageAccountName, storageAccountOperations.getKeys(storageAccountName).getPrimaryKey());
        	
        	BlobContract blobContract = BlobService.create(blobconfig);
        	GetBlobResult blobRes = blobContract.getBlob(containerName, snapShotId);
        	
        	url = AzureUtil.getVHDUrl(storageAccountName, containerName, snapShotId);
        	//for(Container container:blobContract.listContainers().getContainers()){
        	//	if(container.getName().equals(containerName)){
        	//		url = container.getUrl() + "/" + snapShotId;
           	//		break;
        	//	}
        	//}
        	
			if(blobRes!=null){
				ssinfo = new SnapShotInfo();
				ssinfo.setStorageAccountName(storageAccountName);
				ssinfo.setContainerName(containerName);
				ssinfo.setVHDFileName(snapShotId);
				ssinfo.setFileSize(vmd.getLogicalSizeInGB());
				ssinfo.setOSType(vmd.getOperatingSystemType());
				ssinfo.setUrl(url);
				return ssinfo;
			}
		}catch(ServiceException e){
			logger.error("getSnapShotInfo: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("getSnapShotInfo: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
		}
		return ssinfo;
	}

	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public List<String> getContainerByStorageAccount(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("storageAccountName") @Identity String storageAccountName
			)throws CloudManagerFault, InvalidUserPass, InvalidRole{
		List<String> containers = new ArrayList<>();
		Logger logger = Logger.getLogger(this.getClass());

		try {
	        Configuration config = getConfiguration();
			StorageManagementClient storageManagementClient = StorageManagementService.create(config);

			StorageAccountOperations storageAccountOperations = storageManagementClient.getStorageAccountsOperations();

			StorageAccount sa = storageAccountOperations.get(storageAccountName).getStorageAccount();
			if(sa.getName().equals(storageAccountName)){
				StorageAccountGetKeysResponse storageAccountGetKeysResponse = storageAccountOperations.getKeys(sa.getName());

		        Configuration blobconfig = null;
		        blobconfig = AzureUtil.getBlobConfiguration(sa.getName(),storageAccountGetKeysResponse.getPrimaryKey());
                BlobContract service = BlobService.create(blobconfig);

                ListContainersResult listContainersResult = service.listContainers();
                for (Container container:listContainersResult.getContainers()) {
                	containers.add(container.getName());
                }
			}
			logger.info("Successful in getting Container by Storage Account ...");
		} catch (Exception e) {
			throw new CloudManagerFault(e.getMessage(), e);
		}
		return containers;
	}
	
	@Override
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right={SystemPrivilegeMode.READ, SystemPrivilegeMode.ADD})
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public void deleteStorageBackup(
			@ElementId("storageBackupURL") @Identity String storageBackupURL
			) throws CloudManagerFault {
		StorageAccountOperations storageAccountOperations = null;
		Logger logger = Logger.getLogger(this.getClass());
		
		try {
			Configuration config = getConfiguration();
			StorageManagementClient storageManagementClient = StorageManagementService.create(config);
			storageAccountOperations = storageManagementClient.getStorageAccountsOperations();
			
			String storageAccountName = AzureUtil.getStorageAccountNameFromVHD(storageBackupURL);
        	String containerName = AzureUtil.getContainerNameFromVHD(storageBackupURL);
        	String vhdFileName = AzureUtil.getVHDFileNameFromVHD(storageBackupURL);
			
	        AzureUtil.AzureServiceRequestFilter requestFilter = new AzureUtil.AzureServiceRequestFilter();
	        AzureUtil.AzureServiceResponseFilter responseFilter = new AzureUtil.AzureServiceResponseFilter(); 
	        
	        Configuration blobconfig = null;
			blobconfig = AzureUtil.getBlobConfiguration(storageAccountName, storageAccountOperations.getKeys(storageAccountName).getPrimaryKey());
			BlobContract blobContract = BlobService.create(blobconfig).withRequestFilterLast(requestFilter).withResponseFilterLast(responseFilter);
			blobContract.deleteBlob(containerName, vhdFileName);
        	responseFilter.close();
        	
        	logger.info("deleteStorageBackup: RequestId:" + responseFilter.getRequestId());
        	
		}catch(ServiceException e){
			logger.error("deleteStorageBackup: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("deleteStorageBackup: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
		}
	}
	
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right={SystemPrivilegeMode.READ, SystemPrivilegeMode.ADD})
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public CloudStorage restoreStorage(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("SnapShotInfo") @NotNull @Into AzureRestoreStorageRequest restoreRequest
			)throws CloudManagerFault, InvalidUserPass, InvalidRole {

    	Logger logger = Logger.getLogger(this.getClass());

    	Configuration config = getConfiguration();
		ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);
		VirtualMachineDiskOperations virtualMachineDiskOperations = computeManagementClient.getVirtualMachineDisksOperations();
		
		AzureUtil.AzureServiceRequestFilter request = new AzureUtil.AzureServiceRequestFilter();
        AzureUtil.AzureServiceResponseFilter response = new AzureUtil.AzureServiceResponseFilter(); 
		
		try{
			StorageManagementClient storageManagementClient = StorageManagementService.create(config);
			StorageAccountOperations storageAccountOperations = storageManagementClient.getStorageAccountsOperations();
			
			VirtualMachineDiskGetResponse vmd = computeManagementClient.getVirtualMachineDisksOperations().getDisk(restoreRequest.getSourceDiskName());
			
			String storageAccountName = restoreRequest.getSnapShotInfo().getStorageAccountName();
			String containerName = restoreRequest.getSnapShotInfo().getContainerName();
			String vhdFileName = AzureUtil.getVHDFileNameFromVHD(vmd.getMediaLinkUri().toString());
			String snapshot = restoreRequest.getSnapShotInfo().getVHDFileName().replace(storageAccountName,"").replace(containerName, "").replace("_", "");
			snapshot = AzureUtil.getConvertSnapshotValue(snapshot);
			request.setHeader("x-ms-copy-source","/"+storageAccountName+"/"+containerName+"/"+vhdFileName+"?snapshot="+snapshot);
			
			Configuration blobconfig = null;
			blobconfig = AzureUtil.getBlobConfiguration(storageAccountName, storageAccountOperations.getKeys(storageAccountName).getPrimaryKey());
			BlobContract blobContract = BlobService.create(blobconfig).withRequestFilterLast(request).withResponseFilterLast(response);
			
			String backUpName =vhdFileName+"_Restore";
			
			blobContract.createBlockBlob(containerName, backUpName, null);
			
			String url = AzureUtil.getVHDUrl(storageAccountName, containerName, backUpName);

			VirtualMachineDiskCreateParameters vmDiskParams = new VirtualMachineDiskCreateParameters();
			vmDiskParams.setOperatingSystemType(restoreRequest.getSnapShotInfo().getOSType());
        	vmDiskParams.setMediaLinkUri(new URI(url));
        	vmDiskParams.setLabel(restoreRequest.getDiskName());
        	vmDiskParams.setName(restoreRequest.getDiskName());

        	OperationResponse operationResponse = virtualMachineDiskOperations.createDisk(vmDiskParams);

        	response.close();
        	
        	for(;;){
        		OperationStatusResponse operationStatusResponse = computeManagementClient.getOperationStatus(operationResponse.getRequestId());
     			if (!operationStatusResponse.getStatus().equals(OperationStatus.InProgress)) {
     				logger.info("creating " + getClass().getSimpleName() + "...");
     				if(operationStatusResponse.getStatus().equals(OperationStatus.Succeeded)){
     					logger.info("successful in creating " + getClass().getSimpleName() + "...");
     				}else{
     					logger.error("failed in creating " + operationStatusResponse.getError());
	 					throw new CloudManagerFault(operationStatusResponse.getError().getMessage(),new Exception(operationStatusResponse.getError().getCode()));
     				}
     				break;
     			}
        	}
		}catch(ServiceException e){
			logger.error("restoreStorage: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("restoreStorage: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
		}
		return null;
	}

	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right={SystemPrivilegeMode.READ, SystemPrivilegeMode.ADD})
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public CloudStorage addStorage(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("CreateStorageRequest") @NotNull @Into AzureCreateStorageRequest request
			) throws CloudManagerFault, InvalidUserPass, InvalidRole {

    	AzureUtil.AzureServiceResponseFilter reponseFilter = null;
		try{
	        Configuration config = getConfiguration();
        	ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);

        	DeploymentGetResponse deploy =  computeManagementClient.getDeploymentsOperations().getBySlot(request.getCloudServiceName(), DeploymentSlot.Production);

        	VirtualMachineDataDiskCreateParameters vmDiskParams = new VirtualMachineDataDiskCreateParameters();
        	vmDiskParams.setHostCaching(request.getVirtualHardDiskHostCaching());

        	URI mediaLinkUri = new URI(AzureUtil.getVHDUrl(request.getStorageAccountName(), request.getContainerName(), request.getStorageName()));

        	vmDiskParams.setMediaLinkUri(mediaLinkUri);
        	vmDiskParams.setLogicalDiskSizeInGB(request.getStorageSize());
        	vmDiskParams.setLogicalUnitNumber(Integer.parseInt(request.getFlavor()));

        	reponseFilter = new AzureUtil.AzureServiceResponseFilter();
        	VirtualMachineDiskOperations virtualMachineDiskOperations = computeManagementClient.withResponseFilterLast(reponseFilter).getVirtualMachineDisksOperations();
    		OperationResponse operationResponse = virtualMachineDiskOperations.createDataDisk(request.getCloudServiceName(),deploy.getName(),request.getInstanceName(),vmDiskParams);

        	Logger logger = Logger.getLogger(this.getClass());
	        logger.info("RequestId:" + operationResponse.getRequestId());

        	for(;;){
        		OperationStatusResponse operationStatusResponse = computeManagementClient.getOperationStatus(operationResponse.getRequestId());
     			if (!operationStatusResponse.getStatus().equals(OperationStatus.InProgress)) {
     				logger.info("creating " + getClass().getSimpleName() + "...");
     				if(operationStatusResponse.getStatus().equals(OperationStatus.Succeeded)){
     					logger.info("successful in creating " + getClass().getSimpleName() + "...");
     				}else{
     					logger.error("failed in creating " + operationStatusResponse.getError());
	 					throw new CloudManagerFault(operationStatusResponse.getError().getMessage(),new Exception(operationStatusResponse.getError().getCode()));
     				}
     				break;
    			}
     			else { //if (operationStatusResponse.getStatus().equals(OperationStatus.InProgress)) {
     	 			AzureAsyncExecution.getSingleton().put(new AzureAsyncOperationStatus(computeManagementClient,operationResponse.getRequestId(),"VirtualMachineDiskOperations.createDataDisk(("+request.getCloudServiceName()+","+deploy.getName()+","+request.getInstanceName()+")"));
     				break;
     			}
        	}
    	}catch(ServiceException e){
    		if (reponseFilter != null && reponseFilter.getCalled() > 0) {
        		switch (reponseFilter.getStatus())
        		{
        		case org.apache.http.HttpStatus.SC_OK:
        		case org.apache.http.HttpStatus.SC_ACCEPTED:
        		case org.apache.http.HttpStatus.SC_CREATED:
        			return getResourceOperator(IStorageOperator.class).createStorage(request);
        		}
    		}
    		throw new CloudManagerFault(e.getMessage(), e);
    	}catch(Exception e){
    		throw new CloudManagerFault(e.getMessage(), e);
    	}
		reponseFilter.close();
		return getResourceOperator(IStorageOperator.class).createStorage(request);
	}

	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right={SystemPrivilegeMode.READ, SystemPrivilegeMode.ADD})
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public void atachStorage(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("storageId") @NotNull String storageId,
			@ElementId("CreateStorageRequest") @NotNull @Into AzureCreateStorageRequest request
			) throws CloudManagerFault, InvalidUserPass, InvalidRole {

		Logger logger = Logger.getLogger(this.getClass());
		try{
			logger.info("atachStorage() start");
			AzureAttachStorage azureAtachStorage = new AzureAttachStorage(getConfiguration(),storageId,request);
			if (AzureOptionPropertyConstants.azure_api_mode.value().equalsIgnoreCase("async")) {
				AzureAsyncExecution.getSingleton().put(azureAtachStorage);
			}else{
				azureAtachStorage.execute();
			}
		}catch(Exception e){
    		throw new CloudManagerFault(e.getMessage(), e);
    	}
		finally {
			logger.info("atachStorage() end");			
		}
	}

	@Override
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right={SystemPrivilegeMode.READ, SystemPrivilegeMode.ADD})
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public Instance createInstance(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("AzureCreateInstanceRequest") @NotNull @Into AzureCreateInstanceRequest createRequest
			) throws CloudManagerFault, InvalidUserPass, InvalidRole {

		Logger logger = Logger.getLogger(this.getClass());
		try{
			logger.info("createInstance() start");
			final ICloudContext context = SessionService.current().get(ICloudContext.class);
			AzureCreateInstance azureCreateInstance = new AzureCreateInstance(getConfiguration(),createRequest,region,context.getAccessDestionation().getCloudUser().getAccessKey());
		    if (AzureOptionPropertyConstants.azure_api_mode.value().equalsIgnoreCase("async")) {
				AzureAsyncExecution.getSingleton().put(azureCreateInstance);		    	
		    }
		    else {
		    	azureCreateInstance.execute();
		    }
		}catch(Exception e){
    		throw new CloudManagerFault(e.getMessage(), e);
    	}
		finally {
			logger.info("createInstance() end");			
		}
		
		Instance instance = new Instance();
		return instance;
	}

	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
    public List<AzureImage> getImagesWithFilter(
    	@ElementId("roleId") @Identity String roleId,
		@ElementId("region") @NotNull @Size(max=64) String region,
        @ElementId("filter") @Identity String filter
       ) throws CloudManagerFault, InvalidUserPass, InvalidRole {
		List<AzureImage> images = new ArrayList<>();
		VirtualMachineOSImageListResponse virtualMachineOSImageListResponse;
		try {
			Configuration config = getConfiguration();
			ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);

			virtualMachineOSImageListResponse = computeManagementClient.getVirtualMachineOSImagesOperations().list();
			for (VirtualMachineOSImageListResponse.VirtualMachineOSImage virtualMachineImage : virtualMachineOSImageListResponse.getImages()) {
	            if ((virtualMachineImage.getLocation()!=null) && virtualMachineImage.getLocation().toLowerCase().contains(region.replace("-", " "))) {
	            	if((virtualMachineImage.getLabel()!=null)&&virtualMachineImage.getLabel().contains(filter)){
		            	AzureImage img = new AzureImage();
		            	img.setImageId(virtualMachineImage.getName());
		            	img.setName(virtualMachineImage.getLabel());
		            	img.setOStype(virtualMachineImage.getOperatingSystemType());
		            	images.add(img);
	            	}
	            }
	        }
			
			for (VirtualMachineOSImageListResponse.VirtualMachineOSImage virtualMachineImage : virtualMachineOSImageListResponse.getImages()) {
				if (virtualMachineImage.getLocation()==null) {
	            	if((virtualMachineImage.getLabel()!=null)&&virtualMachineImage.getLabel().contains(filter)){
		            	AzureImage img = new AzureImage();
		            	img.setImageId(virtualMachineImage.getName());
		            	img.setName(virtualMachineImage.getLabel());
		            	img.setOStype(virtualMachineImage.getOperatingSystemType());
		            	images.add(img);
	            	}
	            }
			}
			
		} catch(Exception e){
    		throw new CloudManagerFault(e.getMessage(), e);
    	}
		return images;
	}

	@Override
	public void start() {
	}

	@Override
	public void stop() {
	}
}