/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.HinemosManagerMain;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.commons.util.JdbcBatchExecutor;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.monitor.bean.ConfirmConstant;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.model.NotifyEventInfoEntity;
import com.clustercontrol.notify.monitor.model.EventLogEntity;


/**
 * イベント情報を更新するクラス<BR>
 *
 * @version 4.0.0
 * @since 3.0.0
 */
public class OutputEvent implements DependDbNotifier {

	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( OutputEvent.class );

	/**
	 * 最終出力日時
	 * cc_event_logテーブルは主キーに出力日時を含む。
	 * 同じプラグインID,監視項目ID, ファシリティIDの同一イベントを出力するためには、出力日時を変更する必要がある
	 * **/
	private volatile static long _lastOutputDateTime = 0;

	/**
	 * イベントの出力を行います。
	 * 
	 * @param message 出力・通知情報
	 */
	@Override
	public void notify(NotifyRequestMessage message){
		outputEvent(message.getOutputInfo(), message.getNotifyId());
	}

	private void outputEvent(OutputBasicInfo outputInfo, String notifyId) {
		outputEvent(outputInfo, notifyId, null, null);

	}

	/**
	 * イベントの出力を行います。
	 * 
	 * @param outputInfo 出力・通知情
	 * @param notifyId 通知ID
	 * @param ownerRoleId
	 * @param notifyEventInfoEntity
	 * @return
	 */
	private EventLogEntity outputEvent(OutputBasicInfo outputInfo,
			String notifyId, String ownerRoleId,
			NotifyEventInfoEntity notifyEventInfoEntity) {
		m_log.debug("outputEvent() notifyId=" + notifyId + ", ownerRoleId=" + ownerRoleId);
		// FIXME
		if(notifyId == null || "SYS".equals(outputInfo.getPluginId())) {
			// インターナルイベントの場合
			return this.insertEventLog(outputInfo, ConfirmConstant.TYPE_UNCONFIRMED);
		} else {
			if (notifyEventInfoEntity == null) {
				try {
					notifyEventInfoEntity = QueryUtil.getNotifyEventInfoPK(notifyId);
				} catch (NotifyNotFound e) {
					m_log.debug("notify not found.", e);
				}
			}

			if (notifyEventInfoEntity != null) {
				Integer eventNormalState = getEventNormalState(notifyEventInfoEntity, outputInfo);
				return insertEventLog(outputInfo, eventNormalState, ownerRoleId);
			}
		}

		return null;
	}

	private Integer getEventNormalState(NotifyEventInfoEntity notifyEventInfo, OutputBasicInfo outputInfo) {
		int priority = outputInfo.getPriority();
		switch (priority) {
		case PriorityConstant.TYPE_INFO:
			return notifyEventInfo.getInfoEventNormalState();
		case PriorityConstant.TYPE_WARNING:
			return notifyEventInfo.getWarnEventNormalState();
		case PriorityConstant.TYPE_CRITICAL:
			return notifyEventInfo.getCriticalEventNormalState();
		case PriorityConstant.TYPE_UNKNOWN:
			return notifyEventInfo.getUnknownEventNormalState();

		default:
			break;
		}
		m_log.warn("priority=" + priority + ", id=" + notifyEventInfo.getId() + ", output=" + outputInfo.getMonitorId());
		return null;
	}

	public EventLogEntity insertEventLog(OutputBasicInfo output, int confirmState) {
		return insertEventLog(output, confirmState, null);
	}

	/**
	 * イベント情報を作成します。
	 * @param output 出力情報
	 * @param confirmState 確認フラグ ConfirmConstant.TYPE_UNCONFIRMED / ConfirmConstant.TYPE_CONFIRMED
	 * @param ownerRoleId
	 * @return
	 */
	public EventLogEntity insertEventLog(OutputBasicInfo output, int confirmState, String ownerRoleId) {
		if (m_log.isDebugEnabled())
			m_log.debug("start event creation. (monitorId=" + output.getMonitorId() + ", pluginId=" + output.getPluginId()
					+ ", facilityId=" + output.getFacilityId() + ", generationDate=" + output.getGenerationDate()
					+ ", priority=" + output.getPriority() + ", message=" + output.getMessage() + ", ownerRoleId=" + ownerRoleId + ")");

		// 出力日時を生成
		Timestamp outputDate = createOutputDate();

		// インスタンス生成
		EventLogEntity entity = null;
		if (ownerRoleId != null) {
			entity = new EventLogEntity(output.getMonitorId(),
					output.getSubKey(),
					output.getPluginId(),
					outputDate,
					output.getFacilityId(),
					ownerRoleId);
		} else {
			entity = new EventLogEntity(output.getMonitorId(),
					output.getSubKey(),
					output.getPluginId(),
					outputDate,
					output.getFacilityId());
		}

		entity.setApplication(output.getApplication());
		entity.setComment("");
		entity.setCommentDate(null);
		entity.setCommentUser("");
		entity.setConfirmDate(null);
		entity.setConfirmFlg(new Integer(confirmState));
		entity.setConfirmUser("");
		entity.setDuplicationCount(new Long(0));
		entity.setGenerationDate(new Timestamp(output.getGenerationDate()));
		entity.setInhibitedFlg(new Integer(YesNoConstant.TYPE_NO));
		String message = output.getMessage();
		if (message == null) {
			message = "";
		}
		if (message.length() > 255) {
			entity.setMessage(message.substring(0, 255));
		} else {
			entity.setMessage(message);
		}
		entity.setMessageId(output.getMessageId());
		entity.setMessageOrg(output.getMessageOrg());
		entity.setPriority(new Integer(output.getPriority()));
		entity.setScopeText(output.getScopeText());

		if (m_log.isDebugEnabled())
			m_log.debug("created event successfully. (monitorId=" + output.getMonitorId() + ", pluginId=" + output.getPluginId()
					+ ", facilityId=" + output.getFacilityId() + ", generationDate=" + output.getGenerationDate()
					+ ", priority=" + output.getPriority() + ", message=" + output.getMessage() + ")");

		return entity;
	}

	/**
	 * イベントの出力日時を払い出します。
	 * 前回払い出した日時とは重複しないように払い出します。
	 * 
	 * @return 出力日時
	 */
	private static synchronized Timestamp createOutputDate(){
		// 現在時刻を取得
		long now = System.currentTimeMillis();
		now = now - now % HinemosManagerMain._instanceCount + HinemosManagerMain._instanceId;
		long outputDateTime = 0;

		if((_lastOutputDateTime - 1000) < now && now <= _lastOutputDateTime){
			// 現在時刻と最終出力日時の誤差が1秒以内であり（時刻同期により大幅にずれた場合を想定）、
			// 現在時刻が最後に払い出した出力日時より昔の場合は、最終出力日時より1msだけ進める
			outputDateTime = _lastOutputDateTime + HinemosManagerMain._instanceCount;
			if (m_log.isDebugEnabled()) {
				m_log.debug("create OutputDate=" + outputDateTime);
			}
		} else {
			outputDateTime = now;
		}

		_lastOutputDateTime = outputDateTime;

		return new Timestamp(outputDateTime);
	}

	/**
	 * 通知失敗時の内部エラー通知を定義します
	 */
	@Override
	public void internalErrorNotify(String notifyId, String msgID, String detailMsg) throws Exception {
		//FIXME
		// 何もしない
	}

	public void notify(List<NotifyRequestMessage> msgList) throws NotifyNotFound {
		if (msgList.isEmpty()) {
			return;
		}
		
		m_log.debug("227 ownerRoleId=" + NotifyUtil.getOwnerRoleId(msgList.get(0), true));
		//一括通知で通知するメッセージは同一ロールなので、1個目のロールIDを利用する。
		
		NotifyEventInfoEntity notifyEvent = null;
		if (msgList.get(0).getNotifyId() != null) {
			// NOTIFY_IDを保持しないイベントも通知される（HAにおけるセルフチェック通知など)
			notifyEvent = getNotifyEventInfo(msgList.get(0));
		}
		
		List<EventLogEntity> entities = collectEntities(msgList, NotifyUtil.getOwnerRoleId(msgList.get(0), true), notifyEvent);
		JdbcBatchExecutor.execute(new EventLogEntityJdbcBatchInsert(entities));
	}

	private List<EventLogEntity> collectEntities(
			List<NotifyRequestMessage> msgList, String ownerRoleId,
			NotifyEventInfoEntity notifyEventInfoEntity) {
		List<EventLogEntity> entities = new ArrayList<EventLogEntity>();
		for (NotifyRequestMessage msg : msgList) {
			entities.add(outputEvent(msg.getOutputInfo(), msg.getNotifyId(), ownerRoleId, notifyEventInfoEntity));
		}
		return entities;
	}

	private NotifyEventInfoEntity getNotifyEventInfo(
			NotifyRequestMessage notifyRequestMessage) throws NotifyNotFound {
		return QueryUtil.getNotifyEventInfoPK(notifyRequestMessage.getNotifyId());
	}
}
