#!/bin/bash

# Copyright (C) since 2015 NTT DATA Corporation
# 
# This program is free software; you can redistribute it and/or
# Modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation, version 2.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details

########################################
#  Local Variable
########################################

ID="$1"
MANAGER_IP_ADDRESS="$2"
HINEMOS_AGENT_HOME="/opt/hinemos_agent"
DEST_FOLDER=${HINEMOS_AGENT_HOME}_${ID}
RSYSLOGD=/etc/rsyslog.d
RSYSLOG_CONF=/etc/rsyslog.conf
RSYSLOG_AGENT_CONF=/etc/rsyslog.d/rsyslog_hinemos_agent_${ID}.conf
RSYSLOG_AGENT_CONF_SRC=${HINEMOS_AGENT_HOME}/conf.template/rsyslog_hinemos_agent.conf
SYSLOG_CONF=/etc/syslog.conf

########################################
#  Functions
########################################

#
# show usage
#
function usage {
    name=`basename $0`
    cat <<EOF
Usage: ./$name ID MANAGER_IP_ADDRESS

	ID                : hinemos agent identifier
	MANAGER_IP_ADDRESS: hinemos manager ip address

For example,
	./$name 1 192.168.0.1

EOF
}
# check parameters
for OPT in $@; do
	case $OPT in
		-h|--help)
			usage
			exit 0
			;;
	esac
done

if [ "$1" == "" -o "$2" == "" ]; then
	usage
	exit 1
fi

########################################
#  Main
########################################

# Prerequisites
if [ -e $DEST_FOLDER ]; then
	echo "The destination directory (${DEST_FOLDER}) already exists."
	echo "The installation process quits."
	exit 1
fi

#copy folder
cp -r $HINEMOS_AGENT_HOME $DEST_FOLDER
rm -f $DEST_FOLDER/var/log/*

#replace ID in hinemos.cfg
sed -i -e "s/^ID=.*$/ID=_$ID/" $DEST_FOLDER/conf/hinemos_agent.cfg

#replace manager address in Agent.properties
sed -i -e "s/^managerAddress=.*$/managerAddress=http:\/\/${MANAGER_IP_ADDRESS}:8081\/HinemosWS\//" $DEST_FOLDER/conf/Agent.properties

#copy rsyslog/syslog config file
if [ "${MANAGER_IP_ADDRESS}" != "127.0.0.1" ]
then
	if [ -f ${RSYSLOG_CONF} ]; then
	
		# check rsyslog.d directory
		if [ ! -e ${RSYSLOGD} ]
		then
			mkdir ${RSYSLOGD}
	
			grep "^\$IncludeConfig /etc/rsyslog\.d/\*\.conf" $RSYSLOG_CONF > /dev/null
			if [ $? -ne 0 ]
			then
				echo "\$IncludeConfig /etc/rsyslog.d/*.conf" >> $RSYSLOG_CONF
			fi
		fi
	
		# copy rsyslog_hinemos_agent.conf
		cp ${RSYSLOG_AGENT_CONF_SRC} ${RSYSLOG_AGENT_CONF}
	
		# configure rsyslog_hinemos_agent.conf
		sed -i -e "s|%%MANAGER_IP_ADDRESS%%|${MANAGER_IP_ADDRESS}|" \
			-e "s/%%ID%%/${ID}/" \
			${RSYSLOG_AGENT_CONF}
	
		service rsyslog restart
	
	elif [ -f ${SYSLOG_CONF} ]; then
		cat >> $SYSLOG_CONF << EOS
#
# Hinemos Agent ${ID} (for syslog monitoring)
#
*.info;mail.none;authpriv.none;cron.none		@${MANAGER_IP_ADDRESS}:514
EOS
	
		service syslog restart
	fi
fi

#copy service file
cp /etc/init.d/hinemos_agent /etc/init.d/hinemos_agent_$ID
update-rc.d hinemos_agent_${ID} defaults

#print info
echo "The new agent with ID [${ID}] was created. The information of the new agent is as followings.

Folder: $DEST_FOLDER
Service file: /etc/init.d/hinemos_agent_$ID

"

exit 0

