/*
 
Copyright (C) NEC Corporation 2012. All Rights Reserved. 
Copyright (C) NEC Soft, Ltd. 2012. All Rights Reserved. 
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.necsoft.hinemos.webclient.jobmanagement.util;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.clustercontrol.fault.JobInvalid;
import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.bean.JudgmentObjectConstant;
import com.clustercontrol.notify.util.NotifyGroupIdGenerator;
import com.clustercontrol.ws.jobmanagement.JobInfo;
import com.clustercontrol.ws.jobmanagement.JobManagementUserInfo;
import com.clustercontrol.ws.jobmanagement.JobNotificationsInfo;
import com.clustercontrol.ws.jobmanagement.JobObjectInfo;
import com.clustercontrol.ws.jobmanagement.JobTreeItem;
import com.clustercontrol.ws.jobmanagement.JobWaitRuleInfo;
import com.clustercontrol.ws.notify.NotifyRelationInfo;
import com.necsoft.hinemos.webclient.lang.WebMessages;

/**
 * ジョブユーティリティクラス
 * 
 * 以下ィを提供します。<BR>
 * <li>ジョブツリーアイテムに関するユーティリティ
 * <li>ログインユーザが参照可能なジョブユニットかどうかをチェックするユーティリティ
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class JobUtil {
	/** ログ出力のインスタンス<BR> */
	protected static Log m_log = LogFactory.getLog( JobUtil.class );

	/** コピーしたジョブIDの接頭語 */
	private static final String COPY_OF = "Copy_Of_";

	/**
	 * 引数で指定されたジョブツリーアイテムのコピーを作成する
	 * 
	 * @param original コピー元ジョブツリーアイテム
	 * @return コピーとして作成されたジョブツリーアイテム
	 */
	private static JobTreeItem copy(JobTreeItem original) {
		JobTreeItem clone = null;
		if(original != null){
			clone = JobTreeItemUtil.clone(original, null);
		}

		return clone;
	}

	/**
	 * @param original コピー元ジョブツリーアイテム
	 * @param top コピー元ジョブツリーアイテム
	 * @return コピーとして作成されたジョブツリーアイテム
	 */
	public static JobTreeItem copy(JobTreeItem original, JobTreeItem top, String jobunitId) {
		JobTreeItem clone = copy(original);
		clone.getData().setJobunitId(jobunitId);

		//待ち条件を削除する
		deleteWaitRule(clone);

		//ジョブIDの変更
		changeJobId(clone, top, clone);

		//変更有無フラグの変更
		setEditFlg(clone);

		return clone;
	}

	/**
	 * 所属しているジョブユニットのeditフラグをtrueに変更する。
	 */
	public static void setEditFlg(JobTreeItem item) {
		JobInfo info = item.getData();
		if (info.getType() == JobConstant.TYPE_JOBUNIT) {
			m_log.debug("info.isEdit " + info.isEdit() + " -> true");
			info.setEdit(true);
		} else {
			JobTreeItem parent = item.getParent();
			if (parent == null) {
				m_log.warn("setEditFlg() error, parent == null");
			} else {
				setEditFlg(item.getParent());
			}
		}
	}

	/**
	 * 例1：COPY_OF_(13)_job0001をjob0001に変換させる。
	 * 例2：COPY_OF_job0001をjob0001に変換させる。
	 */
	private static String getBaseJobId(String jobId) {
		// COPY_OF_(13)_job0001 にマッチさせる。
		Pattern p_number = Pattern.compile(COPY_OF + "\\[1-9][0-9]*\\_");
		Matcher m_number = p_number.matcher(jobId);
		if (m_number.find()) {
			return jobId.substring(m_number.end());
		}

		// COPY_OF_job0001 にマッチさせる。
		Pattern p_org = Pattern.compile(COPY_OF);
		Matcher m_org = p_org.matcher(jobId);
		if (m_org.find()) {
			return jobId.substring(m_org.end());
		}

		// マッチしなかったら、そのまま返す。
		return jobId;
	}

	/**
	 * ジョブIDを一括変更する<BR>
	 * ジョブツリーアイテムのツリー階層の全てに対して一括変更する。<BR>
	 * topとcloneに指定されたジョブツリーアイテムを、ジョブIDの重複チェック対象とする。<BR>
	 * ジョブIDが重複した場合、コピーしたジョブIDの接頭語とカウンタを付加してジョブIDを決定する。
	 * 
	 * @param item ジョブID変更対象のジョブツリーアイテム
	 * @param top ジョブIDの重複チェック対象のジョブツリーアイテム
	 * @param clone ジョブIDの重複チェック対象のジョブツリーアイテム
	 */
	private static void changeJobId(JobTreeItem item, JobTreeItem top, JobTreeItem clone) {
		if(item == null || top == null)
			return;

		//ジョブIDを変更
		JobInfo info = item.getData();
		if(info != null && info instanceof JobInfo){
			int count = 0;
			String jobId = "";
			jobId = info.getId();
			String baseJobId = getBaseJobId(jobId);

			while(true){
				if(!findJobId(jobId, top) && !findJobId(jobId, clone)){
					break;
				}
				if(count == 0){
					jobId = COPY_OF;
				} else {
					jobId = COPY_OF + count + "_";
				}
				jobId += baseJobId;
				count++;

				// 最大で999とする。
				if (count == 1000) {
					break;
				}
			}
			info.setId(jobId.toString());

			// ジョブユニットの場合のみジョブユニットIDを上書き
			if (info.getType() == JobConstant.TYPE_JOBUNIT) {
				info.setJobunitId(jobId.toString());
				//m_log.debug("changeJobId() setJobunitId = " + jobId.toString());
			}
			//else {
			//	info.setJobunitId(item.getParent().getData().getJobunitId());
			//	m_log.debug("changeJobId() setJobunitId = " + item.getParent().getData().getJobunitId() + " from jobid = " + item.getParent().getData().getJobunitId());
			//}

			// 通知グループIDの変更
			List<JobNotificationsInfo> jobNotifyList = info.getNotifications();
			Iterator<JobNotificationsInfo> itrJob = jobNotifyList.iterator();

			Iterator<NotifyRelationInfo> itrNotify = null;
			List<NotifyRelationInfo> ctNotify = null;
			List<NotifyRelationInfo> modCtNotify = null;

			NotifyRelationInfo notify = null;
			NotifyRelationInfo modNotify = null;	//変更後格納用

			// 新しい通知グループIDを作成
			String newNotifyGroupId = NotifyGroupIdGenerator.createNotifyGroupIdJob(info.getJobunitId(), info.getId(), 0);
			//m_log.debug("changeJobId() newNotifyGroupId = " + newNotifyGroupId);
			// ジョブの通知の種類（開始・通知・警告・危険）分調べる
			while(itrJob.hasNext()){
				JobNotificationsInfo jobNotify = itrJob.next();
				jobNotify.setNotifyGroupId(newNotifyGroupId);

				// 通知関連情報を調査
				ctNotify = jobNotify.getNotifyId();

				if(ctNotify != null) {
					itrNotify = ctNotify.iterator();
					modCtNotify = new ArrayList<NotifyRelationInfo>();

					// 通知関連情報内の通知グループIDの変更
					while(itrNotify.hasNext()){
						notify = itrNotify.next();

						// ディープコピー（通知グループIDは新規）を作成
						modNotify = new NotifyRelationInfo();
						modNotify.setNotifyGroupId(newNotifyGroupId);
						modNotify.setNotifyId(notify.getNotifyId());
						modNotify.setNotifyType(notify.getNotifyType());
						modNotify.setNotifyFlg(notify.getNotifyFlg());

						modCtNotify.add(modNotify);

					}

					if(modCtNotify.size() != 0){
						jobNotify.getNotifyId().clear();
						jobNotify.getNotifyId().addAll(modCtNotify);
					}
				}
			}
		}

		//子JobTreeItemを取得
		List<JobTreeItem> childrens = item.getChildren();
		for(int i = 0; i < childrens.size(); i++){
			childrens.get(i).getData().setJobunitId(info.getJobunitId());
			//m_log.debug("changeJobId() set childrens[i] " + info.getJobunitId());
			changeJobId(childrens.get(i), top, clone);
		}
	}

	/**
	 * ジョブツリーアイテムからジョブ待ち条件情報を削除する<BR>
	 * ジョブツリーアイテムのツリー階層の全てが削除対象
	 * 
	 * @param item ジョブ待ち条件情報を削除するジョブツリーアイテム
	 */
	private static void deleteWaitRule(JobTreeItem item) {
		if(item == null)
			return;

		JobInfo info = item.getData();
		if(info != null){
			//待ち条件を削除する
			JobWaitRuleInfo waitRule = info.getWaitRule();
			if(waitRule != null && waitRule.getObject() != null){
				waitRule.getObject().clear();
			}
		}

		//子JobTreeItemを取得
		List<JobTreeItem> childrens = item.getChildren();
		for(int i = 0; i < childrens.size(); i++){
			deleteWaitRule(childrens.get(i));
		}
	}

	/**
	 * ジョブツリーアイテムからジョブIDが一致するインスタンスの有無を返す<BR>
	 * ジョブツリーアイテムのツリー階層の全てが検索対象
	 * 
	 * @param jobId ジョブID
	 * @param item ジョブツリーアイテム
	 * @return ジョブIDが一致するジョブツリーアイテムがあればtrue、なければfalse。
	 */
	public static boolean findJobId(String jobId, JobTreeItem item) {
		boolean find = false;

		//ジョブIDをチェック
		JobInfo info = item.getData();
		if(info != null && info instanceof JobInfo){
			if(jobId.compareTo(info.getId()) == 0){
				find = true;
				return find;
			}
		}

		//子JobTreeItemを取得
		List<JobTreeItem> childrens = item.getChildren();
		for(int i = 0; i < childrens.size(); i++){
			find = findJobId(jobId, childrens.get(i));
			if(find){
				break;
			}
		}

		return find;
	}

	/**
	 * ジョブツリーアイテム内のジョブIDが一致するインスタンスの有無を返す<BR>
	 * 
	 * @param item ジョブツリーアイテム
	 * @param isTop ツリーのトップか否か
	 * @return ジョブIDが一致するジョブ/ネット/ユニットが存在すればtrue、なければfalse。
	 * @throws JobInvalid
	 */
	public static void findDuplicateJobId(JobTreeItem item, boolean isTop) throws JobInvalid{
		m_log.debug("findDuplicateJobId() start : isTop = " + isTop);

		// 自身がtopの場合は何もしない
		if(!isTop){

			// 自身がジョブユニット/ジョブネット/ジョブの場合
			if(item.getData() instanceof JobInfo){
				String jobId = item.getData().getId();
				m_log.debug("findDuplicateJobId() jobId = " + jobId);

				List<JobTreeItem> children = item.getChildren();
				for (JobTreeItem child : children) {
					m_log.debug("findDuplicateJobId() child = " + child.getData().getId());

					// 配下のツリーにトップのジョブIDが含まれるか？
					if(findJobId(jobId, child)){

						// jobunitid内にjobidが重複している
						m_log.debug("findDuplicateJobId() jobId is in child " + JobTreeItemUtil.getPath(child));
						Object[] args = {jobId, child.getData().getJobunitId()};
						throw new JobInvalid(WebMessages.getString("message.job.65",args));
					} else {
						m_log.debug("findDuplicateJobId() jobId is not in child " + JobTreeItemUtil.getPath(child));
					}
				}
			}
		}

		List<JobTreeItem> children = item.getChildren();
		for (JobTreeItem child : children) {
			m_log.debug("findDuplicateJobId() call child " + child.getData().getId());
			findDuplicateJobId(child, false);
		}

		m_log.debug("findDuplicateJobId() success!!");
	}

	/**
	 * ジョブツリーアイテム内のジョブユニットIDが一致するインスタンスの有無を返す<BR>
	 * 
	 * @param item ジョブツリーアイテム
	 * @return ジョブユニットIDが一致するユニットが存在すればtrue、なければfalse。
	 * @throws JobInvalid
	 */
	public static void findDuplicateJobunitId(JobTreeItem item) throws JobInvalid{
		m_log.debug("findDuplicateJobunitId() start " + JobTreeItemUtil.getPath(item));

		JobTreeItem top = item.getChildren().get(0);
		List<JobTreeItem> jobunits = top.getChildren();

		HashSet<String> set = new HashSet<String>();
		for (JobTreeItem jobunit : jobunits) {
			// ジョブユニットID
			String jobunitId = jobunit.getData().getJobunitId();
			m_log.debug("findDuplicateJobunitId() jobunitId = " + jobunitId);

			if(set.contains(jobunitId)){
				m_log.debug("findDuplicateJobunitId() hit " + jobunitId);
				Object[] args = {jobunitId};
				throw new JobInvalid(WebMessages.getString("message.job.64",args));
			} else {
				m_log.debug("findDuplicateJobunitId() add " + jobunitId + " to set");
				set.add(jobunitId);
			}
		}

		m_log.debug("findDuplicateJobunitId() success!!");
	}

	/**
	 * ジョブツリーアイテムの最上位のインスタンスを取得する
	 * 
	 * @param item ジョブツリーアイテム
	 * @return 最上位のジョブツリーアイテム
	 */
	public static JobTreeItem getTopJobTreeItem(JobTreeItem item) {
		if(item == null)
			return null;

		while (item.getParent() != null) {
			if(item.getParent().getData().getType() == JobConstant.TYPE_COMPOSITE){
				item = item.getParent();
				break;
			}
			else{
				item = item.getParent();
			}
		}

		return item;
	}

	/**
	 * ジョブツリーアイテムの付属するジョブユニットのインスタンスを取得する
	 * 
	 * @param item ジョブツリーアイテム
	 * @return 付属するジョブユニットのジョブツリーアイテム
	 */
	public static JobTreeItem getTopJobUnitTreeItem(JobTreeItem item) {
		if(item == null)
			return null;

		while (item.getParent() != null) {
			//m_log.debug("getTopJobUnitTreeItem() " + item.getParent().getData().getJobunitId() + "." + item.getParent().getData().getId());
			if(item.getParent().getData().getType() == JobConstant.TYPE_JOBUNIT){
				item = item.getParent();
				break;
			}
			else if(item.getData().getType() == JobConstant.TYPE_JOBUNIT){
				break;
			}
			else{
				item = item.getParent();
			}
		}

		return item;
	}

	/**
	 * ジョブツリーアイテムのジョブ待ち条件情報をチェックする
	 * 
	 * @param item ジョブ待ち条件情報をチェックするジョブツリーアイテム
	 */
	public static boolean checkWaitRule(JobTreeItem item) throws JobInvalid{
		boolean check = true;

		if(item == null)
			return check;

		if(item.getData() != null && item.getData() instanceof JobInfo){
			//ジョブID取得
			String jobId = item.getData().getId();

			//待ち条件情報を取得する
			JobWaitRuleInfo waitRule = item.getData().getWaitRule();
			if(waitRule != null && waitRule instanceof JobWaitRuleInfo &&
					waitRule.getObject() != null && waitRule.getObject().size() > 0){

				Iterator<JobObjectInfo> itr = waitRule.getObject().iterator();
				while(itr.hasNext()) {
					//判定対象を取得
					JobObjectInfo objectInfo = itr.next();
					if(objectInfo.getType() != JudgmentObjectConstant.TYPE_TIME){
						//判定対象のジョブIDが同一階層に存在するかチェック
						boolean find = false;
						String targetJobId = objectInfo.getJobId();
						List<JobTreeItem> childrens = item.getParent().getChildren();
						for(int i = 0; i < childrens.size(); i++){
							//ジョブIDをチェック
							JobInfo childInfo = childrens.get(i).getData();
							if(childInfo != null && childInfo instanceof JobInfo &&
									!jobId.equals(childInfo.getId())){
								if(targetJobId.compareTo(childInfo.getId()) == 0){
									find = true;
									break;
								}
							}
						}
						if(!find){
							String args[] = {jobId, targetJobId};
							throw new JobInvalid(WebMessages.getString("message.job.59", args));
						}
					}
				}
			}
		}

		//子JobTreeItemを取得
		List<JobTreeItem> childrens = item.getChildren();
		for(int i = 0; i < childrens.size(); i++){
			check = checkWaitRule(childrens.get(i));
			if(!check){
				break;
			}
		}

		return check;
	}

	/**
	 * ログインユーザで参照可能なジョブユニットかどうかチェックします。<BR>
	 * @param jobunit 選択されたジョブユニット
	 * @param loginUser ログインユーザのユーザID
	 * @return ログインユーザで参照可能なジョブユニットである場合true
	 */
	public static boolean isReferableJobunit (JobTreeItem jobunit, String loginUser) {

		boolean ret = false;

		List<JobManagementUserInfo> userInfo = jobunit.getData().getManagementUser();
		if (userInfo == null || userInfo.size() == 0) {
			ret = true;
		} else {
			for (JobManagementUserInfo i : userInfo) {
				if (loginUser.equals(i.getUserId())) {
					ret = true;
					break;
				}
			}
		}

		return ret;
	}
}