/*
 
Copyright (C) NEC Corporation 2012. All Rights Reserved. 
Copyright (C) NEC Soft, Ltd. 2012. All Rights Reserved. 
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.necsoft.hinemos.webclient.util;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;

import javax.xml.namespace.QName;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.WebServiceException;
import javax.xml.ws.soap.SOAPBinding;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.ws.access.AccessEndpointService;
import com.clustercontrol.ws.jobmanagement.JobEndpointService;
import com.clustercontrol.ws.monitor.MonitorEndpointService;
import com.clustercontrol.ws.repository.RepositoryEndpointService;

/**
 * Hinemosマネージャとの通信をするクラス。 HAのような複数マネージャ対応のため、このクラスを実装する。
 * 
 * Hinemosマネージャと通信できない場合は、WebServiceExceptionがthrowされる。
 * WebServiceExeptionが出力された場合は、もう一台のマネージャと通信する。
 */
public class EndpointManager {

    // ログ
    private static Log m_log = LogFactory.getLog(EndpointManager.class);

    private EndpointList endpointList;

    private final static String ACCESS = AccessEndpointService.class
            .getSimpleName();

    private final static String JOB = JobEndpointService.class.getSimpleName();

    private final static String MONITOR = MonitorEndpointService.class
            .getSimpleName();

    private final static String REPOSITORY = RepositoryEndpointService.class
            .getSimpleName();

    private String username = "";

    private String password = "";

    public int m_httpConnectTimeout = LoginManager.VALUE_HTTP_CONNECT_TIMEOUT;

    public int m_httpRequestTimeout = LoginManager.VALUE_HTTP_REQUEST_TIMEOUT;

    private class EndpointList {
        /*
         * endpointListはマネージャ一覧を示す。 これは初期化時のみ変更可能(add可能)。
         * 
         * lastSuccessEndpointは最後に成功したendpointを示す。
         * endpointList.contains(lastSuccessEndpoint)は必ずtrueになる。
         */
        private ArrayList<HashMap<String, EndpointSetting>> endpointList = 
                            new ArrayList<HashMap<String, EndpointSetting>>();

        private HashMap<String, EndpointSetting> lastSuccessEndpoint = null;

        private void add(HashMap<String, EndpointSetting> endpointMap) {
            if (lastSuccessEndpoint == null) {
                lastSuccessEndpoint = endpointMap;
            }
            endpointList.add(endpointMap);
        }

        private ArrayList<EndpointSetting> getList(String key) {
            ArrayList<EndpointSetting> list = new ArrayList<EndpointSetting>();
            list.add(lastSuccessEndpoint.get(key));
            for (HashMap<String, EndpointSetting> endpoint : endpointList) {
                if (!endpoint.equals(lastSuccessEndpoint)) {
                    list.add(endpoint.get(key));
                }
            }
            return list;
        }

        /**
         * リストの順番を変えるメソッド。 endpointListの順番で、lastSuccesEndpointをローテートする。
         * 
         * @param endpoint
         */
        private void changeEndpoint() {
            m_log.debug("changeEndpoint");
            boolean flag = false;
            for (HashMap<String, EndpointSetting> e : endpointList) {
                if (flag) {
                    lastSuccessEndpoint = e;
                    flag = false;
                    return;
                }
                if (lastSuccessEndpoint.equals(e)) {
                    flag = true;
                }
            }
            lastSuccessEndpoint = endpointList.get(0);
        }
    }

    public class EndpointSetting {
        private String key;

        private String urlPrefix;

        private Object endpoint;

        private String wsdlSuffix = "?wsdl";

        private EndpointSetting(String key, String urlPrefix) {
            this.key = key;
            this.urlPrefix = urlPrefix;
        }

        public Object getEndpoint() {
            // ログインしていない場合は、ここでログインする。
            
            if (endpoint != null) {
                return endpoint;
            }
            String tmpKey = null;
            tmpKey = ACCESS;
            if (tmpKey.equals(key)) {
                String urlStr = urlPrefix + tmpKey + wsdlSuffix;
                AccessEndpointService tmpEndpointService = null;
                try {
                    tmpEndpointService = new AccessEndpointService(new URL(
                            urlStr), new QName(
                            "http://access.ws.clustercontrol.com", tmpKey));
                } catch (MalformedURLException e) {
                    m_log.warn(
                            "getEndpoint():AccessEndpointService, "
                                    + e.getMessage(), e);
                } catch (Exception ex) {
                    throw new WebServiceException();
                }
                endpoint = tmpEndpointService.getAccessEndpointPort();
                setBindingProvider(endpoint, username, password, urlStr);
            }
            tmpKey = JOB;
            if (tmpKey.equals(key)) {
                String urlStr = urlPrefix + tmpKey + wsdlSuffix;
                JobEndpointService tmpEndpointService = null;
                try {
                    tmpEndpointService = new JobEndpointService(
                            new URL(urlStr),
                            new QName(
                                    "http://jobmanagement.ws.clustercontrol.com",
                                    tmpKey));
                } catch (MalformedURLException e) {
                    m_log.warn(
                            "getEndpoint():JobEndpointService, "
                                    + e.getMessage(), e);
                } catch (Exception ex) {
                    throw new WebServiceException();
                }
                endpoint = tmpEndpointService.getJobEndpointPort();
                setBindingProvider(endpoint, username, password, urlStr);
            }
            tmpKey = MONITOR;
            if (tmpKey.equals(key)) {
                String urlStr = urlPrefix + tmpKey + wsdlSuffix;
                MonitorEndpointService tmpEndpointService = null;
                try {
                    tmpEndpointService = new MonitorEndpointService(new URL(
                            urlStr), new QName(
                            "http://monitor.ws.clustercontrol.com", tmpKey));
                } catch (MalformedURLException e) {
                    m_log.warn(
                            "getEndpoint():MonitorEndpointService, "
                                    + e.getMessage(), e);
                } catch (Exception ex) {
                    throw new WebServiceException();
                }
                
                endpoint = tmpEndpointService.getMonitorEndpointPort();
                setBindingProvider(endpoint, username, password, urlStr);
            }
            tmpKey = REPOSITORY;
            if (tmpKey.equals(key)) {
                String urlStr = urlPrefix + tmpKey + wsdlSuffix;
                RepositoryEndpointService tmpEndpointService = null;
                try {
                    tmpEndpointService = new RepositoryEndpointService(new URL(
                            urlStr), new QName(
                            "http://repository.ws.clustercontrol.com", tmpKey));
                } catch (MalformedURLException e) {
                    m_log.warn(
                            "getEndpoint():RepositoryEndpointService, "
                                    + e.getMessage(), e);
                } catch (Exception ex) {
                    throw new WebServiceException();
                }
                endpoint = tmpEndpointService.getRepositoryEndpointPort();
                setBindingProvider(endpoint, username, password, urlStr);
            }
            return endpoint;
        }
    }

    public void init(String user, String pass, String managerAddressList,
            int httpConnectTimeout, int httpRequestTimeout)
            throws MalformedURLException {
        username = user;
        password = pass;
        endpointList = new EndpointList();
        m_httpConnectTimeout = httpConnectTimeout;
        m_httpRequestTimeout = httpRequestTimeout;

        for (String managerAddress : managerAddressList.split(",")) {

            // ユーザ/パスワードチェックを実装する必要あり。
            HashMap<String, EndpointSetting> map = 
                                    new HashMap<String, EndpointSetting>();
            String wsdlPrefix = managerAddress.trim();

            map.put(ACCESS, new EndpointSetting(ACCESS, wsdlPrefix));
            map.put(JOB, new EndpointSetting(JOB, wsdlPrefix));
            map.put(MONITOR, new EndpointSetting(MONITOR, wsdlPrefix));
            map.put(REPOSITORY, new EndpointSetting(REPOSITORY, wsdlPrefix));

            endpointList.add(map);
        }
    }

    private void setBindingProvider(Object o, String user,
            String password, String urlStr) {
        BindingProvider bp = (BindingProvider) o;
        bp.getRequestContext().put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY,
                urlStr);
        bp.getRequestContext().put(BindingProvider.USERNAME_PROPERTY, user);
        bp.getRequestContext().put(BindingProvider.PASSWORD_PROPERTY, password);

        bp.getRequestContext().put("com.sun.xml.internal.ws.connect.timeout",
                m_httpConnectTimeout * 1000);
        bp.getRequestContext().put("com.sun.xml.internal.ws.request.timeout",
                m_httpRequestTimeout * 1000);

        ((SOAPBinding) bp.getBinding()).setMTOMEnabled(true);
    }

    /**
     * 使用可能な順番でAccessEndpointを返す。
     * 
     * @return
     */
    public ArrayList<EndpointSetting> getAccessEndpoint() {
        return endpointList.getList(ACCESS);
    }

    /**
     * 使用可能な順番でJobEndpointを返す。
     * 
     * @return
     */
    public ArrayList<EndpointSetting> getJobEndpoint() {
        return endpointList.getList(JOB);
    }

    /**
     * 使用可能な順番でMonitorEndpointを返す。
     * 
     * @return
     */
    public ArrayList<EndpointSetting> getMonitorEndpoint() {
        return endpointList.getList(MONITOR);
    }

    /**
     * 使用可能な順番でRepositoryEndpointを返す。
     * 
     * @return
     */
    public ArrayList<EndpointSetting> getRepositoryEndpoint() {
        return endpointList.getList(REPOSITORY);
    }

    /**
     * Endpointの利用時にWebServiceExceptionが出たらこのメソッドを呼ぶこと。
     */
    public void changeEndpoint() {
        endpointList.changeEndpoint();
    }
}
