
#include "Parser.h"
#include "sysDep.h"
#include <string.h>
#include <assert.h>
#include <stdio.h>

extern void p__init(void); // initialize generated parsers

namespace Houken {

    PtrArray<Parser>*   Parser::_holder = NULL;
    InputBuffer*        Parser::_inp = NULL;
    TArray<Parser::Error_t>*    Parser::_errors = NULL;
    int                 Parser::_printIntermediateLevel = 0;
    bool                Parser::_bUseMemoize = true;
    PtrArray<SyntaxTree>* const Parser::_INFIX_COMMON_FATAL_ERROR = (PtrArray<SyntaxTree>*)1;

    Parser::Parser(void)
    {
        assert(_holder != NULL); // Parser::initialize() needed
        _holder->add(this);
    }


    SyntaxTree* Parser::createSyntaxTree(u32 startPos, PtrArray<SyntaxTree>* childs)
    {
        Substr ss(startPos, _curPos());
        return new SyntaxTree(this, ss, childs);
    }

    void Parser::initialize(InputBuffer* inp, int initHolderCapacity)
    {
        _inp = inp;
        _holder = new PtrArray<Parser>(initHolderCapacity);
        _errors = new TArray<Error_t>(10);
        p__init();
    }

    void Parser::finalize(void)
    {
        PtrArrayIterator<Parser> itr(_holder);
        while (itr.hasMore())
            delete itr.next();
        delete _holder;
        _holder = NULL;
        delete _errors;
        _errors = NULL;
    }

    bool Parser::strcmp(SyntaxTree* st, const char* str)
    {
        return _inp->strcmp(st->str, str);
    }
    
    void Parser::addSequenceToParent(PtrArray<Parser>* parentSeq)
    {
        parentSeq->add(this);
    }

    void Parser::addSyntaxTreeToParent(PtrArray<SyntaxTree>* arr, SyntaxTree* st)
    {
        if (st->isValidTree())
            arr->add(st);
    }

    void Parser::_reduceSyntaxTreeToParent(PtrArray<SyntaxTree>* arr, SyntaxTree* st)
    {
        for (PtrArrayIterator<SyntaxTree> itr(st->childs); itr.hasMore(); ) {
            SyntaxTree* t = itr.extract_next();
            if (t->isValidTree())
                arr->add(t);
        }
        if (! _bUseMemoize)
            delete st;      // no more need
    }

    void Parser::_reduceOpSyntaxTreeToParent(PtrArray<SyntaxTree>* arr, SyntaxTree* st)
    {
        if (st->isValidTree()) {
            if (st->numChild() > 1) {
                _reduceSyntaxTreeToParent(arr, st);
            } else {
                arr->add(st);
            }
        }
    }

    Parser* Parser::seq(Parser* p)
    {
        return new Parser_Sequence(this, p);
    }

    Parser* Parser::choice(Parser* p)
    {
        return new Parser_OrderedChoice(this, p);
    }

    Parser* Parser::star(void)
    {
        return new Parser_ZeroOrMore(this);
    }
    
    Parser* Parser::plus(void)
    {
        return new Parser_OneOrMore(this);
    }
    
    Parser* Parser::optional(void)
    {
        return new Parser_Optional(this);
    }
    
    Parser* Parser::andPred(void)
    {
        return new Parser_AndPredicate(this);
    }
    
    Parser* Parser::notPred(void)
    {
        return new Parser_NotPredicate(this);
    }

    Parser* Parser::noTree(void)
    {
        return new Parser_NoTree(this);
    }




    Parser::_ErrorPos Parser::_curErrPos(void)
    {
        _ErrorPos errPos;
        errPos.parsePos = _curPos();
        errPos.errIdx = _errors->size();
        return errPos;
    }

    void Parser::_back(Parser::_ErrorPos& errPos)
    {
        _errors->chop(errPos.errIdx);
        _inp->setPos(errPos.parsePos);
    }

    void Parser::_fail(Parser::_ErrorPos& errPos)
    {
        Error_t err = {this, _curPos()};
        _errors->add(err);
        _inp->setPos(errPos.parsePos);
    }

    SyntaxTree* Parser::_parse_sequence(PtrArray<Parser>* ps)
    {
        _ErrorPos startPos = _curErrPos();
        PtrArray<SyntaxTree>* childs = new PtrArray<SyntaxTree>(ps->size());
        int errorCutId = 0;
        u32 errorCutPos = 0;

        for (PtrArrayIterator<Parser> itr(ps); itr.hasMore(); ) {
            Parser* p = itr.next();
            SyntaxTree* st = p->parse();
            if (st->isErrorCut()) {
                errorCutId = st->errorCutId;
                errorCutPos = st->str.startPos;
            } else if (! st->isFail()) {
                p->addSyntaxTreeToParent(childs, st);
            } else {
                if (st->isFailNotError() && errorCutId > 0) {
                    SyntaxTree* e = new SyntaxTree(errorCutPos, errorCutId, this);
                    if (! _bUseMemoize)
                        delete st;
                    st = e;
                }
                _fail(startPos);
                if (! _bUseMemoize) {
                    PtrArrayIterator<SyntaxTree> stitr(childs);
                    while (stitr.hasMore())
                        delete stitr.next();
                }
                delete childs;
                return st;
            }
        }
        return createSyntaxTree(startPos.parsePos, childs);
    }
    
    SyntaxTree* Parser::_parse_orderedChoice(PtrArray<Parser>* ps)
    {
        _ErrorPos startPos = _curErrPos();

        for (PtrArrayIterator<Parser> itr(ps); itr.hasMore(); ) {
            Parser* p = itr.next();
            SyntaxTree* st = p->parse();
            if (st->isError()) {
                _fail(startPos);
                return st;
            }
            if (! st->isFail())
                return st;
            if (itr.hasMore())
                _back(startPos);
        }
        _fail(startPos);
        return _PARSE_FAILED;
    }
    
    SyntaxTree* Parser::_parse_m2n(Parser* p, int m, int n)
    {
        int count = 0;
        _ErrorPos startPos = _curErrPos();
        PtrArray<SyntaxTree>* childs = new PtrArray<SyntaxTree>(m > 0 ? m : 1);
        for (;;) {
            SyntaxTree* st = p->parse();
            if (st->isError()) {
                _fail(startPos);
                return st;
            }
            if (! st->isFail()) {
                p->addSyntaxTreeToParent(childs, st);
                if (++count >= MAX_MANY) count = MAX_MANY - 1;
                if (count >= n)
                    break;      // all ok
            } else {
                break;
            }
        }
        if (count < m) {
            _fail(startPos);
            if (! _bUseMemoize) {
                PtrArrayIterator<SyntaxTree> stitr(childs);
                while (stitr.hasMore())
                    delete stitr.next();
            }
            delete childs;
            return _PARSE_FAILED;
        }
        _errors->clean();
        return createSyntaxTree(startPos.parsePos, childs);
    }

    SyntaxTree* Parser::_parse_andPredicate(Parser* p)
    {
        _ErrorPos startPos = _curErrPos();
        SyntaxTree* st = p->parse();
        if (st->isError()) {
            _fail(startPos);
            return st;
        }
        _back(startPos);
        if (st->isValidTree())
            st->parser = this;
        return st;
    }
    
    SyntaxTree* Parser::_parse_notPredicate(Parser* p)
    {
        _ErrorPos startPos = _curErrPos();
        SyntaxTree* st = p->parse();
        if (st->isFatalError()) {
            _fail(startPos);
            return st;
        }
        _back(startPos);
        if (! st->isFail()) {
            if (! _bUseMemoize) {
                if (st->isValidTree())
                    delete st;
            }
            return _PARSE_FAILED;
        }
        return _NO_SYNTAX_TREE;
    }

    PtrArray<SyntaxTree>* Parser::_parse_infix_common(Parser* exp, Parser* op, bool rep)
    {
        _ErrorPos startPos = _curErrPos();
        SyntaxTree* st = exp->parse();
        if (st->isError()) {
            _fail(startPos);
            return _INFIX_COMMON_FATAL_ERROR;
        }
        if (st->isFail()) {
            _fail(startPos);
            return NULL;
        }
        PtrArray<SyntaxTree>* childs = new PtrArray<SyntaxTree>(3);
        childs->add(st);
        for (;;) {
            _ErrorPos midPos = _curErrPos();
            SyntaxTree* sto = op->parse();
            if (sto->isError()) {
                _fail(startPos);
                return _INFIX_COMMON_FATAL_ERROR;
            }
            if (sto->isFail()) {
                _back(midPos);
                break;
            }
            SyntaxTree* ste = exp->parse();
            if (ste->isError()) {
                _fail(startPos);
                return _INFIX_COMMON_FATAL_ERROR;
            }
            if (ste->isFail()) {
                if (! _bUseMemoize)
                    delete sto;
                _back(midPos);
                break;
            }
            childs->add(sto);
            childs->add(ste);
            if (!rep)
                break;
        }
        return childs;
    }

    SyntaxTree* Parser::_parse_op_infixl(Parser* my, Parser* exp, Parser* op)
    {
        PtrArray<SyntaxTree>* childs = _parse_infix_common(exp, op, true);
        if (childs == _INFIX_COMMON_FATAL_ERROR)
            return _FATAL_PARSER_ERROR;
        if (childs == NULL)
            return _PARSE_FAILED;
        int size = childs->size();
        if (size == 1)
            return new SyntaxTree(my, childs->get(0)->str, childs);
        SyntaxTree* st = childs->extract(0);
        for (int i = 1; i < size; i += 2) {
            SyntaxTree* op = childs->extract(i);
            SyntaxTree* exp = childs->extract(i+1);
            PtrArray<SyntaxTree>* node = new PtrArray<SyntaxTree>(3);
            node->add(st); node->add(op); node->add(exp);
            Substr ss(st->str.startPos, exp->str.endPos);
            st = new SyntaxTree(my, ss, node);
        }
        delete childs;
        return st;
    }
    
    SyntaxTree* Parser::_parse_op_infixr(Parser* my, Parser* exp, Parser* op)
    {
        PtrArray<SyntaxTree>* childs = _parse_infix_common(exp, op, true);
        if (childs == _INFIX_COMMON_FATAL_ERROR)
            return _FATAL_PARSER_ERROR;
        if (childs == NULL)
            return _PARSE_FAILED;
        int size = childs->size();
        if (size == 1)
            return new SyntaxTree(my, childs->get(0)->str, childs);
        SyntaxTree* st = childs->extract(size-1);
        for (int i = size-3; i >= 0 ; i -= 2) {
            SyntaxTree* exp = childs->extract(i);
            SyntaxTree* op = childs->extract(i+1);
            PtrArray<SyntaxTree>* node = new PtrArray<SyntaxTree>(3);
            node->add(exp); node->add(op); node->add(st);
            Substr ss(exp->str.startPos, st->str.endPos);
            st = new SyntaxTree(my, ss, node);
        }
        delete childs;
        return st;
    }
    
    SyntaxTree* Parser::_parse_op_infixn(Parser* my, Parser* exp, Parser* op)
    {
        PtrArray<SyntaxTree>* childs = _parse_infix_common(exp, op, false);
        if (childs == _INFIX_COMMON_FATAL_ERROR)
            return _FATAL_PARSER_ERROR;
        if (childs == NULL)
            return _PARSE_FAILED;
        int size = childs->size();
        if (size == 1)
            return new SyntaxTree(my, childs->get(0)->str, childs);
        SyntaxTree* st = childs->extract(0);
        if (size == 3) {
            SyntaxTree* op = childs->extract(1);
            SyntaxTree* exp = childs->extract(2);
            PtrArray<SyntaxTree>* node = new PtrArray<SyntaxTree>(3);
            node->add(st); node->add(op); node->add(exp);
            Substr ss(st->str.startPos, exp->str.endPos);
            st = new SyntaxTree(my, ss, node);
        }
        delete childs;
        return st;
    }
    
    SyntaxTree* Parser::_parse_op_prefix(Parser* exp, Parser* op)
    {
        _ErrorPos startPos = _curErrPos();
        SyntaxTree* sto = op->parse();
        if (sto->isError()) {
            _fail(startPos);
            return sto;
        }
        PtrArray<SyntaxTree>* childs = new PtrArray<SyntaxTree>(2);
        if (sto->isFail()) {
            _back(startPos);
        } else {
            childs->add(sto);
        }
        SyntaxTree* ste = exp->parse();
        if (ste->isFail()) {
            _fail(startPos);
            if (! _bUseMemoize) {
                if (sto->isValidTree())
                    delete sto;
            }
            delete childs;
            return ste;
        }
        childs->add(ste);
        return createSyntaxTree(startPos.parsePos, childs);
    }
    
    SyntaxTree* Parser::_parse_op_postfix(Parser* exp, Parser* op)
    {
        _ErrorPos startPos = _curErrPos();
        SyntaxTree* ste = exp->parse();
        if (ste->isFail()) {
            _fail(startPos);
            return ste;
        }
        PtrArray<SyntaxTree>* childs = new PtrArray<SyntaxTree>(2);
        childs->add(ste);
        _ErrorPos midPos = _curErrPos();
        SyntaxTree* sto = op->parse();
        if (sto->isError()) {
            _fail(startPos);
            if (! _bUseMemoize) {
                if (ste->isValidTree())
                    delete ste;
            }
            delete childs;
            return sto;
        }
        if (sto->isFail()) {
            _back(midPos);
        } else {
            childs->add(sto);
        }
        return createSyntaxTree(startPos.parsePos, childs);
    }

    SyntaxTree* Parser::_parse_op_ternary(Parser* exp, Parser* op1, Parser* op2)
    {
        _ErrorPos startPos = _curErrPos();
        SyntaxTree* st = exp->parse();
        if (st->isFail()) {
            _fail(startPos);
            return _PARSE_FAILED;
        }

        PtrArray<SyntaxTree>* childs = new PtrArray<SyntaxTree>(3);
        _ErrorPos midPos = _curErrPos();
        exp->addSyntaxTreeToParent(childs, st);
        
        st = op1->parse();
        if (! st->isFail()) {
            childs->add(st);
            st = exp->parse();
            if (! st->isFail()) {
                exp->addSyntaxTreeToParent(childs, st);
                st = op2->parse();
                if (! st->isFail()) {
                    childs->add(st);
                    st = exp->parse();
                    if (! st->isFail()) {
                        // success
                        exp->addSyntaxTreeToParent(childs, st);
                        return createSyntaxTree(startPos.parsePos, childs);
                    }
                }
            }
        }

        for (int i = childs->size() - 1; i > 0; --i)
            childs->remove(i);
        childs->shrink();
        if (st->isError()) {
            _fail(midPos);
            return st;
        }
        _back(midPos);
        return createSyntaxTree(startPos.parsePos, childs);
    }

    SyntaxTree* Parser::_parse_noTree(Parser* p)
    {
        SyntaxTree* st = p->parse();
        if (st->isFail())
            return st;
        if (! _bUseMemoize) {
            if (st->isValidTree())
                delete st;
        }
        return _NO_SYNTAX_TREE;
    }

    SyntaxTree* Parser::_parse_string(const char* str)
    {
        _ErrorPos startPos = _curErrPos();
        size_t len = strlen(str);
        if (_inp->remain() >= len) {
            if (!strncmp(str, _inp->addr(), len)) {
                _inp->advance(len);
                return createSyntaxTree(startPos.parsePos);
            }
        }
        _fail(startPos);
        return _PARSE_FAILED;
    }


    SyntaxTree* Parser::_parse_char(wchar_t chr)
    {
        _ErrorPos startPos = _curErrPos();
        wchar_t c;
        try {
            c = _inp->getChar();
        } catch (u32 pos) {
            _fail(startPos);
            return _PARSE_FAILED;
        }
        if (c == chr) {
            return createSyntaxTree(startPos.parsePos);
        }
        _fail(startPos);
        return _PARSE_FAILED;
    }

    SyntaxTree* Parser::_parse_anyChar(void)
    {
        _ErrorPos startPos = _curErrPos();
        if (_inp->isEOF()) {
            _fail(startPos);
            return _PARSE_FAILED;
        }
        try {
            _inp->getChar();
        } catch (u32 pos) {
            _fail(startPos);
            return _PARSE_FAILED;
        }
        return createSyntaxTree(startPos.parsePos);
    }

    SyntaxTree* Parser::_parse_rangeChar(wchar_t c1, wchar_t c2)
    {
        _ErrorPos startPos = _curErrPos();
        wchar_t c;
        try {
            c = _inp->getChar();
        } catch (u32 pos) {
            _fail(startPos);
            return _PARSE_FAILED;
        }
        if ((c >= c1) && (c <= c2)) {
            return createSyntaxTree(startPos.parsePos);
        }
        _fail(startPos);
        return _PARSE_FAILED;
    }

    SyntaxTree* Parser::_parse_EOF(void)
    {
        _ErrorPos startPos = _curErrPos();
        if (_inp->isEOF()) {
            return createSyntaxTree(startPos.parsePos);
        }
        _fail(startPos);
        return _PARSE_FAILED;
    }


    SyntaxTree* Parser_ErrorCut::parse(void)
    {
        u32 cp = _curPos();
        Substr ss(cp, cp);
        return new SyntaxTree(this, ss, _errorCutId);
    }

    Parser_Sequence::Parser_Sequence(Parser* ps[], int n)
        : Parser()
    {
        _ps = new PtrArray<Parser>(n);
        for (int i = 0; i < n; i++) {
            ps[i]->addSequenceToParent(_ps);
        }
    }

    Parser_Sequence::Parser_Sequence(Parser* p1, Parser* p2)
        : Parser()
    {
        _ps = new PtrArray<Parser>(2);
        p1->addSequenceToParent(_ps);
        p2->addSequenceToParent(_ps);
    }

    void Parser_Sequence::addSequenceToParent(PtrArray<Parser>* parentSeq)
    {
        for (PtrArrayIterator<Parser> itr(_ps); itr.hasMore(); )
            parentSeq->add(itr.next());
    }

    Parser_OrderedChoice::Parser_OrderedChoice(Parser* ps[], int n)
        : Parser()
    {
        _ps = new PtrArray<Parser>(n);
        for (int i = 0; i < n; i++)
            _ps->add(ps[i]);
    }

    Parser_OrderedChoice::Parser_OrderedChoice(Parser* p1, Parser* p2)
        : Parser()
    {
        _ps = new PtrArray<Parser>(2);
        _ps->add(p1);
        _ps->add(p2);
    }

    SyntaxTree* UserParser::parse(void)
    {
        _ErrorPos startPos = _curErrPos();
        SyntaxTree* st = _parse1(startPos);
        if (st != NULL)
            return st;

        st = uParse();

        return _parse2(startPos, st);
    }

    SyntaxTree* UserParser::_parse1(_ErrorPos startPos)
    {
        const char* n = name();
        SyntaxTree* st;
        
        if (_bUseMemoize) {
            st = _memo.getAt(startPos.parsePos);
            if (st == _PARSING) {
                // left recursion
                char pbuf[128];
                _inp->posInfo(pbuf, 128, startPos.parsePos);
                sys_printf("LEFT RECURSION DETECTED: %s %s\n", n, pbuf);
                return _FATAL_PARSER_ERROR;
            } else if (st != _NOT_PARSED_YET) {
                if (st->isValidTree())
                    _inp->setPos(st->str.endPos);
                if (_printIntermediateLevel > 1 ||
                    (_printIntermediateLevel > 0 && !st->isFail())) {
                    char pbuf[128];
                    _inp->posInfo(pbuf, 128, startPos.parsePos);
                    sys_printf("%s %s -> memoized ", n, pbuf);
                    if (st->isValidTree()) {
                        char b[44];
                        _inp->copySummary(b, 40, st->str);
                        sys_printf("pos=%d-%d '%s'\n", st->str.startPos, st->str.endPos, b);
                    } else {
                        if (st == _NO_SYNTAX_TREE)
                            sys_printf("no syntax tree\n");
                        else if (st == _PARSE_FAILED)
                            sys_printf("parse failed\n");
                        else if (st->isErrorCut())
                            sys_printf("ErrorCut(%d)\n",st->errorCutId);
                        else
                            sys_printf("(UNKNOWN BUG?)\n");
                    }
                }
                return st;
            }

            // not parsed yet
            _memo.setAt(startPos.parsePos, _PARSING);
        }
        
        if (_printIntermediateLevel > 1) {
            char pbuf[128];
            _inp->posInfo(pbuf, 128, startPos.parsePos);
            sys_printf("try %s %s pos=%d:\n", n, pbuf, startPos.parsePos);
        }

        return NULL;
    }
    
    SyntaxTree* UserParser::_parse2(_ErrorPos startPos, SyntaxTree* st)
    {
        const char* n = name();

        if (st->isError() || st->isErrorCut()) {
            _fail(startPos);
            return _FATAL_PARSER_ERROR;
        }
        if (st->isFail()) {
            if (_printIntermediateLevel > 1) {
                char pbuf[128];
                _inp->posInfo(pbuf, 128, startPos.parsePos);
                sys_printf("%s %s -> fail\n", n, pbuf);
            }
            _fail(startPos);
            if (_bUseMemoize)
                _memo.setAt(startPos.parsePos, _PARSE_FAILED);
            return _PARSE_FAILED;
        }

        if (_printIntermediateLevel > 0) {
            char pbuf[128];
            _inp->posInfo(pbuf, 128, startPos.parsePos);
            if (st->isValidTree()) {
                char b[44];
                _inp->copySummary(b, 40, st->str);
                sys_printf("%s %s -> pos=%d-%d '%s'\n", n, pbuf, st->str.startPos, st->str.endPos, b);
            } else {
                sys_printf("%s %s -> (notree)\n", n, pbuf);
            }
        }

        if (_bUseMemoize)
            _memo.setAt(startPos.parsePos, st);

        return st;
    }

    void UserParser::removeMemo(SyntaxTree* st)
    {
        if (_memo.size() < 1)
            return;
        _memo.remove(st);
    }

    void UserParser::errorMessage(int errorCutId, u32 pos)
    {
        char pbuf[128];
        _inp->posInfo(pbuf, 128, pos);
        sys_printf("error %d in parser %s at %s\n", errorCutId, pbuf, name());
    }

    SyntaxTree* GenParser::uParse(void)
    {
        assert(_parser);
        return _parser->parse();
    }

    SyntaxTree* GenParser::_parse2(_ErrorPos startPos, SyntaxTree* st)
    {
        const char* n = name();

        if (st->isError() || st->isErrorCut()) {
            _fail(startPos);
            if (! st->isFatalError()) {
                errorMessage(st->errorCutId, st->str.startPos);
                //char pbuf[128];
                //_inp->posInfo(pbuf, 128, startPos.parsePos);
                //sys_printf("%s %s -> ERROR %d\n", n, pbuf, st->errorCutId);
            }
            return _FATAL_PARSER_ERROR;
        }
        if (st->isFail()) {
            if (_printIntermediateLevel > 1) {
                char pbuf[128];
                _inp->posInfo(pbuf, 128, startPos.parsePos);
                sys_printf("%s %s -> fail\n", n, pbuf);
            }
            _fail(startPos);
            if (_bUseMemoize)
                _memo.setAt(startPos.parsePos, _PARSE_FAILED);
            return _PARSE_FAILED;
        }

        if (_printIntermediateLevel > 0) {
            char pbuf[128];
            _inp->posInfo(pbuf, 128, startPos.parsePos);
            if (st->isValidTree()) {
                char b[44];
                _inp->copySummary(b, 40, st->str);
                sys_printf("%s %s -> pos=%d-%d '%s'\n", n, pbuf, st->str.startPos, st->str.endPos, b);
            } else {
                sys_printf("%s %s -> (notree)\n", n, pbuf);
            }
        }
        PtrArray<SyntaxTree>* childs = new PtrArray<SyntaxTree>(1);
        _parser->addSyntaxTreeToParent(childs, st);
        SyntaxTree* myTree = createSyntaxTree(startPos.parsePos, childs);
        if (_bUseMemoize)
            _memo.setAt(startPos.parsePos, myTree);
        actionAtParse(myTree);
        return myTree;
    }


    SyntaxTree* OperatorParser::parse(void)
    {
        assert(_parser);
        _ErrorPos startPos = _curErrPos();
        SyntaxTree* st = _parser->parse();
        const char* n = name();
        if (st->isError()) {
            _fail(startPos);
            if (! st->isFatalError()) {
                char pbuf[128];
                _inp->posInfo(pbuf, 128, startPos.parsePos);
                sys_printf("%s %s -> ERROR %d\n", n, pbuf, st->errorCutId);
            }
            return _FATAL_PARSER_ERROR;
        }
        if (st->isFail()) {
            if (_printIntermediateLevel > 2) {
                char pbuf[128];
                _inp->posInfo(pbuf, 128, startPos.parsePos);
                sys_printf("%s %s -> fail\n", n, pbuf);
            }
            _fail(startPos);
            return _PARSE_FAILED;
        }

        if (_printIntermediateLevel > 1) {
            char pbuf[128];
            _inp->posInfo(pbuf, 128, startPos.parsePos);
            if (st->isValidTree()) {
                char b[44];
                _inp->copySummary(b, 40, st->str);
                sys_printf("%s %s -> '%s'\n", n, pbuf, b);
            } else {
                sys_printf("%s %s -> (notree)\n", n, pbuf);
            }
        }
        
        if (st->isValidTree()) {
            if (st->numChild() == 1) {
                // no operator, so that this node is not needed
                SyntaxTree* tmp = st->extract(0);
                if (! _bUseMemoize)
                    delete st;
                return tmp;
            }
        }

        PtrArray<SyntaxTree>* childs = new PtrArray<SyntaxTree>(1);
        _parser->addSyntaxTreeToParent(childs, st);
        return createSyntaxTree(startPos.parsePos, childs);
    }
}
