﻿#!/usr/bin/env pytnon
# -*- coding: utf-8 -*-

import ctypes
import codecs
import datetime
import os
import re
import time

from svn import fs, repos, core, client
from trac.config import Option
from trac.core import *
from trac.ticket import ITicketChangeListener
from trac.util.datefmt import utc, to_timestamp
from trac.util.text import CRLF


class Sap2SvnError(TracError):

    def __init__(self, cls, method, msg=None):
        self.cls = cls
        self.method = method
        self.msg = msg


class Sap2SvnPlugin(Component):
    """RESTRICTION:
        - No user like '__SAP2SVN__'. (Because of using internal)
        - No process updating commit files

    """
    implements(ITicketChangeListener)

    # Display on admin console
    Option('sap2svn', 'applicationserver', None,
           'SAP application server for logon')
    Option('sap2svn', 'client', None,
           'SAP client for logon')
    Option('sap2svn', 'codepage', '8000',
           'SAP codepage for logon (default "8000: Shift_JIS")')
    Option('sap2svn', 'language', 'J',
           'SAP language for logon (default "J: Japanese")')
    Option('sap2svn', 'password', None,
           'SAP password for logon')
    Option('sap2svn', 'saprouter', None,
           'SAP router for logon')
    Option('sap2svn', 'system', None,
           'SAP system for logon')
    Option('sap2svn', 'systemnumber', None,
           'SAP system number for logon')
    Option('sap2svn', 'user', None,
           'SAP user for logon')
    Option('sap2svn', 'workingcopy', None,
           'Directory path putting SAP source')

    def _check_environment(self):
        conf = [
                    ('ticket-custom', 'released'),
                    ('ticket-custom', 'transport'),
                    ('sap2svn',       'applicationserver'),
                    ('sap2svn',       'client'),
#                    ('sap2svn',       'codepage'),     # allowed blank
#                    ('sap2svn',       'language'),     # allowed blank
                    ('sap2svn',       'password'),
#                    ('sap2svn',       'saprouter'),    # allowed blank
#                    ('sap2svn',       'system'),       # allowed blank
#                    ('sap2svn',       'systemnumber'), # allowed blank
                    ('sap2svn',       'user'),
                    ('sap2svn',       'workingcopy'),
                ]

        is_error = False

        for section, name in conf:
            if not self.env.config.get(section, name):
                self.log.warn("No setting in trac.ini : '[%s] %s'" %
                              (section, name))
                is_error = True

        if is_error:
            raise Sap2SvnError('Sap2SvnPlugin', '_check_environment')

    def _sleep_for_duplicated(self, ticket):
        """Avoid inserting duplicated (same time) record."""
        now = datetime.datetime.now(utc)
        if to_timestamp(now) == to_timestamp(ticket.time_changed):
            time.sleep(1)

    ### methods for ITicketChangeListener

    def ticket_created(self, ticket):
        pass

    def ticket_changed(self, ticket, comment, author, old_values):
        try:
            self._check_environment()

            # Escape infinite loop
            if author == '__SAP2SVN__': return

            if ticket.values.get('released') == '1' and \
               old_values.get('released') == '0' and \
               ticket.values.get('transport') <> '':

                usr = SapClient(self).get_sources(ticket.values['transport'])
                summary = '%s (#%d)' % (ticket.values['summary'], ticket.id)
                revision = \
                    SvnClient(self).add_and_commit(usr,
                                                   summary.encode('utf-8'))
                if revision < 0: return  # No commit file

                self._sleep_for_duplicated(ticket)
                ticket.save_changes('__SAP2SVN__',
                                    'Reference changeset: [%d]' % revision)

        except Sap2SvnError, e:
            if e.msg:
                self.log.warn(e.msg)
            self.log.warn("Stop SAP2SVN plugin 'class : %s, method : %s'"
                            % (e.cls, e.method))

    def ticket_deleted(self, ticket):
        pass


class SapClient(object):

    def __init__(self, component):
        self._component = component

    def _get_pythoncompath(self):
        pattern = re.compile('pythoncom\d{2}\.dll')

        for tracpath in os.environ.get('TRACPATH').split(';'):
            for dpath, dnames, fnames in os.walk(tracpath):
                for fname in fnames:
                    if pattern.search(fname):
                        return os.path.join(dpath, fname)

    def _logon(self):
        try:
            import pythoncom
            import pywintypes
            import win32com.client
        except ImportError:
            try:
                ctypes.windll.LoadLibrary(self._get_pythoncompath())
                import pythoncom
                import pywintypes
                import win32com.client
            except (ImportError, WindowsError):
                raise Sap2SvnError('SapClient', '_logon',
                                   "Not found DLL : 'pythoncomXX.dll'")

        try:
            pythoncom.CoInitialize()
            self._r3 = win32com.client.Dispatch('SAP.Functions')
        except pywintypes.com_error:
            raise Sap2SvnError('SapClient', '_logon',
                               "Not found COM Object : 'SAP.Functions'")

        self._r3.Connection.applicationserver = \
            self._component.env.config.get('sap2svn', 'applicationserver')
        self._r3.Connection.client = \
            self._component.env.config.get('sap2svn', 'client')
        self._r3.Connection.codepage = \
            self._component.env.config.get('sap2svn', 'codepage')
        self._r3.Connection.language = \
            self._component.env.config.get('sap2svn', 'language')
        self._r3.Connection.password = \
            self._component.env.config.get('sap2svn', 'password')
        self._r3.Connection.saprouter = \
            self._component.env.config.get('sap2svn', 'saprouter')
        self._r3.Connection.system = \
            self._component.env.config.get('sap2svn', 'system')
        self._r3.Connection.systemNumber = \
            self._component.env.config.get('sap2svn', 'systemnumber')
        self._r3.Connection.user = \
            self._component.env.config.get('sap2svn', 'user')

        if self._r3.Connection.Logon(0, True):
            self._component.log.debug('SAP Logon')
        else:
            raise Sap2SvnError('SapClient', '_logon',
                               "Can't logon SAP Application Server")

    def _get_objects(self, transport):
        rfcname = 'TR_OBJECTS_OF_REQ_AN_TASKS_RFC'

        rfc = self._r3.Add(rfcname)
        trkorr = rfc.Exports('IV_TRKORR')
        trkorr.value = transport

        rfc.Call
        if   rfc.Exception == 'INVALID_INPUT':
            raise Sap2SvnError('SapClient', '_get_objects',
                               "Request/task '%s' does not exist" % transport)
        elif rfc.Exception:
            raise Sap2SvnError('SapClient', '_get_objects',
                               "Can't call RFC : %s" % rfcname)


        self._objects = rfc.Tables('ET_OBJECTS')

    def _get_objectsource(self):
        rfcname = 'RPY_PROGRAM_READ'

        for object in self._objects.rows:
            if object('PGMID') == 'R3TR' and object('OBJECT') == 'PROG' or \
               object('PGMID') == 'LIMU' and object('OBJECT') == 'REPS':
                rfc = self._r3.Add(rfcname)
                progname = rfc.Exports('PROGRAM_NAME')
                progname.value = object('OBJ_NAME')

                rfc.Call
                if   rfc.Exception == 'CANCELLED':
                    raise Sap2SvnError('SapClient', '_get_objects',
                               "Processing cancelled by user")
                elif rfc.Exception == 'NOT_FOUND':
                    raise Sap2SvnError('SapClient', '_get_objects',
                               "Object '%s' not found" % object('OBJ_NAME'))
                elif rfc.Exception == 'PERMISSION_ERROR':
                    raise Sap2SvnError('SapClient', '_get_objects',
                               "There is no permissions for this operation")
                elif rfc.Exception:
                    raise Sap2SvnError('SapClient', '_get_objectsource',
                                       "Can't call RFC : %s" % rfcname)

                wc = self._component.env.config.get('sap2svn', 'workingcopy')
                dname = os.path.join(wc, 'SAP', 'Programs', object('OBJ_NAME'))
                fname = os.path.join(wc, dname, object('OBJ_NAME'))
                if not os.access(dname, os.F_OK):
                  os.makedirs(dname)

                f = codecs.open(fname, 'w', 'mbcs')
                for source in rfc.Tables('SOURCE_EXTENDED').rows:
                  f.write(source('LINE') + CRLF)
                f.close()

    def _logoff(self):
        self._r3.Connection.Logoff()
        self._r3 = None
        self._component.log.debug('SAP Logoff')

    def get_sources(self, transport):
        self._logon()
        self._get_objects(transport)
        self._get_objectsource()
        self._logoff()

        return '__SAP2SVN__'


class SvnClient(object):

    def __init__(self, component):
        self._component = component

    def _get_sappath(self):
        wc = self._component.env.config.get('sap2svn', 'workingcopy')
        return (os.path.join(wc, 'SAP')).encode('mbcs')

    def _add(self):
        force = True
        no_ignore = True
        add_parents = True
        client.svn_client_add4(self._get_sappath(),
                               core.svn_depth_infinity, force,
                               no_ignore, add_parents,
                               client.svn_client_create_context())

    def _commit(self, usr, summary):
        ctx = client.svn_client_create_context()

        # Set commit log
        def __log_message_func3(items, pool):
            return summary
        ctx.log_msg_func3 = client.svn_swig_py_get_commit_log_func
        ctx.log_msg_baton3 = __log_message_func3
        ctx.log_msg_func3(None, ctx.log_msg_baton3)

        # Set commit user
        def __prompt_func(realm, maysave, pool):
            username_cred = core.svn_auth_cred_username_t()
            username_cred.username = usr
            username_cred.may_save = False
            return username_cred
        retry_limit = 1
        providers = [
            core.svn_auth_get_username_prompt_provider(__prompt_func,
                                                       retry_limit),
        ]
        ctx.auth_baton = core.svn_auth_open(providers)

        # Commit
        keep_locks = False
        keep_changelists = False
        changelists = []
        revprop_table = {}
        return client.svn_client_commit4([self._get_sappath()],
                                         core.svn_depth_infinity, keep_locks,
                                         keep_changelists, changelists,
                                         revprop_table, ctx)

    def add_and_commit(self, usr, summary):
        self._add()
        commit_info = self._commit(usr, summary)

        return commit_info.revision
