#/*
# *  Copyright 2007-2010 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: Metadata.pm 1986 2010-01-31 15:23:41Z hikarin $
#

package Img0ch::Metadata;

use strict;

BEGIN {
    my $pkg     = __PACKAGE__;
    my @methods = (
        [ 'head',          'head.txt', 0, 'head' ],
        [ 'meta',          'meta.txt', 0, 'meta' ],
        [ 'foot',          'foot.txt', 0, 'foot' ],
        [ 'main_banner',   undef,      1, 'banner.main' ],
        [ 'sub_banner',    undef,      1, 'banner.sub' ],
        [ 'mobile_banner', undef,      1, 'banner.mobile' ],
        [ 'thread_end',    '1000.txt', 0, 'thread' ],
    );

    no strict 'refs';
    *{"${pkg}::method_info"} = sub {@methods};

    for my $method (@methods) {
        my ( $name, $file, $is_global, $plugin_type ) = @$method;
        my $utf8_name = $name . '_utf8';
        *{"${pkg}::${name}"} = sub { shift->_load( $name, $is_global, @_ ) };
        *{"${pkg}::${utf8_name}"} = sub {
            my ( $iMeta, @args ) = @_;
            return $iMeta->{_kernel}
                ->get_encoded_str( $iMeta->$name(@args), 'utf8',
                $iMeta->{_encoding} );
        };
    }
}

sub new {
    my ( $iClass, $iBBS ) = @_;
    my $iKernel      = $iBBS->get_kernel();
    my $iReposCommon = $iBBS->get_common_repos();
    my $iReposSystem = $iBBS->get_system_repos();

    if ( $iKernel->get_config()->get('compatible.metadata') ) {
        require Img0ch::Compat::Metadata;
        Img0ch::Compat::Metadata::load( $iReposSystem, $iBBS );
    }

    bless {
        _bbs      => $iBBS->get_name(),
        _common   => $iReposCommon,
        _encoding => $iBBS->get_encoding(),
        _kernel   => $iKernel,
        _system   => $iReposSystem,
        _utf8     => $iBBS->is_utf8(),
    }, $iClass;
}

sub random_data {
    my ( $iClass, $size ) = @_;
    if ( defined $Crypt::Random::VERSION
        or eval 'require Crypt::Random' )
    {
        return Crypt::Random::makerandom( Size => $size, Strength => 1 );
    }
    else {
        defined $Bytes::Random::VERSION or require Bytes::Random;
        return Bytes::Random::random_bytes($size);
    }
}

sub random_data_ymd {
    my ( $iMeta, $now ) = @_;
    my @time = localtime($now);
    my $ymd
        = sprintf( '%04d%02d%02d', $time[5] + 1900, $time[4] + 1, $time[3] );
    my $iReposCommon = $iMeta->{_common};
    my $start        = $iReposCommon->get('I:M.random.ymd');
    my $data;
    if ( $start ne $ymd ) {
        $data = __PACKAGE__->random_data(4096);
        $iReposCommon->set( 'I:M.random.ymd', $ymd );
        $iReposCommon->set_binary( 'I:M.random.data', $data );
        $iReposCommon->save();
    }
    else {
        $data = $iReposCommon->get_binary('I:M.random.data');
    }
    return $data;
}

sub _load {
    my ( $iMeta, $type, $is_global ) = @_;
    my $iKernel = $iMeta->{_kernel};
    if ($is_global) {
        my $iReposCommon = $iMeta->{_common};
        my $utf8         = $iReposCommon->get("I:M.${type}.utf8");

        # >= 2.1.27 and board uses Shift_JIS
        if ( !$iMeta->{_utf8} and $utf8 ) {
            return $iKernel->get_encoded_str(
                $iReposCommon->get_binary("I:M.${type}")
                    || $iReposCommon->get_binary("I:M.test.${type}"),
                $iMeta->{_encoding}, 'utf8',
            );
        }

        # < 2.1.27
        elsif ( !$utf8 ) {
            return $iKernel->get_encoded_str(
                $iReposCommon->get_binary("I:M.${type}")
                    || $iReposCommon->get_binary("I:M.test.${type}"),
                $iMeta->{_encoding}, 'sjis',
            );
        }

        # >= 2.1.27 and board uses UTF-8
        else {
            return $iReposCommon->get_binary("I:M.${type}")
                || $iReposCommon->get_binary("I:M.test.${type}");
        }
    }
    else {
        my $iReposSystem = $iMeta->{_system};
        my $bbs          = $iMeta->{_bbs};
        my $utf8         = $iReposSystem->get("I:M.${bbs}.${type}.utf8");

        # >= 2.1.27 and board uses Shift_JIS
        if ( !$iMeta->{_utf8} and $utf8 ) {
            return $iKernel->get_encoded_str(
                $iReposSystem->get_binary("I:M.${bbs}.${type}"),
                $iMeta->{_encoding}, 'utf8', );
        }

        # < 2.1.27
        elsif ( !$utf8 ) {
            return $iKernel->get_encoded_str(
                $iReposSystem->get_binary("I:M.${bbs}.${type}")
                    || $iMeta->{_common}->get_binary("I:M.${bbs}.${type}"),
                $iMeta->{_encoding}, 'sjis',
            );
        }

        # >= 2.1.27 and board uses UTF-8
        else {
            return $iReposSystem->get_binary("I:M.${bbs}.${type}");
        }
    }
}

sub _load_internal {
}

1;
__END__

=head1 NAME

Img0ch::Config - メタデータを格納するクラス

=head1 SYNOPSYS

  use Img0ch::Metadata

  my $iMeta = Img0ch::Metadata->new($iBBS);

  # head.txtの中身を取り出す
  my $head = $iMeta->head();

  # meta.txtの中身を取り出す
  my $meta = $iMeta->meta();

  # foot.txtの中身を取り出す
  my $foot = $iMeta->foot();

  # 告知欄の中身を取り出す
  my $main_banner = $iMeta->main_banner();

  # 副告知欄の中身を取り出す
  my $sub_banner = $iMeta->sub_banner();

  # 携帯用告知欄の中身を取り出す
  my $mobile_banner = $iMeta->mobile_banner();

  # 1000.txtの中身を表示する
  my $thread_end = $iMeta->thread_end();

=head1 DESCRIPTION

1つのメタデータを1つのオブジェクトとするクラスです。

=head2 new

=over 4

=item Arguments

$iBBS (Img0ch::BBS)

=item Return Value

$iMeta (Img0ch::Metadata itself)

=back

I<Img0ch::Metadata>のオブジェクトを作成します。

=head2 head

=over 4

=item Arguments

none

=item Return Value

$content

=back

head.txtの内容を返します。

=head2 meta

=over 4

=item Arguments

none

=item Return Value

$content

=back

meta.txtの内容を返します。

=head2 foot

=over 4

=item Arguments

none

=item Return Value

$content

=back

foot.txtの内容を返します。

=head2 main_banner

=over 4

=item Arguments

none

=item Return Value

$content

=back

告知欄の内容を返します。

=head2 sub_banner

=over 4

=item Arguments

none

=item Return Value

$content

=back

副告知欄の内容を返します。

=head2 mobile_banner

=over 4

=item Arguments

none

=item Return Value

$content

=back

携帯用告知欄の内容を返します。

=head2 thread_end

=over 4

=item Arguments

none

=item Return Value

$content

=back

1000.txtの内容を返します。

=head1 AUTHOR

hkrn E<lt>hikarin@users.sourceforge.jpE<gt>

=cut
