/*
 * swdrv.c -- driver for the Power control switch.
 *
 * This driver will also support the I-O DATA Device, Inc. Julian Board.
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 *
 * Copylight (C) 2002 Atom Create Engineering Co., Ltd.
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/major.h>
#include <linux/poll.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/timer.h>

#include <asm/system.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/iodata_julian.h>

#define SHUTDOWN_BTN_MINOR	1	/* Shutdown button device minor no. */

static int openCnt;
static int shutdown_sw_flag;
static int shutdown_complete_flag;

/*
 * Functions prototypes
 */
static void shutdown_check(void);
static int swdrv_open(struct inode *, struct file *);
static int swdrv_close(struct inode *, struct file *);
static int swdrv_read(struct file *, char *, size_t, loff_t *);
static void sw_interrupt(int, void *, struct pt_regs *);

static void shutdown_check(void)
{
	int i;

	if (shutdown_sw_flag) {
		for (i=0; i<SHUTDOWN_LOOP_CNT; i++)
			if (!(ctrl_inb(PA_STATUS) & 0x10))
				mdelay(SHUTDOWN_DELAY);
			else
				break;
		if (i == SHUTDOWN_LOOP_CNT)
			shutdown_complete_flag = 1;
		else
			;
		shutdown_sw_flag = 0;
	} else
		;
}

static int swdrv_open(struct inode *inode, struct file *filp)
{
	int minor;

	minor = MINOR(inode->i_rdev);
	if (minor != SHUTDOWN_BTN_MINOR)
		return -ENOENT;

	if (openCnt > 0)
		return -EALREADY;

	openCnt++;

	return 0;
}

static int swdrv_close(struct inode *inode, struct file *filp)
{

	openCnt--;

	return 0;
}

static int swdrv_read(struct file *filp, char *buff, size_t count, loff_t* ppos)
{
	int error ;

	if ((error = verify_area(VERIFY_WRITE, (void *)buff, count)))
		return error;

	if (shutdown_complete_flag) {
		put_user(1, buff);
		return 1;
	} else
		return 0;
}

static void sw_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	if (!shutdown_sw_flag) {
		disable_irq(irq);
		/* 2003.10.31 I-O DATA NSD NWG	add.	for shutdown port clear ----> */
		ctrl_outb(0x00, PA_PWRINT_CLR);
		/* 2003.10.31 I-O DATA NSD NWG	add.	for shutdown port clear	<---- */
		shutdown_sw_flag = 1;
		shutdown_check();
		if (!shutdown_complete_flag)
			enable_irq(irq);
	}
}

static struct file_operations swdrv_fops = {
	read:		swdrv_read,	/* read */
	open:		swdrv_open,	/* open */
	release:	swdrv_close,	/* release */
};

static char banner[] __initdata =
	KERN_INFO "Julian Shutdown button driver initialized\n";

int __init swdrv_init(void)
{
	int error;

	printk("%s", banner);

	shutdown_sw_flag = shutdown_complete_flag = 0;
	openCnt = 0;

	if ((error=register_chrdev(SHUTDOWN_BTN_MAJOR, "swdrv", &swdrv_fops))) {
		printk(KERN_ERR "Shutdown button driver:Couldn't register driver, error=%d\n", error);
		return 1;
	}

	if (request_irq(IRQ_POWER, sw_interrupt, 0, "SHUTDOWNSWITCH", NULL)) {
		printk(KERN_ERR "Unable to get IRQ 10.\n");
		return 1;
	}
	return 0;
}

module_init(swdrv_init);
