/*
 * linux/arch/sh/kernel/rtc.c -- SH3 / SH4 on-chip RTC support
 *
 *  Copyright (C) 2000  Philipp Rumpf <prumpf@tux.org>
 *  Copyright (C) 1999  Tetsuya Okada & Niibe Yutaka
 */

#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/time.h>
/* 2003.09.25 I-O DATA NSD NWG	SH RTC -> RICOH RS5C313	----> */
#include <linux/delay.h>
/* 2003.09.25 I-O DATA NSD NWG	SH RTC -> RICOH RS5C313	<---- */

#include <asm/io.h>
#include <asm/rtc.h>

#ifndef BCD_TO_BIN
#define BCD_TO_BIN(val) ((val)=((val)&15) + ((val)>>4)*10)
#endif

#ifndef BIN_TO_BCD
#define BIN_TO_BCD(val) ((val)=(((val)/10)<<4) + (val)%10)
#endif

/* 2003.09.25 I-O DATA NSD NWG	SH RTC -> RICOH RS5C313	----> */
#define SCSMR1	0xFFE00000
#define SCSCR1	0xFFE00008

#define SCSMR1_CA	0x80

#define SCSCR1_CKE	0x03


#define SCSPTR1	0xFFE0001C

#define SCSPTR1_EIO	0x80
#define SCSPTR1_SPB1IO	0x08
#define SCSPTR1_SPB1DT	0x04
#define SCSPTR1_SPB0IO	0x02
#define SCSPTR1_SPB0DT	0x01

#define SDA_OEN	SCSPTR1_SPB1IO
#define SDA	SCSPTR1_SPB1DT
#define SCL_OEN	SCSPTR1_SPB0IO
#define SCL	SCSPTR1_SPB0DT

/* RICOH RS5C313 CE port */
#define RS5C313_CE		0xB0000003

/* RICOH RS5C313 CE port bit */
#define RS5C313_CE_RTCCE	0x02

/* RICOH RS5C313 address */
#define RS5C313_ADDR_SEC	0x00
#define RS5C313_ADDR_SEC10	0x01
#define RS5C313_ADDR_MIN	0x02
#define RS5C313_ADDR_MIN10	0x03
#define RS5C313_ADDR_HOUR	0x04
#define RS5C313_ADDR_HOUR10	0x05
#define RS5C313_ADDR_WEEK	0x06
#define RS5C313_ADDR_INTINTVREG	0x07
#define RS5C313_ADDR_DAY	0x08
#define RS5C313_ADDR_DAY10	0x09
#define RS5C313_ADDR_MON	0x0A
#define RS5C313_ADDR_MON10	0x0B
#define RS5C313_ADDR_YEAR	0x0C
#define RS5C313_ADDR_YEAR10	0x0D
#define RS5C313_ADDR_CNTREG	0x0E
#define RS5C313_ADDR_TESTREG	0x0F

/* RICOH RS5C313 control register */
#define RS5C313_CNTREG_ADJ_BSY	0x01
#define RS5C313_CNTREG_WTEN_XSTP	0x02
#define RS5C313_CNTREG_12_24	0x04
#define RS5C313_CNTREG_CTFG	0x08

/* RICOH RS5C313 test register */
#define RS5C313_TESTREG_TEST	0x01

/* RICOH RS5C313 control bit */
#define RS5C313_CNTBIT_READ	0x40
#define RS5C313_CNTBIT_AD	0x20
#define RS5C313_CNTBIT_DT	0x10


/* SCSPTR1 data */
unsigned char scsptr1_data;


static void rs5c313_initialize(void)
{
  scsptr1_data = ctrl_inb(SCSPTR1) | SCL;	/* SCL:H */
  ctrl_outb(scsptr1_data, SCSPTR1);
  scsptr1_data = ctrl_inb(SCSPTR1) | SCL_OEN;	/* SCL output enable */
  ctrl_outb(scsptr1_data, SCSPTR1);
  ctrl_outb(0x00, RS5C313_CE);			/* CE:L */
}

static void rs5c313_write(unsigned char data)
{
  int i;

  for (i = 0; i < 8; i++){
    /* write data */
    scsptr1_data = (scsptr1_data & ~SDA) | ((((0x80 >> i ) & data) >> (7 - i)) << 2);	/* SDA:Write Data */
    ctrl_outb(scsptr1_data, SCSPTR1);
    if (i == 0){
      scsptr1_data |= SDA_OEN;		/* SDA:output enable */
      ctrl_outb(scsptr1_data, SCSPTR1);
    }
    ndelay(700);
    scsptr1_data &= ~SCL;		/* SCL:L */
    ctrl_outb(scsptr1_data, SCSPTR1);
    ndelay(700);
    scsptr1_data |= SCL;		/* SCL:H */
    ctrl_outb(scsptr1_data, SCSPTR1);
  }
  
  scsptr1_data &= ~SDA_OEN;		/* SDA:output disable */
  ctrl_outb(scsptr1_data, SCSPTR1);

}

static void rs5c313_write_addr(unsigned char addr, unsigned char read)
{
  unsigned char data;

  data = addr | read | RS5C313_CNTBIT_AD;
  rs5c313_write(data);

}

static void rs5c313_write_data(unsigned char data)
{

  /* make write data */
  data |= RS5C313_CNTBIT_DT;
  rs5c313_write(data);

}

static unsigned char rs5c313_read_data(void)
{
  int i;
  unsigned char data = 0;

  for (i = 0; i < 8; i++){
    ndelay(700);
    data |= ((ctrl_inb(SCSPTR1) & SDA) >> 2) << (7 - i);	/* SDA:Read Data */
    scsptr1_data &= ~SCL;		/* SCL:L */
    ctrl_outb(scsptr1_data, SCSPTR1);
    ndelay(700);
    scsptr1_data |= SCL;		/* SCL:H */
    ctrl_outb(scsptr1_data, SCSPTR1);
  }

  return data & 0x0F;

}


static unsigned char rs5c313_read_cntreg(void)
{

  rs5c313_write_addr(RS5C313_ADDR_CNTREG, RS5C313_CNTBIT_READ);
  return rs5c313_read_data();

}

static void rs5c313_write_cntreg(unsigned char data)
{

  rs5c313_write_addr(RS5C313_ADDR_CNTREG, 0);
  rs5c313_write_data(data);
  return;

}

static void rs5c313_write_intintvreg(unsigned char data)
{

  rs5c313_write_addr(RS5C313_ADDR_INTINTVREG, 0);
  rs5c313_write_data(data);
  return;

}

static void rs5c313_get_cur_time(unsigned char *sec, unsigned char *min, unsigned char *hr/* , unsigned char *wk */, unsigned char *day, unsigned char *mon, unsigned char *yr)
{

#if 1
  while (1){
    ctrl_outb(RS5C313_CE_RTCCE, RS5C313_CE);	/* CE:H */
    rs5c313_write_cntreg(0x04);		/* Initialize control reg. 24 hour */

    if (!(rs5c313_read_cntreg() & RS5C313_CNTREG_ADJ_BSY))
      break;
    ctrl_outb(0x02, 0xB0000008);
    ctrl_outb(0x00, RS5C313_CE); ndelay(700);	/* CE:L */
  }
#else
  /* bysy check. */
  while (rs5c313_read_cntreg() & RS5C313_CNTREG_ADJ_BSY);
#endif

  rs5c313_write_addr(RS5C313_ADDR_SEC, RS5C313_CNTBIT_READ);
  *sec = rs5c313_read_data();
  rs5c313_write_addr(RS5C313_ADDR_SEC10, RS5C313_CNTBIT_READ);
  *sec |= ((rs5c313_read_data() & 0x07) << 4);
  rs5c313_write_addr(RS5C313_ADDR_MIN, RS5C313_CNTBIT_READ);
  *min = rs5c313_read_data();
  rs5c313_write_addr(RS5C313_ADDR_MIN10, RS5C313_CNTBIT_READ);
  *min  |= ((rs5c313_read_data() & 0x07) << 4);
  rs5c313_write_addr(RS5C313_ADDR_HOUR, RS5C313_CNTBIT_READ);
  *hr = rs5c313_read_data();
  rs5c313_write_addr(RS5C313_ADDR_HOUR10, RS5C313_CNTBIT_READ);
  *hr |= ((rs5c313_read_data() & 0x03) << 4);
/*   rs5c313_write_addr(RS5C313_ADDR_WEEK, RS5C313_CNTBIT_READ); */
/*   *wk = rs5c313_read_data() & 0x07; */
 
  rs5c313_write_addr(RS5C313_ADDR_DAY, RS5C313_CNTBIT_READ);
  *day = rs5c313_read_data();
  rs5c313_write_addr(RS5C313_ADDR_DAY10, RS5C313_CNTBIT_READ);
  *day |= ((rs5c313_read_data() & 0x03) << 4);
  rs5c313_write_addr(RS5C313_ADDR_MON, RS5C313_CNTBIT_READ);
  *mon = rs5c313_read_data();
  rs5c313_write_addr(RS5C313_ADDR_MON10, RS5C313_CNTBIT_READ);
  *mon |= ((rs5c313_read_data() & 0x01) << 4);
  rs5c313_write_addr(RS5C313_ADDR_YEAR, RS5C313_CNTBIT_READ);
  *yr = rs5c313_read_data();
  rs5c313_write_addr(RS5C313_ADDR_YEAR10, RS5C313_CNTBIT_READ);
  *yr |= (rs5c313_read_data() << 4);

  ctrl_outb(0x00, RS5C313_CE); ndelay(700);	/* CE:L */

}

static void rs5c313_set_cur_time(unsigned char sec, unsigned char min, unsigned char hr/* , unsigned char wk */, unsigned char day, unsigned char mon, unsigned char yr)
{

  /* bysy check. */
#if 1
  while (1){
    ctrl_outb(RS5C313_CE_RTCCE, RS5C313_CE);	/* CE:H */
    rs5c313_write_cntreg(0x04);		/* Initialize control reg. 24 hour */

    if (!(rs5c313_read_cntreg() & RS5C313_CNTREG_ADJ_BSY))
      break;
    ctrl_outb(0x02, 0xB0000008);
    ctrl_outb(0x00, RS5C313_CE); ndelay(700);	/* CE:L */
  }
#else
  while (rs5c313_read_cntreg() & RS5C313_CNTREG_ADJ_BSY);
#endif

  rs5c313_write_addr(RS5C313_ADDR_SEC, 0);
  rs5c313_write_data(sec & 0x0F);
  rs5c313_write_addr(RS5C313_ADDR_SEC10, 0);
  rs5c313_write_data((sec >> 4) & 0x07);

  rs5c313_write_addr(RS5C313_ADDR_MIN, 0);
  rs5c313_write_data(min & 0x0F);
  rs5c313_write_addr(RS5C313_ADDR_MIN10, 0);
  rs5c313_write_data((min >> 4) & 0x07);
  rs5c313_write_addr(RS5C313_ADDR_HOUR, 0);
  rs5c313_write_data(hr & 0x0F);
  rs5c313_write_addr(RS5C313_ADDR_HOUR10, 0);
  rs5c313_write_data((hr >> 4 ) & 0x03);
/*   rs5c313_write_addr(RS5C313_ADDR_WEEK, 0); */
/*   rs5c313_write_data(wk  & 0x07); */
 
  rs5c313_write_addr(RS5C313_ADDR_DAY, 0);
  rs5c313_write_data(day & 0x0F);
  rs5c313_write_addr(RS5C313_ADDR_DAY10, 0);
  rs5c313_write_data((day >> 4) & 0x03);
  rs5c313_write_addr(RS5C313_ADDR_MON, 0);
  rs5c313_write_data(mon & 0x0F);
  rs5c313_write_addr(RS5C313_ADDR_MON10, 0);
  rs5c313_write_data((mon >> 4) & 0x01);
  rs5c313_write_addr(RS5C313_ADDR_YEAR, 0);
  rs5c313_write_data(yr & 0x0F);
  rs5c313_write_addr(RS5C313_ADDR_YEAR10, 0);
  rs5c313_write_data((yr >> 4) & 0x0F);

  ctrl_outb(0x00, RS5C313_CE); ndelay(700);	/* CE:H */

}

/* 2003.09.25 I-O DATA NSD NWG	SH RTC -> RICOH RS5C313	<---- */

/* 2003.09.25 I-O DATA NSD NWG	SH RTC -> RICOH RS5C313	----> */
#if defined(CONFIG_SH_JULIAN)
void sh_rtc_gettimeofday(struct timeval *tv)
{
  unsigned int sec128, sec, min, hr,/*  wk, */ day, mon, yr, yr100;
  int clkstop = 0;

  /* Set SCK as I/O port and Initialize SCSPTR1 data & I/O port. */
  ctrl_outb(ctrl_inb(SCSMR1) & ~SCSMR1_CA, SCSMR1);
  ctrl_outb(ctrl_inb(SCSCR1) & ~SCSCR1_CKE, SCSCR1);

  /* Initialize SCL for RS5C313 clock */
  rs5c313_initialize();

 again:
  /* check XSTP bit for clock stoped */
  ctrl_outb(RS5C313_CE_RTCCE, RS5C313_CE);	/* CE:H */
  if (rs5c313_read_cntreg() & RS5C313_CNTREG_WTEN_XSTP){
    /* INT interval reg. OFF */
    rs5c313_write_intintvreg(0x00);
    /* Initialize control reg. 24 hour & adjust */
    rs5c313_write_cntreg(0x07);
    /* bysy check. */
    while (rs5c313_read_cntreg() & RS5C313_CNTREG_ADJ_BSY)
      ctrl_outb(0x02, 0xB0000008);
    /* Initialize control reg. 24 hour */
    rs5c313_write_cntreg(0x04);
    clkstop = 1;
/*     ctrl_outb(0x01, 0xB0000008); */
  } else {
    clkstop = 0;
/*     ctrl_outb(0x08, 0xB0000008); */
  }
  ctrl_outb(0x00, RS5C313_CE); ndelay(700);	/* CE:L */


  /* Get current time. */
  sec = 0; min = 0; hr = 0;/*  wk = 0; */ day = 0; mon = 0; yr = 0;
  rs5c313_get_cur_time((unsigned char *)&sec,
		       (unsigned char *)&min,
		       (unsigned char *)&hr,
/* 		       (unsigned char *)&wk, */
		       (unsigned char *)&day,
		       (unsigned char *)&mon,
		       (unsigned char *)&yr);

  /* S-3531A count year from 2000 to 2099. */
  yr100 = 0x20;
  /* S-3531A can't get sec128. */
  sec128 = 0;

#if RTC_BIT_INVERTED != 0
  /* Work around to avoid reading incorrect value. */
  if (sec128 == RTC_BIT_INVERTED) {
    udelay(5);
    goto again;
  }
#endif

  BCD_TO_BIN(yr100);
  BCD_TO_BIN(yr);
  BCD_TO_BIN(mon);
  BCD_TO_BIN(day);
  BCD_TO_BIN(hr);
  BCD_TO_BIN(min);
  BCD_TO_BIN(sec);

  if (yr > 99 || mon < 1 || mon > 12 || day > 31 || day < 1 ||
      hr > 23 || min > 59 || sec > 59 || 
      clkstop) {
    printk(KERN_ERR
	   "SH RTC: invalid value, resetting to 1 Jan 2000\n");
    /* Reset S-3531A set (20)00year/01month/01day/0week 00hour 00minute 00second */
    sec = 0; min = 0; hr = 0;/*  wk = 0; */ day = 1; mon = 1; yr = 00;
    rs5c313_set_cur_time((unsigned char)sec,
			 (unsigned char)min,
			 (unsigned char)hr,
/* 			 (unsigned char)wk, */
			 (unsigned char)day,
			 (unsigned char)mon,
			 (unsigned char)yr);

/*     ndelay(10000); */
    goto again;
  }

#if RTC_BIT_INVERTED != 0
  if ((sec128 & RTC_BIT_INVERTED))
    sec--;
#endif

  tv->tv_sec = mktime(yr100 * 100 + yr, mon, day, hr, min, sec);
  tv->tv_usec = (sec128 * 1000000) / 128;
}
#else
void sh_rtc_gettimeofday(struct timeval *tv)
{
	unsigned int sec128, sec, min, hr, wk, day, mon, yr, yr100;

 again:
	do {
		ctrl_outb(0, RCR1);  /* Clear CF-bit */
		sec128 = ctrl_inb(R64CNT);
		sec = ctrl_inb(RSECCNT);
		min = ctrl_inb(RMINCNT);
		hr  = ctrl_inb(RHRCNT);
		wk  = ctrl_inb(RWKCNT);
		day = ctrl_inb(RDAYCNT);
		mon = ctrl_inb(RMONCNT);
#if defined(__SH4__)
		yr  = ctrl_inw(RYRCNT);
		yr100 = (yr >> 8);
		yr &= 0xff;
#else
		yr  = ctrl_inb(RYRCNT);
		yr100 = (yr == 0x99) ? 0x19 : 0x20;
#endif
	} while ((ctrl_inb(RCR1) & RCR1_CF) != 0);

#if RTC_BIT_INVERTED != 0
	/* Work around to avoid reading incorrect value. */
	if (sec128 == RTC_BIT_INVERTED) {
		schedule_timeout(1);
		goto again;
	}
#endif

	BCD_TO_BIN(yr100);
	BCD_TO_BIN(yr);
	BCD_TO_BIN(mon);
	BCD_TO_BIN(day);
	BCD_TO_BIN(hr);
	BCD_TO_BIN(min);
	BCD_TO_BIN(sec);

	if (yr > 99 || mon < 1 || mon > 12 || day > 31 || day < 1 ||
	    hr > 23 || min > 59 || sec > 59) {
		printk(KERN_ERR
		       "SH RTC: invalid value, resetting to 1 Jan 2000\n");
		ctrl_outb(RCR2_RESET, RCR2);  /* Reset & Stop */
		ctrl_outb(0, RSECCNT);
		ctrl_outb(0, RMINCNT);
		ctrl_outb(0, RHRCNT);
		ctrl_outb(6, RWKCNT);
		ctrl_outb(1, RDAYCNT);
		ctrl_outb(1, RMONCNT);
#if defined(__SH4__)
		ctrl_outw(0x2000, RYRCNT);
#else
		ctrl_outb(0, RYRCNT);
#endif
		ctrl_outb(RCR2_RTCEN|RCR2_START, RCR2);  /* Start */
		goto again;
	}

#if RTC_BIT_INVERTED != 0
	if ((sec128 & RTC_BIT_INVERTED))
		sec--;
#endif

	tv->tv_sec = mktime(yr100 * 100 + yr, mon, day, hr, min, sec);
	tv->tv_usec = (sec128 * 1000000) / 128;
}
#endif
/* 2003.09.25 I-O DATA NSD NWG	SH RTC -> RICOH RS5C313	<---- */

int sh_rtc_settimeofday(const struct timeval *tv)
{
	unsigned long nowtime = tv->tv_sec;
	int retval = 0;
	int real_seconds, real_minutes, cmos_minutes;

	ctrl_outb(RCR2_RESET, RCR2);  /* Reset pre-scaler & stop RTC */

	cmos_minutes = ctrl_inb(RMINCNT);
	BCD_TO_BIN(cmos_minutes);

	/*
	 * since we're only adjusting minutes and seconds,
	 * don't interfere with hour overflow. This avoids
	 * messing with unknown time zones but requires your
	 * RTC not to be off by more than 15 minutes
	 */
	real_seconds = nowtime % 60;
	real_minutes = nowtime / 60;
	if (((abs(real_minutes - cmos_minutes) + 15)/30) & 1)
		real_minutes += 30;	/* correct for half hour time zone */
	real_minutes %= 60;

	if (abs(real_minutes - cmos_minutes) < 30) {
		BIN_TO_BCD(real_seconds);
		BIN_TO_BCD(real_minutes);
		ctrl_outb(real_seconds, RSECCNT);
		ctrl_outb(real_minutes, RMINCNT);
	} else {
		printk(KERN_WARNING
		       "set_rtc_time: can't update from %d to %d\n",
		       cmos_minutes, real_minutes);
		retval = -1;
	}

	ctrl_outb(RCR2_RTCEN|RCR2_START, RCR2);  /* Start RTC */

	return retval;
}

#if defined(CONFIG_SH_JULIAN)
static int months[48] = {
	   0,   31,   59,   90,  120,  151,  181,  212,  243,  273,  304,  334,
	 365,  396,  424,  455,  485,  516,  546,  577,  608,  638,  669,  699,
	 730,  761,  790,  821,  851,  882,  912,  943,  974, 1004, 1035, 1065,
	1096, 1127, 1155, 1186, 1216, 1247, 1277, 1308, 1339, 1369, 1400, 1430
};

/* 2003.09.25 I-O DATA NSD NWG	SH RTC -> RICOH RS5C313	----> */
#if 0
int julian_rtc_settimeofday(const struct timeval *tv)
{
	int year, month, day, hour, min, sec, year4, year100;
	int i, tmp;

	sec = tv->tv_sec % 60;
	tmp = tv->tv_sec / 60;
	min = tmp % 60;
	tmp /= 60;
	hour = tmp % 24;
	tmp /= 24;
	year4 = tmp / 1461;
	tmp -= year4 * 1461;
	for (i=1; i<48 && tmp>=months[i]; ++i)
		;
	year = year4 * 4 + (i-1) / 12;
	month = ((i -1) % 12) + 1;
	day = tmp - months[i - 1] + 1;

	year += 1970;
	year100 = year / 1000;
	tmp = year % 1000;
	tmp /= 100;
	year100 = (year100 * 10) + tmp;
	year -= year100 * 100;

	BIN_TO_BCD(sec);
	BIN_TO_BCD(min);
	BIN_TO_BCD(hour);
	BIN_TO_BCD(day);
	BIN_TO_BCD(month);
	BIN_TO_BCD(year100);
	BIN_TO_BCD(year);

	ctrl_outb(RCR2_RESET, RCR2);	/* Reset & Stop */

	ctrl_outb(sec, RSECCNT);
	ctrl_outb(min, RMINCNT);
	ctrl_outb(hour, RHRCNT);
	ctrl_outb(day, RDAYCNT);
	ctrl_outb(month, RMONCNT);
	ctrl_outw((year100<<8)|year, RYRCNT);

	ctrl_outb(RCR2_RTCEN|RCR2_START, RCR2);	/* Start */
	time_status |= STA_UNSYNC;

	return 0;
}
#else
int julian_rtc_settimeofday(const struct timeval *tv)
{
  int year, month, day, hour, min, sec, year4, year100;
  int i, tmp;

/* printk(KERN_ERR "julian_rtc_settimeofday()\n"); */
/* schedule_timeout(10000); */

  sec = tv->tv_sec % 60;
  tmp = tv->tv_sec / 60;
  min = tmp % 60;
  tmp /= 60;
  hour = tmp % 24;
  tmp /= 24;
  year4 = tmp / 1461;
  tmp -= year4 * 1461;
  for (i=1; i<48 && tmp>=months[i]; ++i)
    ;
  year = year4 * 4 + (i-1) / 12;
  month = ((i -1) % 12) + 1;
  day = tmp - months[i - 1] + 1;

  year += 1970;
  year100 = year / 1000;
  tmp = year % 1000;
  tmp /= 100;
  year100 = (year100 * 10) + tmp;
  year -= year100 * 100;

  BIN_TO_BCD(sec);
  BIN_TO_BCD(min);
  BIN_TO_BCD(hour);
  BIN_TO_BCD(day);
  BIN_TO_BCD(month);
  BIN_TO_BCD(year100);
  BIN_TO_BCD(year);

  rs5c313_set_cur_time((unsigned char)sec,
		       (unsigned char)min,
		       (unsigned char)hour,
/* 		       (unsigned char)wk, */
		       (unsigned char)day,
		       (unsigned char)month,
		       (unsigned char)year);
  time_status |= STA_UNSYNC;

  return 0;
}
#endif
/* 2003.09.25 I-O DATA NSD NWG	SH RTC -> RICOH RS5C313	<---- */
#endif
