/*
 * linux/arch/sh/kernel/setup_julian.c
 *
 * Copyright (C) 2000  Kazumoto Kojima
 *
 * I-O DATA Device, Inc. Julian Support.
 *
 * Modified for Julian by
 * Atom Create Engineering Co., Ltd. 2002.
 */

#include <linux/config.h>
#include <linux/init.h>
#include <linux/irq.h>

#include <linux/hdreg.h>
#include <linux/ide.h>
#include <linux/pci.h>
#include <asm/io.h>
#include <asm/iodata_julian.h>

#include <linux/mm.h>
#include <linux/vmalloc.h>

/* defined in mm/ioremap.c */
extern void * p3_ioremap(unsigned long phys_addr, unsigned long size, unsigned long flags);

static unsigned int debug_counter;

static void enable_julian_irq(unsigned int irq);
static void disable_julian_irq(unsigned int irq);

/* shutdown is same as "disable" */
#define shutdown_julian_irq disable_julian_irq

static void ack_julian_irq(unsigned int irq);
static void end_julian_irq(unsigned int irq);

static unsigned int startup_julian_irq(unsigned int irq)
{
	enable_julian_irq(irq);
	return 0; /* never anything pending */
}

static void disable_julian_irq(unsigned int irq)
{
	unsigned long flags;
	unsigned char val;
	unsigned char mask = 0xff ^ (0x01 << (irq-5));

	/* Set the priority in IPR to 0 */
	save_and_cli(flags);
	val = ctrl_inb(PA_IMASK);
	val &= mask;
	ctrl_outb(val, PA_IMASK);
	restore_flags(flags);
}

static void enable_julian_irq(unsigned int irq)
{
	unsigned long flags;
	unsigned char val;
	unsigned char value = (0x01 << (irq-5));

	/* Set priority in IPR back to original value */
	save_and_cli(flags);
	val = ctrl_inb(PA_IMASK);
	val |= value;
	ctrl_outb(val, PA_IMASK);
	restore_flags(flags);
}

static void ack_julian_irq(unsigned int irq)
{
	disable_julian_irq(irq);
}

static void end_julian_irq(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS)))
		enable_julian_irq(irq);
}

static struct hw_interrupt_type julian_irq_type = {
	"Julian IRQ",
	startup_julian_irq,
	shutdown_julian_irq,
	enable_julian_irq,
	disable_julian_irq,
	ack_julian_irq,
	end_julian_irq,
};

static void make_julian_irq(unsigned int irq)
{
	disable_irq_nosync(irq);
	irq_desc[irq].handler = &julian_irq_type;
	disable_julian_irq(irq);
}

/*
 * Initialize IRQ setting
 */
void __init init_julian_IRQ(void)
{
	int i;

	for (i=5; i<12; i++)
		make_julian_irq(i);
}

/*
 * Initialize the board
 */
void __init setup_julian(void)
{
/* IODATA. */
#if 1
	printk(KERN_INFO "I-O DATA DEVICE, INC. landisk3 support.\n");
	ctrl_outb(0x09, PA_LED);
#else
	printk(KERN_INFO "I-O DATA DEVICE, INC. Julian support.\n");
	/* Clear LED */
#if defined(CONFIG_FALCON)
	ctrl_outb(0x00, PA_LED);
#else
	ctrl_outb(~0x00, PA_LED);
#endif
#endif
	debug_counter = 0;
}

void *area5_io_base;
void *area6_io_base;

int __init cf_init(void)
{
	pgprot_t prot;
	unsigned long paddrbase, psize;

	/* open I/O area window */
	paddrbase = virt_to_phys((void*)PA_AREA5_IO);
	psize = PAGE_SIZE;
	prot = PAGE_KERNEL_PCC(1, _PAGE_PCC_IO16);
	area5_io_base = p3_ioremap(paddrbase, psize, prot.pgprot);
	if (!area5_io_base) {
		printk("allocate_cf_area : can't open CF I/O window!\n");
		return -ENOMEM;
	}
#if 0
	printk("p3_ioremap(paddr=0x%08lx, psize=0x%08lx, port=0x%08lx)=0x%08lx\n",
		paddrbase, psize, port.pgport, area5_iobase);
#endif

	paddrbase = virt_to_phys((void*)PA_AREA6_IO);
	psize = PAGE_SIZE;
	prot = PAGE_KERNEL_PCC(0, _PAGE_PCC_IO16);
	area6_io_base = p3_ioremap(paddrbase, psize, prot.pgprot);
	if (!area6_io_base) {
		printk("allocate_cf_area : can't open HDD I/O window!\n");
		return -ENOMEM;
	}
#if 0
	printk("p3_ioremap(paddr=0x%08lx, psize=0x%08lx, port=0x%08lx)=0x%08lx\n",
		paddrbase, psize, port.pgport, area6_iobase);
#endif
	printk(KERN_INFO "Allocate Area5/6 success.\n");

	/* XXX : do we need attribute and common-memory area also? */

	return 0;
}

__initcall (cf_init);

#ifdef CONFIG_HEARTBEAT

#include <linux/sched.h>

/* Cycle the LED's in the clasic knightrider/Sun pattern */
void heartbeat_julian(void)
{
	static unsigned int cnt = 0, period = 0;
	volatile unsigned char *p = (volatile unsigned char *)PA_LED;
	static unsigned bit = 0, up = 1;

	cnt += 1;
	if (cnt < period)
		return;

	cnt = 0;

	/* Go through the points (roughly!):
	 * f(0)=10, f(1)=16, f(2)=20, f(5)=35, f(int)->110
	 */
	period = 110 - ((300 <<FSHIFT)/((avenrun[0]/5) + (3 << FSHIFT)));

#if defined(CONFIG_FALCON)
	*p = (1 << bit);
#else
	*p = ~(1 << bit);
#endif
	if (up)
		if (bit == 7) {
			bit--;
			up = 0;
		} else
			bit++;
	else if (bit == 0)
		up = 1;
	else
		bit--;
}
#endif

void debug_led_disp_value(unsigned char value)
{
	ctrl_outb(value, PA_LED);
}

void debug_led_disp(void)
{
	unsigned char value;

	value = (unsigned char)debug_counter++;
#if defined(CONFIG_FALCON)
	ctrl_outb(value, PA_LED);
#else
	ctrl_outb(~value, PA_LED);
#endif
	if (value == 0xff)
		debug_counter = 0;
}
