/* $Id: ide.c,v 1.10 2000/08/10 22:37:37 gniibe Exp $
 *
 * sh-ipl+g/ide.c
 *
 * Support for IDE disk drive
 *
 *  Copyright (C) 2000  Niibe Yutaka
 *
 * This file is subject to the terms and conditions of the GNU Lesser
 * General Public License.  See the file "COPYING.LIB" in the main
 * directory of this archive for more details.
 *
 */
#include "config.h"
#include "defs.h"


/* Machine dependent part */

#include "io.h"

//01/11/06 ATOM sakuma isyoku
#define PA_SUPERIO	0xb4000000 /* IDE Primary */
#define PA_SUPERIO2	0xb8000000 /* IDE Secondary */

//IDE Primary PIO default MODE
#define	DEFAULT_MODE	1

void delay (void);
static	void delay256 (void);
static	void delay10000 (void);
static	void	hd_chg(int sw);

static unsigned long ide_offset;
static unsigned long super;
static unsigned long super2;

/* Initialize hardware for IDE support */
void
init_ide (void)
{
  super = PA_SUPERIO;
  super2 = PA_SUPERIO2;
  ide_offset = super;
}

void delay (void)
{
  volatile unsigned short trash;
  trash = *(volatile unsigned short *) 0xa0000000;
}

void delay256 (void)
{
  int i;
  for (i=0; i<256; i++)
    delay ();
}

void delay10000 (void)
{
  int i;
  for (i=0; i<10000; i++)
    delay ();
}

/* Machine independent part */

#define IDE_BSY 0x80
#define IDE_DRQ 0x08
#define IDE_ERR 0x01

//01/09/24 ATOM sakuma isyoku
#define IDE_DATA			0x040/2
#define IDE_ERROR			0x042/2
#define IDE_FEATURES			0x042/2
#define IDE_SECTOR_COUNT		0x044/2
#define IDE_SECTOR_NUMBER		0x046/2
#define IDE_SECTOR_CYLINDER_LOW		0x048/2
#define IDE_SECTOR_CYLINDER_HIGH	0x04a/2
#define IDE_DEVICE_HEAD			0x04c/2
#define IDE_STATUS			0x04e/2
#define IDE_COMMAND			0x04e/2
#define IDE_ALTERNATE_STATUS		0x02c/2
#define IDE_DEVICE_CONTROL		0x02c/2

#define IDE_LBA 0x40

#define IDE_COMMAND_READ_SECTORS				0x20
#define IDE_COMMAND_IDLE					0xE3
#define IDE_COMMAND_IDENTIFY					0xEC
#define IDE_COMMAND_IDENTIFY_PACKET				0xA1
#define IDE_COMMAND_SET_FEATURES				0xEF
#define IDE_COMMAND_INITIALIZE_DEVICE_PARAMETERS		0x91
//ATOM debug 48bit LBA support
#define IDE_COMMAND_READ_SECTORS_EXT				0x24

/* Don't loop forever */
//ATOM debug
//#define TIMEOUT 32767
#define TIMEOUT 10000/10

static int
ide_register_check (void)
{
  ide_outb (0x55, IDE_SECTOR_CYLINDER_LOW);
  ide_outb (0xaa, IDE_SECTOR_CYLINDER_HIGH);
  if (ide_inb (IDE_SECTOR_CYLINDER_LOW) != 0x55) {
    return -1;
  }
  if (ide_inb (IDE_SECTOR_CYLINDER_HIGH) != 0xaa) {
    return -1;
  }

  return 0;
}

/* Reset the bus, set it polling mode. */
static int
ide_reset (void)
{
  unsigned long status;
  int i;

  ide_outb (0x04|0x02, IDE_DEVICE_CONTROL);
  delay10000 ();

  /* Polling mode (nIEN = 0x02)*/
  ide_outb (0x02, IDE_DEVICE_CONTROL);
  for(i=0;i<12;i++) {
    delay10000 ();
  }

  for (i=0; i<TIMEOUT; i++) {
    status = ide_inb (IDE_ALTERNATE_STATUS);
    if ((status & IDE_BSY) == 0) {
      break;
    }
    delay256 ();
  }
  if (i == TIMEOUT) {
    return -1;
  }
  else if (ide_inb (IDE_ERROR) != 1) {
    return -1;
  }

  return 0;
}

/* Device selection protocol */
static int
ide_device_selection (int dev)
{
  unsigned long status;
  int i;

  for (i=0; i<TIMEOUT; i++) {
    status = ide_inb (IDE_STATUS);
    if ((status & (IDE_BSY|IDE_DRQ)) == 0) {
      break;
    }
    delay256 ();
  }
  if (i == TIMEOUT) {
    return -1;
  }
  /* Polling */
  ide_outb (0x02, IDE_DEVICE_CONTROL);
  ide_outb (dev << 4, IDE_DEVICE_HEAD);
  delay256 ();

  for (i=0; i<TIMEOUT; i++) {
    status = ide_inb (IDE_STATUS);
    if ((status & (IDE_BSY|IDE_DRQ)) == 0) {
      break;
    }
    delay256 ();
  }
  if (i == TIMEOUT) {
    return -1;
  }

  return 0;
}

static int
ide_get_data (unsigned char *buf, int count)
{
  unsigned long status;
  unsigned short data;
  int i;

  while (count--) {
    /* Dummy read */
    status = ide_inb (IDE_ALTERNATE_STATUS);

    for (i=0; i<TIMEOUT; i++) {
      status = ide_inb (IDE_ALTERNATE_STATUS);
      if ((status & IDE_BSY) == 0) {
        break;
      }
      delay256 ();
    }
    if (i == TIMEOUT) {
      putString ("Timeout: ");
      printouthex (status);
      putString ("\n");
      return -1;
    }

    if ((status & IDE_ERR)) {		/* Error occurred */
      status = ide_inb (IDE_STATUS);
      putString ("Error: ");
      printouthex (status);
      putString ("\n");
      return -1;
    }

    /* Read a data */
    for (i=0; i<256; i++) {
      data = ide_inw (IDE_DATA);
#if defined(__LITTLE_ENDIAN__)
      *buf++ = data & 0xff;
      *buf++ = data >> 8;
#else
      *buf++ = data >> 8;
      *buf++ = data & 0xff;
#endif
    }
  }
  /* Dummy read */
  ide_inb (IDE_ALTERNATE_STATUS);
  status = ide_inb (IDE_STATUS);

  return 0;
}

static void
ide_fix_string (unsigned char *dest, unsigned char *src, int num)
{
  int i;

  for (i=0; i<num; i+=2) {
    *dest++ = *(src+1);
    *dest++ = *src;
    src += 2;
  }
  while (*--dest == ' ');	/* do nothing */
  ++dest;
  *dest++ = ' ';
  *dest++ = '\0';
}

/* Only for Device 0 (for now) */
/* XXX: Should be struct to support multiple devices.. */
static char ide_device_data;
static char ide_transfer_mode;
static short ide_sectors_per_track;
static short ide_max_head;
//ATOM debug 48bit LBA support
static char  ide_48bit_mode;
//ATOM debug LCD mode
static char  lcd_mode;

static int
ide_identify_device (int dev,unsigned short *buf)
{

  if (ide_device_selection (dev) < 0) {
    return -1;
  }

  ide_outb (IDE_COMMAND_IDENTIFY, IDE_COMMAND);
  delay256 ();

  if (ide_get_data ((unsigned char *)buf, 1) < 0) {
    return -1;
  }
  if (*buf & 0x04) { /* Incomplete */
      /* XXX: Read the word #2, and printout the result */
    return -1;
  }
  /* XXX: check buf[64] for supported PIO mode */
  if((buf[80] & 0x10) != 0) {
    ide_transfer_mode = 4;
  }
  else if((buf[80] & 0x08) != 0) {
    ide_transfer_mode = 3;
  }
  else if((buf[80] & 0x04) != 0) {
    ide_transfer_mode = 2;
  }
  else if((buf[64] & 0x02) != 0) {
    ide_transfer_mode = 1;
  }
  else if((buf[64] & 0x01) != 0) {
    ide_transfer_mode = 0;
  }
//ATOM debug 48bit LBA support
//  if((buf[83] & 0x0200) != 0) {
  if((buf[100] + buf[101]) != 0) {
    ide_48bit_mode = 1;
  }
  else {
    ide_48bit_mode = 0;
  }
  ide_sectors_per_track = buf[6];
  ide_max_head = buf[3] - 1;
  ide_device_data = 0;

  return 0;
}


static int
ide_identify_packet_device (int dev,unsigned short *buf)
{
  unsigned char name[42];

  if (ide_device_selection (dev) < 0) {
    return -1;
  }
  ide_outb (IDE_COMMAND_IDENTIFY_PACKET, IDE_COMMAND);
  delay256 ();

  if (ide_get_data ((unsigned char *)buf, 1) < 0) {
    return -1;
  }
  /* XXX: check buf[64] for supported PIO mode */
  if((buf[80] & 0x10) != 0) {
    ide_transfer_mode = 4;
  }
  else if((buf[80] & 0x08) != 0) {
    ide_transfer_mode = 3;
  }
  else if((buf[80] & 0x04) != 0) {
    ide_transfer_mode = 2;
  }
  else if((buf[64] & 0x02) != 0) {
    ide_transfer_mode = 1;
  }
  else if((buf[64] & 0x01) != 0) {
    ide_transfer_mode = 0;
  }
  putString ("Disk drive detected: ");
  ide_fix_string (name, (unsigned char *)&buf[27], 40);
  putString (name);
  ide_fix_string (name, (unsigned char *)&buf[23], 8);
  putString (name);
  ide_fix_string (name, (unsigned char *)&buf[10], 20);
  putString (name);
  putString ("\n");

  return 0;
}


static	void
ide_identify_print(unsigned short *buf)
{
unsigned char name[42];
unsigned int	sec,size;

//ATOM debug 48bit LBA support
  if(ide_48bit_mode == 1) {
    putString("48bit LBA mode support\n");
  }
  else {
    putString("48bit LBA mode non support\n");
  }
  putString ("Disk drive detected: ");
  ide_fix_string (name, (unsigned char *)&buf[27], 40);
  putString (name);
  ide_fix_string (name, (unsigned char *)&buf[23], 8);
  putString (name);
  ide_fix_string (name, (unsigned char *)&buf[10], 20);
  putString (name);
  putString ("\n");
  putString ("LBA: ");
  if(ide_48bit_mode == 0) {
    sec = (buf[61] << 16) | buf[60];
    hex_string(sec);
    putString ("\n");
  }
  else {
    sec = (buf[101] << 16) | buf[100];
    hex_string(sec);
    putString ("\n");
  }
  putString ("DiskSize: ");
  //1G Over?
  if(sec >= 19050351) {
    size = sec / 1950351;
    dec_string(size);
    putString ("GByte\n");
  }
  else {
    size = sec * 512;
    dec_string(size);
    putString ("Byte\n");
  }
}


#define IDE_SUBCOMMAND_SET_TRANSFER_MODE 0x03
static int
ide_set_transfer_mode (int dev, int mode)
{
  unsigned long status;
  int i;

  /* XXX: Don't set the mode, but use the default... for now */

  if (ide_device_selection (dev) < 0) {
    return -1;
  }
  ide_outb (IDE_SUBCOMMAND_SET_TRANSFER_MODE, IDE_FEATURES);
//ATOM debug
  ide_outb (0x8 | mode, IDE_SECTOR_COUNT); /* PIO Default mode */
  ide_outb (IDE_COMMAND_SET_FEATURES, IDE_COMMAND);
  delay256 ();

  for (i=0; i<TIMEOUT; i++) {
    status = ide_inb (IDE_ALTERNATE_STATUS);
    if ((status & IDE_BSY) == 0) {
      break;
    }
    delay256 ();
  }
  if (i == TIMEOUT) {
    return -1;
  }
  status = ide_inb (IDE_STATUS);
  putString ("Set Transfer Mode result: ");
  printouthex (status);

  return 0;
}

static int
ide_set_device_params (int dev, int sectors_per_track, int max_head)
{
  unsigned long status;
  int i;

  if (ide_device_selection (dev) < 0) {
    return -1;
  }
  ide_outb (sectors_per_track, IDE_SECTOR_COUNT);
  ide_outb (max_head | (dev << 4), IDE_DEVICE_HEAD);
  ide_outb (IDE_COMMAND_INITIALIZE_DEVICE_PARAMETERS, IDE_COMMAND);
  delay256 ();

  for (i=0; i<TIMEOUT; i++) {
    status = ide_inb (IDE_ALTERNATE_STATUS);
    if ((status & IDE_BSY) == 0) {
      break;
    }
    delay256 ();
  }
  if (i == TIMEOUT) {
    return -1;
  }
  status = ide_inb (IDE_STATUS);
  putString ("Initialize Device Parameters result: ");
  printouthex (status);

  return 0;
}

static int
ide_idle (int dev)
{
  unsigned long status;
  int i;

  if (ide_device_selection (dev) < 0) {
    return -1;
  }
  ide_outb (0x00, IDE_SECTOR_COUNT);
  ide_outb (IDE_COMMAND_IDLE, IDE_COMMAND);
  for(i=0;i<3;i++) {
    delay10000 ();
  }

  for (i=0; i<TIMEOUT; i++) {
    status = ide_inb (IDE_ALTERNATE_STATUS);
    if ((status & IDE_BSY) == 0) {
      break;
    }
    for(i=0;i<3;i++) {
      delay10000 ();
    }
  }
  if (i == TIMEOUT) {
    return -1;
  }
  status = ide_inb (IDE_STATUS);
  putString ("IDLE result: ");
  printouthex (status);

  return 0;
}

/* Read sectors at the address specified by LBA (linear block address),
   and copy the data into the memory at BUF.  The number of sectors are
   specified by COUNT (Should be 1-256).
 */
int
ide_read_sectors (int dev, unsigned long lba, unsigned char *buf, int count)
{
  /* Select the device */
  if (ide_device_selection (dev) < 0) {
    return -1;
  }
  /* Set the parameters */
  if(ide_48bit_mode == 0) {
    ide_outb (count, IDE_SECTOR_COUNT);
    ide_outb (lba & 0xff, IDE_SECTOR_NUMBER);
    ide_outb ((lba & 0xff00)>>8, IDE_SECTOR_CYLINDER_LOW);
    ide_outb ((lba & 0xff0000)>>16, IDE_SECTOR_CYLINDER_HIGH);
    ide_outb (((lba & 0x0f000000)>>24)|IDE_LBA, IDE_DEVICE_HEAD);
  }
  else {
    ide_outb ((count & 0xff00) >> 8, IDE_SECTOR_COUNT);
    ide_outb ((count & 0xff), IDE_SECTOR_COUNT);
    ide_outb ((lba & 0xff000000) >> 24, IDE_SECTOR_NUMBER);
    ide_outb ((lba & 0xff), IDE_SECTOR_NUMBER);
    ide_outb (0, IDE_SECTOR_CYLINDER_LOW);
    ide_outb ((lba & 0xff00)>>8, IDE_SECTOR_CYLINDER_LOW);
    ide_outb (0, IDE_SECTOR_CYLINDER_HIGH);
    ide_outb ((lba & 0xff0000)>>16, IDE_SECTOR_CYLINDER_HIGH);
    ide_outb (IDE_LBA, IDE_DEVICE_HEAD);
  }

  /* Issue the command */
  if(ide_48bit_mode == 0) {
    ide_outb (IDE_COMMAND_READ_SECTORS, IDE_COMMAND);
  }
  else {
    ide_outb (IDE_COMMAND_READ_SECTORS_EXT, IDE_COMMAND);
  }
  if (ide_get_data (buf, count) < 0) {
    return -1;
  }
  return 0;
}

static char ide_detected;
static char ide_started;

int
ide_detect_devices (int dev)
{
unsigned short buf[256];	/* No stack overflow? Cross fingered.. */
int  ret;

  ide_detected = 1;

  /* Clear device information */
  ide_device_data = -1;

  /* Only for Device 0 (for now) */
  if (ide_reset () < 0){		/* Failed on resetting bus */
    return -1;
  }

  if (ide_device_selection (dev) < 0){
    return -1;
  }
  if (ide_register_check () < 0)
    return -1;
  if (ide_identify_device (dev,buf) == 0) {
    ret = 0;
  }
  else if (ide_identify_packet_device (dev,buf) == 0) {
    ret = 1;
  }
  else {
    return -1;
  }

  return ret;
}

void
ide_startup_devices (int dev)
{
  if (ide_device_data != 0) {
    return;
  }
  /* Only for Device 0 (for now) */
  /* SET FEATURES (set transfer mode) */
  /* NOTE: This gets error with Compact Flash */
//ATOM debug
//  ide_set_transfer_mode (dev, 3);
  ide_set_transfer_mode (dev, ide_transfer_mode);

  /* INITIALIZE DEVICE PARAMETERS */
  ide_set_device_params (dev, ide_sectors_per_track, ide_max_head);

  /* IDLE */
  ide_idle (dev);

  ide_started = 1;
}

//01/09/24 ATOM sakuma isyoku
int
hd_reset (void)
{
  unsigned long status;
  int i;

  ide_outb (0x04|0x00, IDE_DEVICE_CONTROL);
  delay10000 ();

  /* Polling mode (nIEN = 0x02)*/
  ide_outb (0x00, IDE_DEVICE_CONTROL);
  for(i=0;i<4;i++) {
    delay10000 ();
  }

  for (i=0; i<TIMEOUT; i++) {
    status = ide_inb (IDE_ALTERNATE_STATUS);
    if ((status & IDE_BSY) == 0) {
      break;
    }
    delay256 ();
//ATOM debug
//  delay10000 ();		/* wait for power on */
  }
  if (i == TIMEOUT){
    return -1;
  }
  else if (ide_inb (IDE_ERROR) != 1) {
    return -1;
  }
  return 0;
}

/*
static const int wcr2_tbl[5][5] = {
	0xbefe66df,0xbefe66df,0xbefe66df,0xbefe66df,0x9efe66df,
	0xbefe66df,0xbefe66df,0xbefe66df,0xbefe66df,0x9efe66df,
	0xbefe66df,0xbefe66df,0xbefe66df,0xbefe66df,0x9efe66df,
	0xbefe66df,0xbefe66df,0xbefe66df,0xbefe66df,0x9efe66df,
	0xbe7e66df,0xbe7e66df,0xbe7e66df,0xbe7e66df,0x9e7e66df
};

static const short pcr_tbl[5][5] = {
	0x5b64,0x5b64,0x5b24,0x4b24,0x4b24,
	0x5b64,0x5b64,0x5b24,0x4b24,0x4b24,
	0x5964,0x5964,0x5924,0x4924,0x4924,
	0x1964,0x1964,0x1924,0x0924,0x0924,
	0x1964,0x1964,0x1924,0x0924,0x0924
};
*/

static const int wcr2_tbl[5][5] = {
	0xbefe66df,0xbefe66df,0xbefe66df,0x9efe66df,0x9efe66df,
	0xbefe66df,0xbefe66df,0xbefe66df,0x9efe66df,0x9efe66df,
	0xbefe66df,0xbefe66df,0xbefe66df,0x9efe66df,0x9efe66df,
	0xbe7e66df,0xbe7e66df,0xbe7e66df,0x9e7e66df,0x9e7e66df,
	0xbe7e66df,0xbe7e66df,0xbe7e66df,0x9e7e66df,0x9e7e66df
};

static const short pcr_tbl[5][5] = {
	0x5b5b,0x5b1b,0x5b1a,0x4b1a,0x4ada,
	0x595b,0x591b,0x591a,0x491a,0x48da,
	0x5953,0x5913,0x5912,0x4912,0x48d2,
	0x1953,0x1913,0x1912,0x0912,0x08d2,
	0x1753,0x1713,0x1712,0x0712,0x06d2
};


int
hd_init() {
unsigned long *wcr2;
unsigned short *pcr;
int  ret,pri,sec;

  pri = sec = 0;
  //BUS con Address SET
  wcr2 = 0xff80000c;
  pcr = 0xff800018;
  //SECONDARY ATA CHECK
  lcd_mode = 0;
  hd_chg(1);
  if (ide_reset () == 0) {		/* warikomi off */
    ret = pio_mode_set(0);
    if(ret != -1) {
      sec = ret;
    }
/*
    pio_mode_set(1);
*/
    //interrupt ON
    hd_reset ();			/* warikomi on */
  }
  //PRIMARY ATA CHECK
  lcd_mode = 1;
  hd_chg(0);
  if (ide_reset () == 0){		/* warikomi off */
    ret = pio_mode_set(0);
    if(ret != -1) {
      pri = ret;
    }
/*
    pio_mode_set(1);
*/
    //interrupt ON
    hd_reset ();			/* warikomi on */
  }
  lcd_mode = 0;
//BUS con Setting
  *wcr2 = wcr2_tbl[pri][sec];
  *pcr = pcr_tbl[pri][sec];
}


int
hd_chk()
{
  hd_chg(0);
  if (ide_reset () != 0) {		//pri chk
    return(1);
  }
  if (ide_device_selection (0) < 0){
    return(1);
  }
  if (ide_register_check () < 0) {
    return(1);
  }
  hd_chg(1);
  if (ide_reset () != 0) {		//sec chk
    return(2);
  }
  if (ide_device_selection (0) < 0){
    return(2);
  }
  if (ide_register_check () < 0) {
    return(2);
  }
  return(0);
}




static int
pio_mode_set (int i)
{
unsigned short buf[256];	/* No stack overflow? Cross fingered.. */

  /* Clear device information */
  ide_device_data = -1;

  if(ide_register_check() < 0) {
    return(-1);
  }
  if(i == 0) {
    putString("MASTER:");
  }
  else {
    putString("SLAVE:");
  }

  if(ide_device_selection(i) < 0) {
    return(-1);
  }
  //ATA init
  if(ide_identify_device(i,buf) < 0) {
    //ATAPI init
    if(ide_identify_packet_device(i,buf) < 0) {
      return(-1);
    }
  }
  ide_identify_print(buf);
  //ATOM debug LCD setuzokuji PIO 0 senyou
  if(lcd_mode == 1) {
    ide_transfer_mode = DEFAULT_MODE;
  }

  if(ide_transfer_mode == 0) {
    putString("PIO MODE0\n");
  }
  else if(ide_transfer_mode == 1) {
    putString("PIO MODE1\n");
  }
  else if(ide_transfer_mode == 2) {
    putString("PIO MODE2\n");
  }
  else if(ide_transfer_mode == 3) {
    putString("PIO MODE3\n");
  }
  else if(ide_transfer_mode == 4) {
    putString("PIO MODE4\n");
  }
  //AREA6 transfer mode set
  ide_set_transfer_mode (i, ide_transfer_mode);
  
  return ide_transfer_mode;
}


static	void
hd_chg(int sw)
{
  if(sw == 0) {
    ide_offset = super;
  }
  else {
    ide_offset = super2;
  }
}

static	const unsigned char	hex_tbl[] = {
	'0','1','2','3',
	'4','5','6','7',
	'8','9','A','B',
	'C','D','E','F'
};
static	void
hex_string(unsigned int data)
{
int	i;
unsigned char	work[16];

	work[0] = hex_tbl[((data & 0xf0000000) >> 28)];
	work[1] = hex_tbl[((data & 0x0f000000) >> 24)];
	work[2] = hex_tbl[((data & 0x00f00000) >> 20)];
	work[3] = hex_tbl[((data & 0x000f0000) >> 16)];
	work[4] = hex_tbl[((data & 0x0000f000) >> 12)];
	work[5] = hex_tbl[((data & 0x00000f00) >> 8)];
	work[6] = hex_tbl[((data & 0x000000f0) >> 4)];
	work[7] = hex_tbl[(data & 0x000f)];
	work[8] = 0x00;
	putString(work);
}


static	const unsigned int	dec_tbl[] = {
	1000000000,
	100000000,
	10000000,
	1000000,
	100000,
	10000,
	1000,
	100,
	10,
	1
};

static	void
dec_string(unsigned int data)
{
unsigned int	i,m,d;
unsigned char	work[16];

	for(i=0,m=data;i<10;i++) {
		d = warizan(m,dec_tbl[i],&m);
		work[i] = d + 0x30;
	}
	work[i] = 0x00;
	for(i=0;i<9;i++) {
		if(work[i] != 0x30) {
			break;
		}
	}
	putString(&work[i]);
}

static	int
warizan(unsigned int a,unsigned int b,unsigned int *c)
{
int	i;

	*c = 0;
	if(a == 0) {
		return(0);
	}
	if(b == 0) {
		return(0);
	}
	for(i=0;;i++) {
		if(a >= b) {
			a -= b;
		}
		else {
			*c = a;
			break;
		}
	}
	return(i);
}


