#include "config.h"
#include "defs.h"

#include "io.h"

static unsigned long ide_offset = 0xb4000000;

/* Machine independent part */

#define IDE_BSY 0x80
#define IDE_DRQ 0x08
#define IDE_ERR 0x01

//01/09/24 ATOM sakuma isyoku
#define IDE_DATA			0x040/2
#define IDE_ERROR			0x042/2
#define IDE_FEATURES			0x042/2
#define IDE_SECTOR_COUNT		0x044/2
#define IDE_SECTOR_NUMBER		0x046/2
#define IDE_SECTOR_CYLINDER_LOW		0x048/2
#define IDE_SECTOR_CYLINDER_HIGH	0x04a/2
#define IDE_DEVICE_HEAD			0x04c/2
#define IDE_STATUS			0x04e/2
#define IDE_COMMAND			0x04e/2
#define IDE_ALTERNATE_STATUS		0x02c/2
#define IDE_DEVICE_CONTROL		0x02c/2

#define IDE_LBA 0x40

#define IDE_COMMAND_READ_SECTORS				0x20
#define IDE_COMMAND_IDLE					0xE3
#define IDE_COMMAND_IDENTIFY					0xEC
#define IDE_COMMAND_SET_FEATURES				0xEF
#define IDE_COMMAND_INITIALIZE_DEVICE_PARAMETERS		0x91

#define TIMEOUT 10000*100

void delay (void);
void delay256 (void);
void delay10000 (void);
static void puthex2(int data);


static int
ide_register_check (void)
{
  ide_outb (0x55, IDE_SECTOR_CYLINDER_LOW);
  ide_outb (0xaa, IDE_SECTOR_CYLINDER_HIGH);
  if (ide_inb (IDE_SECTOR_CYLINDER_LOW) != 0x55)
    return -1;
  if (ide_inb (IDE_SECTOR_CYLINDER_HIGH) != 0xaa)
    return -1;
  return 0;
}

/* Reset the bus, set it polling mode. */
static int
ide_reset (int sw)
{
  unsigned long status;
  int i;

  if(sw == 0) {
    ide_outb (0x04|0x02, IDE_DEVICE_CONTROL);
  }
  else {
    ide_outb (0x04, IDE_DEVICE_CONTROL);
  }
  delay10000 ();

  /* Polling mode (nIEN = 0x02)*/
  if(sw == 0) {
    ide_outb (0x02, IDE_DEVICE_CONTROL);
  }
  else {
    ide_outb (0, IDE_DEVICE_CONTROL);
  }
  delay10000 (); delay10000 (); delay10000 (); delay10000 ();
  delay10000 (); delay10000 (); delay10000 (); delay10000 ();
  delay10000 (); delay10000 (); delay10000 (); delay10000 ();
  delay10000 (); delay10000 (); delay10000 (); delay10000 ();
  delay10000 (); delay10000 (); delay10000 (); delay10000 ();
  delay10000 (); delay10000 (); delay10000 (); delay10000 ();
  delay10000 (); delay10000 (); delay10000 (); delay10000 ();
  delay10000 (); delay10000 (); delay10000 (); delay10000 ();
  delay10000 (); delay10000 (); delay10000 (); delay10000 ();

    for (i=0; i<TIMEOUT; i++)
    {
      status = ide_inb (IDE_ALTERNATE_STATUS);
      if ((status & IDE_BSY) == 0)
		break;
      delay256 ();
      delay10000 ();		/* wait for power on */
    }

  if (i == TIMEOUT){
    return -1;
  }
  else if (ide_inb (IDE_ERROR) != 1){
    return -1;
  }
  return 0;
}

/* Device selection protocol */
static int
ide_device_selection (int dev)
{
  unsigned long status;
  int i;

  for (i=0; i<TIMEOUT; i++)
    {
      status = ide_inb (IDE_STATUS);
      if ((status & (IDE_BSY|IDE_DRQ)) == 0)
	break;
      delay256 ();
    }
  if (i == TIMEOUT)
    return -1;

  /* Polling */
  ide_outb (0x02, IDE_DEVICE_CONTROL);
  ide_outb (dev << 4, IDE_DEVICE_HEAD);
  delay256 ();

  for (i=0; i<TIMEOUT; i++)
    {
      status = ide_inb (IDE_STATUS);
      if ((status & (IDE_BSY|IDE_DRQ)) == 0)
	break;
      delay256 ();
    }
  if (i == TIMEOUT)
    return -1;

  return 0;
}

static int
ide_get_data (unsigned char *buf, int count)
{
  unsigned long status;
  unsigned short data;
  int i;

  while (count--)
    {
      /* Dummy read */
      status = ide_inb (IDE_ALTERNATE_STATUS);

      for (i=0; i<TIMEOUT; i++)
	{
	  status = ide_inb (IDE_ALTERNATE_STATUS);
	  if ((status & IDE_BSY) == 0)
	    break;
	  delay256 ();
	}
      if (i == TIMEOUT)
	{
	  putString ("Timeout: ");
	  printouthex (status);
	  putString ("\n");
	  return -1;
	}

      if ((status & IDE_ERR))
	{				/* Error occurred */
	  status = ide_inb (IDE_STATUS);
	  putString ("Error: ");
	  printouthex (status);
	  putString ("\n");
	  return -1;
	}

      /* Read a data */
      for (i=0; i<256; i++)
	{
	  data = ide_inw (IDE_DATA);
	  *buf++ = data & 0xff;
	  *buf++ = data >> 8;
	}
    }

  /* Dummy read */
  ide_inb (IDE_ALTERNATE_STATUS);

  status = ide_inb (IDE_STATUS);

  return 0;
}

static void
ide_fix_string (unsigned char *dest, unsigned char *src, int num)
{
  int i;

  for (i=0; i<num; i+=2)
    {
      *dest++ = *(src+1);
      *dest++ = *src;
      src += 2;
    }
  while (*--dest == ' ')
    /* do nothing */;
  ++dest;
  *dest++ = ' ';
  *dest++ = '\0';
}

/* Only for Device 0 (for now) */
/* XXX: Should be struct to support multiple devices.. */
static char ide_device_data;
static char ide_transfer_mode;
static short ide_sectors_per_track;
static short ide_max_head;

static int
ide_identify_device (dev)
{
  unsigned short buf[256];	/* No stack overflow? Cross fingered.. */
  unsigned char name[42];

  if (ide_device_selection (dev) < 0)
    return -1;

  ide_outb (IDE_COMMAND_IDENTIFY, IDE_COMMAND);
  delay256 ();

  if (ide_get_data ((unsigned char *)buf, 1) < 0)
    return -1;
  if (*buf & 0x04)		/* Incomplete */
    {
      /* XXX: Read the word #2, and printout the result */
      return -1;
    }

  /* XXX: check buf[64] for supported PIO mode */
//atom debug
puthex2(buf[64]);
putString("  <----- transfer mode\n");
  ide_transfer_mode = buf[64] & 0xff;

  putString ("Disk drive detected: ");

  ide_fix_string (name, (unsigned char *)&buf[27], 40);
  putString (name);
  ide_fix_string (name, (unsigned char *)&buf[23], 8);
  putString (name);
  ide_fix_string (name, (unsigned char *)&buf[10], 20);
  putString (name);
  putString ("\n");

  ide_sectors_per_track = buf[6];
  ide_max_head = buf[3] - 1;

  ide_device_data = 0;

  return 0;
}

static int
ide_set_transfer_mode (int dev, int mode)
{
  unsigned long status;
  int i;

  /* XXX: Don't set the mode, but use the default... for now */

  if (ide_device_selection (dev) < 0)
    return -1;

  ide_outb (mode, IDE_FEATURES);
  ide_outb (0x00, IDE_SECTOR_COUNT); /* PIO Default mode */
  ide_outb (IDE_COMMAND_SET_FEATURES, IDE_COMMAND);
  delay256 ();

  for (i=0; i<TIMEOUT; i++)
    {
      status = ide_inb (IDE_ALTERNATE_STATUS);
      if ((status & IDE_BSY) == 0)
	break;
      delay256 ();
    }
  if (i == TIMEOUT)
    return -1;

  status = ide_inb (IDE_STATUS);
  putString ("Set Transfer Mode result: ");
  printouthex (status);

  return 0;
}

void	puthex2(int data)
{
int	dat;
char	work[16];

	dat = data >> 28;
	dat &= 0x0f;
	if(dat < 10) {
		work[0] = dat + 0x30;
	}
	else {
		work[0] = dat + 0x41 - 10;
	}
	dat = data >> 24;
	dat &= 0x0f;
	if(dat < 10) {
		work[1] = dat + 0x30;
	}
	else {
		work[1] = dat + 0x41 - 10;
	}

	dat = data >> 20;
	dat &= 0x0f;
	if(dat < 10) {
		work[2] = dat + 0x30;
	}
	else {
		work[2] = dat + 0x41 - 10;
	}
	dat = data >> 16;
	dat &= 0x0f;
	if(dat < 10) {
		work[3] = dat + 0x30;
	}
	else {
		work[3] = dat + 0x41 - 10;
	}

	dat = data >> 12;
	dat &= 0x0f;
	if(dat < 10) {
		work[4] = dat + 0x30;
	}
	else {
		work[4] = dat + 0x41 - 10;
	}
	dat = data >> 8;
	dat &= 0x0f;
	if(dat < 10) {
		work[5] = dat + 0x30;
	}
	else {
		work[5] = dat + 0x41 - 10;
	}

	dat = data >> 4;
	dat &= 0x0f;
	if(dat < 10) {
		work[6] = dat + 0x30;
	}
	else {
		work[6] = dat + 0x41 - 10;
	}
	dat = data & 0x0f;
	if(dat < 10) {
		work[7] = dat + 0x30;
	}
	else {
		work[7] = dat + 0x41 - 10;
	}
	work[8] = ' ';
	work[9] = 0x00;
	putString(work);
}


